package sinosoftgz.utils.io;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.nio.channels.FileChannel;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Random;
import java.util.UUID;

/**
 * 文件工具类
 *
 * @Project Augurit
 * @Author LiRuiFa
 * @ClassName FileUtil
 * @Date 2019/5/27 10:46
 * @Version 1.0
 **/
public class FileUtil {

    /**
     * 获取文件信息
     *
     * @param multipartFile
     * @return
     */
    public static FileEntity getFileEntity(MultipartFile multipartFile, String path, String link) {
        try {
            if (null == multipartFile) return null;
            double kb = ((double) multipartFile.getSize()) / 1024;
            double mb = kb / 1024;
            double gb = mb / 1024;
            String id = UUID.randomUUID().toString().replaceAll("-", "");
            String name = new String((multipartFile.getOriginalFilename().substring(0, multipartFile.getOriginalFilename().lastIndexOf("."))).getBytes("GBK"), System.getProperty("sun.jnu.encoding"));
            String extension = multipartFile.getOriginalFilename().substring(multipartFile.getOriginalFilename().lastIndexOf(".") + 1, multipartFile.getOriginalFilename().length());
            return new FileEntity.Builder()
                    .setId(id)
                    .setName(name)
                    .setExtension(extension)
                    .setSize(gb >= 1 ? gb + "GB" : (mb >= 1 ? mb + "MB" : kb + "KB"))
                    .setPath(getPath(path) + File.separator + "temporary" + File.separator + id + "." + extension)
                    .setLink(link + "/" + "temporary" + "/" + id + "." + extension)
                    .setUploadTime(new Date())
                    .build();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取文件信息
     *
     * @param file
     * @param path
     * @param link
     * @return
     */
    public static FileEntity getFileEntity(File file, String path, String link) {
        try {
            if (null == file) return null;
            double kb = ((double) file.length()) / 1024;
            double mb = kb / 1024;
            double gb = mb / 1024;
            String id = UUID.randomUUID().toString().replaceAll("-", "");
            //获取文件原名称
            String name = new String((file.getName().substring(0, file.getName().lastIndexOf("."))).getBytes("GBK"), System.getProperty("sun.jnu.encoding"));
            String extension = file.getName().substring(file.getName().lastIndexOf(".") + 1, file.getName().length());
            return new FileEntity.Builder()
                    .setId(id)
                    .setName(name)
                    .setExtension(extension)
                    .setSize(gb >= 1 ? gb + "GB" : (mb >= 1 ? mb + "MB" : kb + "KB"))
                    .setPath(getPath(path) + File.separator + "temporary" + File.separator + id + "." + extension)
                    .setLink(link + "/" + "temporary" + "/" + id + "." + extension)
                    .setUploadTime(new Date())
                    .build();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取文件信息
     *
     * @param file
     * @return
     */
    public static FileEntity getFileEntity(File file) {
        try {
            if (null == file) return null;
            double kb = ((double) file.length()) / 1024;
            double mb = kb / 1024;
            double gb = mb / 1024;
            String id = UUID.randomUUID().toString().replaceAll("-", "");
            //获取文件原名称
            String name = new String((file.getName().substring(0, file.getName().lastIndexOf("."))).getBytes("GBK"), System.getProperty("sun.jnu.encoding"));
            String extension = file.getName().substring(file.getName().lastIndexOf(".") + 1, file.getName().length());
            return new FileEntity.Builder()
                    .setId(id)
                    .setName(name)
                    .setExtension(extension)
                    .setSize(gb >= 1 ? gb + "GB" : (mb >= 1 ? mb + "MB" : kb + "KB"))
                    .setPath(file.getPath())
                    .setLink(null)
                    .setUploadTime(new Date())
                    .build();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 文件迁移
     *
     * @param source
     * @param dest
     * @throws IOException
     */
    public static void copyFileUsingFileChannels(File source, File dest) throws Exception {
        FileChannel inputChannel = null;
        FileChannel outputChannel = null;
        try {
            inputChannel = new FileInputStream(source).getChannel();
            outputChannel = new FileOutputStream(dest).getChannel();
            outputChannel.transferFrom(inputChannel, 0, inputChannel.size());
        } catch (Exception e) {
            throw new Exception("文件上传异常:" + e.getMessage());
        } finally {
            inputChannel.close();
            outputChannel.close();
        }
    }

    /**
     * 获取随机数字字符串
     *
     * @return
     */
    public static String getRandom() {
        Random random = new Random();
        // 生成随机文件名：当前年月日时分秒+五位随机数（为了在实际项目中防止文件同名而进行的处理）
        int rannum = (int) (random.nextDouble() * (99999 - 10000 + 1)) + 10000; // 获取随机数
        SimpleDateFormat sDateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
        String nowTimeStr = sDateFormat.format(new Date()); // 当前时间
        return nowTimeStr + rannum;
    }

    /**
     * 创建文件夹
     *
     * @param folderPath 文件夹路径
     * @return
     * @throws Exception
     */
    public static boolean createFolder(String folderPath) throws Exception {
        try {
            if (StringUtils.isBlank(folderPath))
                throw new Exception("folderPath is not null!");
            File dirPath = new File(folderPath);
            //如果文件夹不存在，则创建文件夹
            if (!dirPath.exists()) {
                dirPath.mkdirs();
            }
            return true;
        } catch (Exception e) {
            throw new Exception("创建文件夹异常：" + e.getMessage());
        }
    }

    /**
     * 创建文件所在的文件夹
     *
     * @param fileFolder
     * @return
     * @throws Exception
     */
    public static boolean createFileFolder(String fileFolder) throws Exception {
        try {
            if (StringUtils.isBlank(fileFolder))
                throw new Exception("fileFolder is not null!");
            int dir = fileFolder.lastIndexOf(File.separator);
            File dirPath = new File(fileFolder.substring(0, dir));
            //如果文件夹不存在，则创建文件夹
            if (!dirPath.exists()) {
                dirPath.mkdirs();
            }
            return true;
        } catch (Exception e) {
            throw new Exception("创建文件夹异常：" + e.getMessage());
        }
    }

    /**
     * 删除文件
     *
     * @param filePath 文件路径
     * @return
     * @throws Exception
     */
    public static boolean deleteFile(String filePath) throws Exception {
        try {
            if (StringUtils.isBlank(filePath))
                throw new Exception("filePath is not null!");
            File myFilePath = new File(filePath);
            if (myFilePath.exists()) {
                return myFilePath.delete();
            }
            return true;
        } catch (Exception e) {
            throw new Exception("删除文件夹异常：" + e.getMessage());
        }
    }

    /**
     * 创建文件, 如果文件夹不存在则创建文件夹
     *
     * @param file file类
     */
    public static File createFile(File file) {
        if (file.exists() && file.isFile()) {
            file.delete();
            try {
                file.createNewFile();
            } catch (IOException e) {
                e.printStackTrace();
            }
            return file;
        }
        File parentFile = file.getParentFile();
        if (parentFile.exists()) {
            if (parentFile.isFile()) {
                parentFile.delete();
                parentFile.mkdirs();
            }
        } else {
            parentFile.mkdirs();
        }
        try {
            file.createNewFile();
            return file;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }


    /**
     * 删除文件夹
     *
     * @param folderPath
     * @return
     * @throws Exception
     */
    public static boolean deleteFolder(String folderPath) throws Exception {
        try {
            if (StringUtils.isBlank(folderPath))
                throw new Exception("folderPath is not null!");
            deleteAllFile(folderPath); //删除完里面所有内容
            File myFilePath = new File(folderPath);
            myFilePath.delete(); //删除空文件夹
            if (!(new File(folderPath).exists())) {
                return true;
            }
            return false;
        } catch (Exception e) {
            throw new Exception("删除文件夹异常：" + e.getMessage());
        }
    }

    /**
     * 判断是否存在问文件或者文件夹
     *
     * @param path
     * @return
     * @throws Exception
     */
    public static boolean checkFileOrFolder(String path) {
        try {
            if (StringUtils.isBlank(path))
                return false;
            path = checkPath(path);
            File file = new File(path);
            if (file.exists() || file.isDirectory()) {
                return true;
            }
            return false;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 删除文件夹下的文件
     *
     * @param filePath
     * @return
     * @throws Exception
     */
    public static boolean deleteAllFile(String filePath) throws Exception {
        try {
            if (StringUtils.isBlank(filePath))
                throw new Exception("filePath is not null!");
            boolean flag = false;
            File file = new File(filePath);
            if (!file.exists()) {
                return flag;
            }
            if (!file.isDirectory()) {
                return flag;
            }
            String[] tempList = file.list();
            File temp = null;
            for (int i = 0; i < tempList.length; i++) {
                if (filePath.endsWith(File.separator)) {
                    temp = new File(filePath + tempList[i]);
                } else {
                    temp = new File(filePath + File.separator + tempList[i]);
                }
                if (temp.isFile()) {
                    temp.delete();
                }
                if (temp.isDirectory()) {
                    deleteAllFile(filePath + "/" + tempList[i]);//先删除文件夹里面的文件
                    deleteFolder(filePath + "/" + tempList[i]);//再删除空文件夹
                    flag = true;
                }
            }
            return flag;
        } catch (Exception e) {
            throw new Exception("删除文件夹下的文件异常：" + e.getMessage());
        }
    }

    /**
     * 获取文件夹父路径
     *
     * @param filePath
     * @return
     * @throws Exception
     */
    public static String getParent(String filePath) throws Exception {
        try {
            if (StringUtils.isBlank(filePath))
                throw new Exception("路径不能为空!");
            File file = new File(filePath);
            if (null == file)
                throw new Exception("文件不存在!");
            if (!file.exists())
                throw new Exception("文件不存在!");
            return file.getParent();
        } catch (Exception e) {
            throw new Exception("获取文件父节点失败: " + e.getMessage());
        }
    }

    /**
     * 保存文件
     *
     * @param file     文件流
     * @param savePath 文件路径
     * @return
     * @throws Exception
     */
    public static boolean saveFile(File file, String savePath) throws Exception {
        try {
            //创建文件夹
            createFileFolder(savePath);
            FileUtils.copyFile(file, new File(savePath));
            return true;
        } catch (Exception e) {
            throw new Exception("保存文件失败：" + e.getMessage());
        }
    }


    /**
     * 保存文件
     *
     * @param multipartFile 文件流
     * @param savePath      文件路径
     * @return
     * @throws Exception
     */
    public static boolean saveMultipartFile(MultipartFile multipartFile, String savePath) throws Exception {
        //保存文件
        InputStream in = null;
        try {
            //创建文件夹
            createFileFolder(savePath);
            in = multipartFile.getInputStream();
            File out = new File(savePath);
            FileUtils.copyInputStreamToFile(in, out);
            return true;
        } catch (Exception e) {
            throw new Exception("保存文件失败：" + e.getMessage());
        } finally {
            if(null != in){
                try {
                    in.close();
                }catch (Exception e){
                    throw new Exception("保存文件失败：" + e.getMessage());
                }
            }
        }
    }


    /**
     * 美化json数据
     *
     * @param jsonStr
     * @return
     * @throws Exception
     */
    public static String toBeautifyJson(String jsonStr) throws Exception {
        try {
            if (StringUtils.isBlank(jsonStr))
                throw new Exception("jsonStr is not null!");
            ObjectMapper mapper = new ObjectMapper();
            Object obj = mapper.readValue(jsonStr, Object.class);
            return mapper.writerWithDefaultPrettyPrinter().writeValueAsString(obj);
        } catch (Exception e) {
            throw new Exception("格式化json异常：" + e.getMessage());
        }
    }

    /**
     * 读取该根节点下的所有文件
     *
     */
   /* public static void getFileEntityList(File dir, List<FileEntity> fileEntityList) {
        try {
            if (!dir.exists() || null == dir) {
                return;
            }
            if (dir.isDirectory()) {
                String[] children = dir.list();
                for (int i = 0; i < children.length; i++) {
                    getFileEntityList(new File(dir, children[i]), fileEntityList);
                }
            }
            if (dir.isFile()) {
                double kb = ((double) dir.length()) / 1024;
                double mb = kb / 1024;
                double gb = mb / 1024;
                String link = dir.getPath().replaceAll("\\\\", "/");
                String path = rootPath.replaceAll("\\\\", "/");
                fileEntityList.add(new FileEntity.Builder()
                        .setId(UUID.randomUUID().toString().replaceAll("-",""))
                        .setName(dir.getName().substring(0, dir.getName().lastIndexOf(".")))
                        .setPath((link.replaceAll(path, "")).replaceAll("/", "\\\\"))
                        .setExtension(dir.getName().substring(dir.getName().lastIndexOf(".") + 1, dir.getName().length()))
                        .setLink((urlPath + link.replaceAll(path, "")).replaceAll("//", "/"))
                        .setSize(gb >= 1 ? gb + "GB" : (mb >= 1 ? mb + "MB" : kb + "KB"))
                        .build());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }*/
    public static String convertSymbolPath(String path) {
        String symbol = File.separator;
        StringBuilder newPath = new StringBuilder();
        char chars[] = path.toCharArray();
        for (int i = 0; i < chars.length; i++) {
            if (i == 0 && (chars[i] == '/' || chars[i] == '\\')) {
                newPath.append(symbol);
                continue;
            }
            if (i != 0 && (chars[i] == '/' || chars[i] == '\\') && (chars[i - 1] == '/' || chars[i - 1] == '\\')) {
                continue;
            }
            if (i != 0 && (chars[i] == '/' || chars[i] == '\\') && (chars[i - 1] != '/' || chars[i - 1] != '\\')) {
                newPath.append(symbol);
                continue;
            }
            newPath.append(chars[i]);
        }
        return newPath.toString();
    }

    public static String getPath(String path) {
        // OS
        if (("/").equals(File.separator)) {
            path = path.replaceAll("\\\\", "/");
            path = path.replaceAll("//", "/");
        }
        //Win
        if (("\\").equals(File.separator)) {
            path = path.replaceAll("//", "\\\\");
            path = path.replaceAll("/", "\\\\");
            path = path.replaceAll("\\\\", "\\\\");
        }
        return path;
    }

    public static void download(HttpServletRequest request, HttpServletResponse response, File file) {
        FileEntity fileEntity = FileUtil.getFileEntity(file);
        if (file.exists()) {
            //设置下载参数
            setHeader(request, response, fileEntity.getName());
            byte[] buffer = new byte[1024];
            FileInputStream fis = null;
            BufferedInputStream bis = null;
            try {
                fis = new FileInputStream(file);
                bis = new BufferedInputStream(fis);
                OutputStream os = response.getOutputStream();
                int i = bis.read(buffer);
                while (i != -1) {
                    os.write(buffer, 0, i);
                    i = bis.read(buffer);
                }
            } catch (Exception e) {
                setResponseJson(response, "下载失败!");
            } finally {
                if (bis != null) {
                    try {
                        bis.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
                if (fis != null) {
                    try {
                        fis.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        } else {
            setResponseJson(response, "文件不存在，下载失败!");
            return;
        }
    }

    /**
     * 设置下载参数
     *
     * @param request
     * @param response
     * @param fileName
     */
    public static boolean setHeader(HttpServletRequest request, HttpServletResponse response, String fileName) {
        try {
            response.setContentType("application/octet-stream");
            response.setHeader("content-type", "application/octet-stream");
            String browser = request.getHeader("User-Agent");
            if (-1 < browser.indexOf("MSIE 6.0") || -1 < browser.indexOf("MSIE 7.0")) {
                // IE6, IE7 浏览器
                response.addHeader("content-disposition", "attachment;filename="
                        + new String(fileName.getBytes(), "ISO8859-1"));
            } else if (-1 < browser.indexOf("MSIE 8.0")) {
                // IE8
                response.addHeader("content-disposition", "attachment;filename="
                        + URLEncoder.encode(fileName, "UTF-8"));
            } else if (-1 < browser.indexOf("MSIE 9.0")) {
                // IE9
                response.addHeader("content-disposition", "attachment;filename="
                        + URLEncoder.encode(fileName, "UTF-8"));
            } else if (-1 < browser.indexOf("Chrome")) {
                // 谷歌
                response.addHeader("content-disposition",
                        "attachment;filename*=UTF-8''" + URLEncoder.encode(fileName, "UTF-8"));
            } else if (-1 < browser.indexOf("Safari")) {
                // 苹果
                response.addHeader("content-disposition", "attachment;filename="
                        + new String(fileName.getBytes(), "ISO8859-1"));
            } else {
                // 火狐或者其他的浏览器
                response.addHeader("content-disposition",
                        "attachment;filename*=UTF-8''" + URLEncoder.encode(fileName, "UTF-8"));
            }
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 设置json返回
     *
     * @param response
     * @param json
     */
    public static boolean setResponseJson(HttpServletResponse response, String json) {
        try {
            response.setCharacterEncoding("utf-8");
            response.setContentType("text/json;charset=utf-8");
            PrintWriter out = null;
            try {
                out = response.getWriter();
                out.println(json);
            } catch (IOException ex) {
                return false;
            } finally {
                if (null != out)
                    out.close();
            }
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    public static String checkPath(String path) {
        try {
            if (StringUtils.isBlank(path)) {
                return path;
            }
            //Win
            if (("\\").equals(System.getProperty("file.separator"))) {
                return path.replaceAll("\\\\", System.getProperty("file.separator") + System.getProperty("file.separator"));
            } else {
                return path.replaceAll("\\\\", System.getProperty("file.separator"));
            }
        } catch (Exception e) {
            return path;
        }
    }

    public static void main(String[] args) {
        System.out.println(checkPath("test\\com\\jeecms\\common\\util\\1.bmp"));
    }
}

