package sinosoftgz.utils.encrypt;

import lombok.extern.slf4j.Slf4j;
import sinosoftgz.utils.data.StandardCharset;
import sinosoftgz.utils.encrypt.impl.MD5EncryptHandler;
import sinosoftgz.utils.encrypt.impl.SHA1EncryptHandler;
import sinosoftgz.utils.encrypt.impl.SHA256EncryptHandler;
import sinosoftgz.utils.encrypt.impl.SHA512EncryptHandler;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 通用散列加密工具包
 * <p>目前已支持的有MD5、SHA1、SHA256加密，如需扩展，请在impl包下增加名为[<加密方式>EncryptHandler]且继承AbstratEncryptHandler的加密处理器</p>
 * @author Liang Wenxu
 * @since 2018/12/18
 */
public class EncryptUtil {
    private static Map<Type, EncryptHandler> handlerMap = new ConcurrentHashMap<>();

    /**
     * 加密工具
     * @param src 源字符串
     * @param salt 加密盐
     * @param type 加密类型{@link Type}
     * @return 返回加密后的字符串
     */
    public static String encrypt(String src, String salt, Type type) throws UnsupportedEncodingException, NoSuchAlgorithmException {
        EncryptHandler handler = handlerMap.get(type);
        if(handler == null) {
            switch (type) {
                case MD5:
                    handler = new MD5EncryptHandler();
                    break;
                case SHA1:
                    handler = new SHA1EncryptHandler();
                    break;
                case SHA256:
                    handler = new SHA256EncryptHandler();
                    break;
                case SHA512:
                    handler = new SHA512EncryptHandler();
                    break;
                default:
                    return src;
            }

            handlerMap.put(type, handler);
        }
        return handler.encrypt(src, salt == null ? "" : salt, StandardCharset.UTF_8);
    }


    /**
     * 加密类型
     */
    public enum Type {
        /**
         * MD5加密
         */
        MD5
        /**
         * SHA1加密
         */
        ,SHA1
        /**
         * SHA256加密
         */
        ,SHA256
        /**
         * SHA512加密
         */
        ,SHA512;

        /**
         * 根据名称解析加密类型
         * @param name 可选"SHA-1"，"SHA-256", "SHA-512", "MD5"
         * @return
         */
        public static Type fromName(String name) {
            switch (name) {
                case "SHA-1":
                    return SHA1;
                case "SHA-256":
                    return SHA256;
                case "SHA-512":
                    return SHA512;
                case "MD5":
                    return MD5;
                default:
                    return null;
            }
        }

    }
}
