package com.sinosoftgz.starter.sms.core.biz.impl;

import com.sinosoftgz.global.common.request.BaseRequest;
import com.sinosoftgz.starter.sms.core.biz.SmsSendBiz;
import com.sinosoftgz.starter.sms.core.enums.SmsPlatformType;
import com.sinosoftgz.starter.sms.core.handler.AbstractSmsHandler;
import com.sinosoftgz.starter.sms.core.handler.SmsHandlerContext;
import com.sinosoftgz.starter.sms.core.properties.SmsPlatformProperties;
import com.sinosoftgz.starter.sms.core.request.JavaSmsClientReq;
import com.sinosoftgz.starter.sms.core.request.MultiSmsSendReq;
import com.sinosoftgz.starter.sms.core.request.SmsSendReq;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * @author: Roney
 * @date: 2022-05-09 17:28
 * @description: 短信发送业务逻辑实现
 */
@Component
@Slf4j
public class SmsSendBizImpl implements SmsSendBiz {

    @Resource
    SmsHandlerContext smsHandlerContext;

    @Autowired
    SmsPlatformProperties smsPlatformProperties;


    @Override
    public boolean sendSimpleSms(String recipientTo, String content) {
        checkRecipientTo(recipientTo);
        try {
            AbstractSmsHandler handler = getAbstractSmsHandler();
            JavaSmsClientReq javaSmsClientReq = new JavaSmsClientReq.JavaSmsClientReqBuilder()
                    .content(content)
                    .recipientTos(Arrays.asList(recipientTo))
                    .smsPlatformType(getSmsPlatformType().getType())
                    .build();
            return handler.send(javaSmsClientReq);
        } catch (Exception e) {
            log.error("发送短信异常", e);
            return false;
        }
    }

    @Override
    public boolean sendSms(BaseRequest<SmsSendReq> smsSendReq) {
        Assert.notNull(smsSendReq, "短信发送请求对象不能为空");
        Assert.notNull(smsSendReq.getData(), "短信发送请求对象不能为空");
        Assert.notNull(smsSendReq.getData().getRecipientTo(), "接收人不能为空");
        try {
            AbstractSmsHandler handler = getAbstractSmsHandler();

            SmsSendReq smsSendReqReq = smsSendReq.getData();
            JavaSmsClientReq javaSmsClientReq = new JavaSmsClientReq.JavaSmsClientReqBuilder()
                    .content(smsSendReqReq.getContent())
                    .smsPlatformType(getSmsPlatformType().getType())
                    .recipientTos(Arrays.asList(smsSendReqReq.getRecipientTo()))
                    .build();
            return handler.send(javaSmsClientReq);
        } catch (Exception e) {
            log.error("发送短信异常", e);
            return false;
        }
    }

    @Override
    public boolean batchSendSms(BaseRequest<List<SmsSendReq>> smsSendReqs) {
        Assert.notNull(smsSendReqs, "短信发送请求对象不能为空");
        Assert.notEmpty(smsSendReqs.getData(), "短信发送请求对象不能为空");
        try {
            List<SmsSendReq> smsSendReqList = smsSendReqs.getData();
            if (!CollectionUtils.isEmpty(smsSendReqList)) {
                smsSendReqList.parallelStream().forEach((smsSendReq -> {
                    this.sendSms(BaseRequest.simple(smsSendReq));
                }));
            }
        } catch (Exception e) {
            log.error("发送短信异常", e);
            return false;
        }
        return true;
    }

    @Override
    public boolean multiSendSms(BaseRequest<MultiSmsSendReq> multiSmsSendReq) {
        Assert.notNull(multiSmsSendReq, "短信发送请求对象不能为空");
        Assert.notNull(multiSmsSendReq.getData(), "短信发送请求对象不能为空");
        Assert.notEmpty(multiSmsSendReq.getData().getRecipientTos(), "接收人列表不能为空");
        try {
            AbstractSmsHandler handler = getAbstractSmsHandler();

            MultiSmsSendReq multiSmsSendReqReq = multiSmsSendReq.getData();
            JavaSmsClientReq javaSmsClientReq = new JavaSmsClientReq.JavaSmsClientReqBuilder()
                    .content(multiSmsSendReqReq.getContent())
                    .smsPlatformType(getSmsPlatformType().getType())
                    .recipientTos(multiSmsSendReqReq.getRecipientTos())
                    .build();
            return handler.multiSend(javaSmsClientReq);
        } catch (Exception e) {
            log.error("发送短信异常", e);
            return false;
        }
    }

    /**
     * 校验接收人信息
     *
     * @param recipientTo
     */
    private void checkRecipientTo(String recipientTo) {
        Assert.notNull(recipientTo, "接收人不能为空");
    }

    /**
     * 获取处理器信息
     *
     * @return
     */
    private AbstractSmsHandler getAbstractSmsHandler() {
        return (AbstractSmsHandler) smsHandlerContext.getInstance(getSmsPlatformType());
    }

    /**
     * 获取短信平台
     *
     * @return
     */
    private SmsPlatformType getSmsPlatformType() {
        String type = smsPlatformProperties.getType();
        SmsPlatformType smsPlatformType = SmsPlatformType.MOCK;
        if (StringUtils.hasLength(type)) {
            if (!Objects.isNull(SmsPlatformType.smsPlatformTypeMap.get(type))) {
                smsPlatformType = SmsPlatformType.smsPlatformTypeMap.get(type);
            }
        }
        return smsPlatformType;
    }
}
