package ctscore.web.service.cart.service;

import cart.api.CartApi;
import cart.api.CartItemApi;
import cart.api.dto.cart.CartDto;
import cart.api.dto.cart.CartItemDto;
import cart.api.vo.CartItemGroupVo;
import cart.api.vo.CartItemVo;
import cart.api.vo.CartStoreInfoVo;
import cms.api.OrganizationApi;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import goods.api.*;
import goods.dto.goods.GoodsDto;
import goods.dto.product.ProductFashionDto;
import ma.glasnost.orika.MapperFacade;
import member.api.AddressCommonApi;
import member.api.MemberApi;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import store.api.StoreApi;
import store.api.dto.modeldto.core.StoreDto;
import utils.GlobalContants;
import utils.Lang;
import utils.lang.Copys;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 封装购物车前端数据Service
 * Created by Liang Wenxu on 2016/10/28.
 */
@Component
public class CartWebService {
    Logger log  = Logger.getLogger(CartWebService.class);
    @Value("${ctscore.score.proportion}")
    private BigDecimal scorePortion;//积分比例

    @Value("${ctscore.orgCode}")
    private String organizationCode; // 合作企业代码

    @Autowired
    MapperFacade mapperFacade;

    /**
     * 获取按店铺分组的购物车商品列表
     *
     * @param cartId 购物车ID
     * @return Map<String, CartStoreInfoVo> 返回集合中将以店铺的storeId作为索引
     */
    public Map<String, CartStoreInfoVo> findCartItemInfoGroupByStrore(String cartId) {
        Map<String, CartStoreInfoVo> cartItemInfoMap = new HashMap<String, CartStoreInfoVo>();
        if (StringUtils.isBlank(cartId)) {
            return cartItemInfoMap;
        }


        return this.findCartItemInfoGroupByStrore(cartApi.get(cartId));
    }

    /**
     * 获取按店铺分组的购物车商品列表
     *
     * @param cart 购物车对象
     * @return Map<String, CartStoreInfoVo> 返回集合中将以店铺的storeId作为索引
     */
    public Map<String, CartStoreInfoVo> findCartItemInfoGroupByStrore(CartDto cart) {
        Map<String, CartStoreInfoVo> cartItemInfoMap = new HashMap<String, CartStoreInfoVo>();
        if (cart == null) {
            return cartItemInfoMap;
        }

        /** 对象复制器 */
        Copys copys = Copys.create();
        CartStoreInfoVo cartStoreInfo = null;
        List<Map> cartItemList;
        CartItemVo cartItemVo;
        for (CartItemDto item : cart.getCartItems()) {
            /** 使用storeId索引各个商店下的商品列表 */
            cartStoreInfo = (CartStoreInfoVo) cartItemInfoMap.get(item.getStoreId());

            if (cartStoreInfo == null) {
                cartStoreInfo = new CartStoreInfoVo();
                /** 处理店铺信息 */
                StoreDto s = storeApi.getStoreInfo(item.getStoreId());
                if (s != null) {
                    copys.from(s).excludes("cartItems").to(cartStoreInfo).clear();
                }
                cartStoreInfo.setCartItems(new ArrayList<CartItemVo>());
                cartStoreInfo.setStoreExt(mapperFacade.map(s.getStoreExt(),cart.api.dto.store.StoreExtDto.class));
            }


            cartItemVo = new CartItemVo();
            copys.from(item).to(cartItemVo).clear();
//            cartItemVo.setProduct(productApi.findById(item.getProductId()));
//            cartItemVo.setProduct(null);
            GoodsDto goods = goodsApi.findById(item.getGoodsId());
//            Goods goods = goodsApi.findGoodsById(item.getGoodsId());
            cartItemVo.setGoods(mapperFacade.map(goods, cart.api.dto.goods.GoodsDto.class));
            cartItemVo.setProductFashion(mapperFacade.map(productFashionApi.findDtoById(item.getProductFashId()),cart.api.dto.goods.ProductFashionDto.class));
            cartItemVo.getProductFashion();
            cartStoreInfo.getCartItems().add(cartItemVo);
            cartItemInfoMap.put(item.getStoreId(), cartStoreInfo);
        }


        return cartItemInfoMap;
    }

    public List<CartItemGroupVo> findCartItemGroups(HttpServletRequest request, CartDto cart) {
        return this.findCartItemGroups(request, cart, "1");
    }

    /**
     * 获取按店铺分组的购物车商品列表
     *
     * @param cart      购物车对象
     * @param groupType 分组方式，1-- 按商户分组，当前只支持按商户分组
     * @return Map<String, CartStoreInfoVo> 返回集合中将以店铺的storeId作为索引
     */
    public List<CartItemGroupVo> findCartItemGroups(HttpServletRequest request, CartDto cart, String groupType) {
        return findCartItemGroups(request, cart, groupType, true, false, null, null);
    }

    /**
     * 获取按店铺分组的购物车商品列表
     *
     * @param cart            购物车对象
     * @param groupType       分组方式，1-- 按商户分组，当前只支持按商户分组
     * @param withStoreDetail 是否包含店铺详情
     * @param refreshStatus: Boolean 是否刷新规格的实时状态（主要针对第三方商品）
     * @param areaCode: String 当refreshStatus为true时不能为空
     * @param cartItemIds     不为空时，只展示存在给出ID列表中的数据
     * @return Map<String, CartStoreInfoVo> 返回集合中将以店铺的storeId作为索引
     */
//    @Transactional(readOnly = true)
    public List<CartItemGroupVo> findCartItemGroups(HttpServletRequest request, CartDto cart, String groupType, Boolean withStoreDetail, Boolean refreshStatus,
                                                    String areaCode, List<String> cartItemIds) {
        List<CartItemGroupVo> cartItemGroups = new ArrayList<CartItemGroupVo>();
        if (cart == null) {
            return cartItemGroups;
        }

        List<Map<String, Object>> tmpCartItems = null;
        Map ordersTmpMap = (HashMap) request.getSession().getAttribute(GlobalContants.ORDER_CARTITEM_TMPS);
        if(!Lang.isEmpty(ordersTmpMap)) {
            tmpCartItems = (List<Map<String, Object>>) ordersTmpMap.get("tmpCartItems");
        }

        cartItemGroups = cartApi.getCartItemGroups(cart.getId(), groupType, withStoreDetail, refreshStatus, areaCode,
                cartItemIds, tmpCartItems, organizationCode, null);

        /** 重新计算展示用金额合计（计算积分比例） */
      /*  for (CartItemGroupVo groupVo : cartItemGroups) {
            groupVo.setSumPrice(BigDecimal.valueOf(0)); // 积分商城重新计算分组总计
            for(CartItemVo itemVo : groupVo.getCartItems()) {
                itemVo.setSumSalePrice(itemVo.getProductFashion().getSalePrice()
                        .setScale(0, BigDecimal.ROUND_CEILING)
                        .multiply(new BigDecimal(itemVo.getCount()).setScale(0, BigDecimal.ROUND_CEILING)
                                .multiply(scorePortion).setScale(0, BigDecimal.ROUND_CEILING)));

                itemVo.setSumMarketPrice(itemVo.getProductFashion().getMarketPrice().multiply(new BigDecimal(itemVo.getCount())));
                *//** 重新计算分组合计 *//*
                if ((itemVo.getIsDelete() == null || itemVo.getIsDelete() != true) &&
                        itemVo.getStatus().equals(CartItemVo.statusMap.get("NORMAL"))) {
                    if(!Lang.isEmpty(itemVo.getProductFashion()) && !Lang.isEmpty(itemVo.getProductFashion().getSalePrice())) {
//                    gruop.setSumPrice((cartItemVo.getProductFashion().getSalePrice().setScale(0, BigDecimal.ROUND_CEILING).multiply(new BigDecimal(cartItemVo.getCount()).multiply(new BigDecimal(scorePortion)))).add(gruop.getSumPrice()));
                        groupVo.setSumPrice(groupVo.getSumPrice().setScale(0, BigDecimal.ROUND_CEILING).add(itemVo.getSumSalePrice().setScale(0, BigDecimal.ROUND_CEILING)));
                    }
                }
            }
        }*/
        return cartItemGroups;
    }

    /**
     * 设置购物车的过滤购买数量
     * @param cartItemGroups
     * @param tmpCartItems 临时订单： [{id:"<cartItemId", counts: 1}...]
     * @return
     */
    public List<CartItemGroupVo> genBuyCounts(List<CartItemGroupVo> cartItemGroups, List<Map<String, Object>> tmpCartItems) {
        for(CartItemGroupVo g : cartItemGroups) {
            for(CartItemVo it : g.getCartItems()) {
                for(Map m : tmpCartItems) {
                    if(it.getId().equals((String)m.get("id"))) {
                        it.setCount((Integer) m.get("counts"));
                    }
                }
            }
        }

        return cartItemGroups;
    }

    /**
     * 设置购物车的过滤购买数量
     * @param cartItemVo
     * @param tmpCartItems 临时订单： [{id:"<cartItemId", counts: 1}...]
     * @return
     */
    public CartItemVo genBuyCounts(CartItemVo cartItemVo, List<Map<String, Object>> tmpCartItems) {
        if(!Lang.isEmpty(tmpCartItems) && cartItemVo != null) {
            for(Map m : tmpCartItems) {
                if(cartItemVo.getId().equals((String)m.get("id"))) {
                    cartItemVo.setCount((Integer) m.get("counts"));
                }
            }
        }

        return cartItemVo;
    }

    /**
     * 重新计算购物车每个分组的商品金额小计
     *
     * @param cartItemGroups
     */
    public void recalCartItemGroupSumPrice(List<CartItemGroupVo> cartItemGroups) {
        Double groupSumPrice = 0.00;
        for (CartItemGroupVo group : cartItemGroups) {
            if (group != null && group.getStore() != null) {
                for (CartItemVo itemVo : group.getCartItems()) {
                    if (itemVo.getIsDelete() == null || !itemVo.getIsDelete()) {
                        groupSumPrice += itemVo.getProductFashion().getSalePrice().setScale(0, BigDecimal.ROUND_CEILING).doubleValue() * itemVo.getCount() * scorePortion.doubleValue();
                    }
                }
                group.setSumPrice(new BigDecimal(groupSumPrice));
            }
        }
    }

    /**
     * 获取指定店铺的店铺合计金额
     *
     * @param cartItemGroups 购物车分组列表
     * @param storeId        购物车店铺ID
     * @return 指定店铺的商品金额合计
     */
    public BigDecimal findStoreSumPrice(List<CartItemGroupVo> cartItemGroups, String storeId) {
        return this.findStoreSumPrice(cartItemGroups, storeId, true);
    }

    /**
     * 获取指定店铺的店铺合计金额
     *
     * @param cartItemGroups 购物车分组列表
     * @param storeId        购物车店铺ID
     * @param recal          是否重新计算
     * @return 指定店铺的商品金额合计
     */
    public BigDecimal findStoreSumPrice(List<CartItemGroupVo> cartItemGroups, String storeId, boolean recal) {
        if (StringUtils.isBlank(storeId)) {
            return null;
        }
        if (recal) {
            this.recalCartItemGroupSumPrice(cartItemGroups);
        }
        for (CartItemGroupVo group : cartItemGroups) {
            if (group != null && group.getStore() != null) {
                if (group.getStore().getId().equals(storeId)) {
                    return group.getSumPrice();
                }
            }
        }
        return null;
    }

    /**
     * 计算购物车总价格
     *
     * @param cartItemGroups
     * @return
     */
    public Double calSumCartPrice(List<CartItemGroupVo> cartItemGroups) {
        Double sumCartPrice = 0.00;
        for (CartItemGroupVo group : cartItemGroups) {
            for (CartItemVo item : group.getCartItems()) {
                if (item.getStatus().equals(CartItemVo.statusMap.get("NORMAL")) &&
                        !item.getIsDelete()) {
                    if(!Lang.isEmpty(item.getProductFashion()) && !Lang.isEmpty(item.getProductFashion().getSalePrice())) {
                        sumCartPrice += item.getProductFashion().getSalePrice().setScale(0, BigDecimal.ROUND_CEILING).doubleValue() * item.getCount() * scorePortion.doubleValue();
                    }
                }
            }
        }

        return sumCartPrice;
    }

    /**
     * 计算购物车总市场价
     *
     * @param cartItemGroups
     * @return
     */
    public Double calSumMarketCartPrice(List<CartItemGroupVo> cartItemGroups) {
        Double sumMarketCartPrice = 0.00;
        for (CartItemGroupVo group : cartItemGroups) {
            for (CartItemVo item : group.getCartItems()) {
                if (item.getStatus().equals(CartItemVo.statusMap.get("NORMAL")) &&
                        !item.getIsDelete()) {
                    if(!Lang.isEmpty(item.getProductFashion()) && !Lang.isEmpty(item.getProductFashion().getMarketPrice())) {
                        sumMarketCartPrice += item.getProductFashion().getMarketPrice().doubleValue() * item.getCount();
                    }
                }
            }
        }
        return sumMarketCartPrice;
    }

    /**
     * 计算购物车商品总数
     *
     * @param cartItemGroups
     * @return
     */
    public Integer calSumGoodsCount(List<CartItemGroupVo> cartItemGroups) {
        int sumGoodsCount = 0;
//        for (int i = 0; i < cartItemGroups.size(); i++) {
//            for (CartItemVo cartItemVo : cartItemGroups.get(i).getCartItems()) {
//                if (cartItemVo.getIsDelete() == null || !cartItemVo.getIsDelete()) {
//                    sumGoodsCount += 1;
//                }
//            }
//        }
        for(CartItemGroupVo g : cartItemGroups) {
            sumGoodsCount += g.getAllItemCounts();
        }
        return sumGoodsCount;
    }

    /**
     * 计算购物车商品件数合计
     *
     * @param cartItemGroups
     * @return
     */
    public Integer calSumCartItemCounts(List<CartItemGroupVo> cartItemGroups) {
        int sumGoodsCount = 0;
        for (int i = 0; i < cartItemGroups.size(); i++) {
            for (CartItemVo cartItemVo : cartItemGroups.get(i).getCartItems()) {
                if (cartItemVo.getIsDelete() == null || !cartItemVo.getIsDelete()) {
                    sumGoodsCount += cartItemVo.getCount() > 0 ? cartItemVo.getCount() : 1;
                }
            }
        }
        return sumGoodsCount;
    }

    public CartItemGroupVo findCartItemGroupInListByStore(List<CartItemGroupVo> cartItemGroupVoList, String storeId) {
        for (CartItemGroupVo i : cartItemGroupVoList) {
            if (i.getStore() != null && i.getStore().getId().equals(storeId)) {
                return i;
            }
        }

        return null;
    }

    /**
     * 删除用户购物车中的下架商品
     *
     * @param userId 用户UserId
     * @return 删除处理结果返回结果集:
     * [ total: 总处理数据量,
     * sucCount: 成功计数,
     * faildCount: 失败计数,
     * detials: [
     * id: 处理记录ID
     * force: 是否物理删除
     * error: 处理结果，0为成功，1为错误
     * msg: 处理信息
     * ] 处理详情
     * ]
     */
    public Map<String, Object> deleteInvalidItems(String userId) {

        CartDto cart = cartApi.getUserCart(userId);
        if (cart == null) {
            return null;
        }
        List<String> delIds = new ArrayList<>();
        for (CartItemDto item : cart.getCartItems()) {
            /** 优先判断goods状态 */
            if(!Lang.isEmpty(item.getGoodsId())) {
                GoodsDto goods = goodsApi.findById(item.getGoodsId());
                if(!Lang.isEmpty(goods) && goods.getState() != 1) { // goods表已标记下架
                    delIds.add(item.getId());
                }
            }

            if(!delIds.contains(item.getId())) {
                ProductFashionDto fashion = productFashionApi.findById(item.getProductFashId());
                if (fashion == null) {
                    /** 产品已被删除，视为无效商品 */
                    delIds.add(item.getId());
                } else {
                    if (fashion.getOnSale() != null && !fashion.getOnSale()) {
                        /** 下架商品 */
                        delIds.add(item.getId());
                    }
                }
            }
        }

        return cartApi.deleteItem(cart.getId(), delIds.toArray(new String[delIds.size()]));
//        return cartService.deleteInvalidItems(cartId);
    }



    @MotanReferer
//@MotanReferer(directUrl = "127.0.0.1:8189", group = "sinomall")
    CartApi cartApi;

    @MotanReferer
    MemberApi memberApi;

    @MotanReferer
    StoreApi storeApi;

    @MotanReferer
    GoodsApi goodsApi;

    @MotanReferer
    ProductApi productApi;

//    @MotanReferer
//    GoodsFashionApi productFashionApi;

    @MotanReferer
    ProductFashionApi productFashionApi;

    @MotanReferer
    CartItemApi cartItemApi;

    @MotanReferer
    AddressCommonApi addressCommonApi;

    //@MotanReferer
    //OutsideProductApi outsideProductApi;

    @MotanReferer
    ShareApi shareApi;

    @MotanReferer
    SalesPriceRateApi salesPriceRateApi;

    @MotanReferer
    OrganizationApi organizationApi;

    public BigDecimal getScorePortion() {
        return scorePortion;
    }

    public void setScorePortion(BigDecimal scorePortion) {
        this.scorePortion = scorePortion;
    }

    public String getOrganizationCode() {
        return organizationCode;
    }

    public void setOrganizationCode(String organizationCode) {
        this.organizationCode = organizationCode;
    }
}
