package ctscore.web.service.member;

import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFactory;
import ma.glasnost.orika.impl.DefaultMapperFactory;
import member.api.dto.core.CoreUserDto;
import member.api.dto.shop.MemberDto;
import member.model.core.CoreUser;
import member.model.repository.core.UserRepos;
import member.model.repository.member.MemberRepos;
import member.model.shop.Member;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import score.api.ScoreApi;
import score.api.dto.entitydto.ScoreAccountDTO;
import score.api.dto.entitydto.ScoreGiveAwayLogDTO;
import score.model.ScoreAccount;
import score.model.ScoreGiveAwayLog;
import score.model.repository.ScoreAccountRepos;
import score.model.repository.ScoreGiveAwayLogRepos;
import utils.Lang;
import utils.data.BeanMapper;
import utils.lang.Copys;
import utils.lock.RedisLockUtil;

import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

@Slf4j
@Service
public class MemberBatchService {

    /**
     * 批量获取 member
     *
     * @param coreUserIds List<String> coreUserIds
     * @return Map<String, MemberDto> key = coreUserId, value = memberDTO
     */
    public Map<String, MemberDto> findMembersByCoreUserIds(List<String> coreUserIds) {
        long findMembersByCoreUserIdsTime = System.currentTimeMillis();
        // 防止并发
        Map<String, MemberDto> memberDTOMap = new ConcurrentHashMap<>(1);
        int page = 0;
        int pageSize = 500;
        int totalPage = coreUserIds.size() / pageSize;
        if (coreUserIds.size() % pageSize > 0) {
            totalPage++;
        }
        log.info("member 批查询数量 {} 条, 单批次查询数量 {} 条, 需查询 {} 次", coreUserIds.size(), pageSize, totalPage);
        log.info("开始多线程查询积 member 数据...");
        ThreadPoolTaskExecutor threadPoolTaskExecutor = null;
        try {
            threadPoolTaskExecutor = new ThreadPoolTaskExecutor();
            threadPoolTaskExecutor.setCorePoolSize(10);
            threadPoolTaskExecutor.setMaxPoolSize(100);
            threadPoolTaskExecutor.setQueueCapacity(2000);
            threadPoolTaskExecutor.setKeepAliveSeconds(60);
            threadPoolTaskExecutor.setRejectedExecutionHandler(new ThreadPoolExecutor.AbortPolicy());
            threadPoolTaskExecutor.initialize();
            CountDownLatch countDownLatch = new CountDownLatch(totalPage);
            for (; page < totalPage; page++) {
                int startIndex = pageSize * page;
                int endIndexTemp = pageSize * (page + 1);
                if (endIndexTemp > coreUserIds.size()) {
                    endIndexTemp = coreUserIds.size();
                }
                int endIndex = endIndexTemp;
                threadPoolTaskExecutor.execute(() -> {
                    try {
                        List<Member> memberPOs = memberRepos.findByCoreUserIdIn(coreUserIds.subList(startIndex, endIndex));
                        getCoreUserMemberMap(memberPOs, memberDTOMap);
                    } catch (Exception e) {
                        log.error("分页查询 member 数据异常", e);
                    } finally {
                        countDownLatch.countDown();
                    }
                });
            }
            // 阻塞线程
            countDownLatch.await();
        } catch (Exception e) {
            log.error("多线程分页查询 member 异常", e);
        } finally {
            if (!Lang.isEmpty(threadPoolTaskExecutor)) {
                threadPoolTaskExecutor.shutdown();
            }
        }
        log.info("多线程获取 member {} 条数据, 耗时 {} 秒", memberDTOMap.size(), (System.currentTimeMillis() - findMembersByCoreUserIdsTime) / 1000F);
        return memberDTOMap;
    }

    /**
     * 封装 member 数据
     * member 对象中除 coreUser 外其他表关联对象均不提供数据, coreUser 对象中 coreUserExt 表关联对象不提供数据
     *
     * @param memberPOs Member
     * @param memberDTOMap Map<String, MemberDto> 如果在多线程中请求则要求该对象实例化方式为 ConcurrentHashMap
     * @return Map<String, MemberDto> key = coreUserId, value = memberDTO
     */
    private Map<String, MemberDto> getCoreUserMemberMap(List<Member> memberPOs, Map<String, MemberDto> memberDTOMap) {
        Copys copys = Copys.create();
        memberPOs.forEach(memberPO -> {
            if (!memberDTOMap.containsKey(memberPO.getCoreUser().getId())) {
                MemberDto memberDTO = new MemberDto();
                copys.from(memberPO).excludes("coreUser", "memberGroup", "addresses", "contactses", "creditses", "coupons", "moneys", "viewHises", "favorites", "favorites", "roles").to(memberDTO).clear();

                CoreUserDto coreUserDTO = new CoreUserDto();
                copys.from(memberPO.getCoreUser()).excludes("coreUserExt").to(coreUserDTO).clear();
                memberDTO.setCoreUser(coreUserDTO);

                memberDTOMap.put(memberPO.getCoreUser().getId(), memberDTO);
            }
        });
        return memberDTOMap;
    }

    /**
     * 获取 member 总数据量
     * @param member MemberDto
     * @return int
     */
    public int getMemberCount(MemberDto member) {
        Specification<Member> memberSpecification = findMemberSpecification(member);
        return (int) memberRepos.count(memberSpecification);
    }

    /**
     *  查询 member
     *
     * @param member MemberDto
     * @param pageable Pageable
     * @return Map<String, MemberDto> key = coreUserId, value = memberDTO
     */
    public Map<String, MemberDto> findMember(MemberDto member, Pageable pageable, List<String> coreUserIdSort) {
        long methodStartTime = System.currentTimeMillis();
        Map<String, MemberDto> memberDTOMap = new HashMap<>(1);
        Specification<Member> memberSpecification = findMemberSpecification(member);
        Page<Member> memberPage = memberRepos.findAll(memberSpecification, pageable);
        coreUserIdSort.addAll(memberPage.getContent().stream().map(Member::getCoreUser).map(CoreUser::getId).distinct().collect(Collectors.toList()));
        getCoreUserMemberMap(memberPage.getContent(), memberDTOMap);
        log.info("一次性获取 member {} 条数据, 耗时 {} 秒", memberPage.getContent().size(), (System.currentTimeMillis() - methodStartTime) / 1000F);
        return memberDTOMap;
    }

    public Map<String, MemberDto> findMember(MemberDto member, Sort sort, List<String> coreUserIdSort) {
        long methodStartTime = System.currentTimeMillis();
        // 线程安全
        Map<String, MemberDto> memberDTOMap = new ConcurrentHashMap<>(1);
        // 线程安全
        Map<Integer, List<String>> coreUserIdPageMap = new ConcurrentHashMap<>(1);
        Specification<Member> memberSpecification = findMemberSpecification(member);
        int page = 0;
        int pageSize = 500;
        int total = (int) memberRepos.count(memberSpecification);
        int totalPage = total / pageSize;
        if(total % pageSize > 0) {
            totalPage++;
        }
        ThreadPoolTaskExecutor threadPoolTaskExecutor = null;
        try {
            threadPoolTaskExecutor = new ThreadPoolTaskExecutor();
            threadPoolTaskExecutor.setCorePoolSize(10);
            threadPoolTaskExecutor.setMaxPoolSize(100);
            threadPoolTaskExecutor.setQueueCapacity(2000);
            threadPoolTaskExecutor.setKeepAliveSeconds(60);
            threadPoolTaskExecutor.setRejectedExecutionHandler(new ThreadPoolExecutor.AbortPolicy());
            threadPoolTaskExecutor.initialize();
            CountDownLatch countDownLatch = new CountDownLatch(totalPage);
            for (; page < totalPage; page++) {
                int currentPage = page;
                threadPoolTaskExecutor.execute(() -> {
                    try {
                        Pageable pageable = new PageRequest(currentPage, pageSize, sort);
                        Page<Member> memberPage = memberRepos.findAll(memberSpecification, pageable);
                        coreUserIdPageMap.put(currentPage, memberPage.getContent().stream().map(Member::getCoreUser).map(CoreUser::getId).distinct().collect(Collectors.toList()));
                        getCoreUserMemberMap(memberPage.getContent(), memberDTOMap);
                    } catch (Exception e) {
                        log.error("分页查询 member 数据异常", e);
                    } finally {
                        countDownLatch.countDown();
                    }
                });
            }
            // 阻塞线程
            countDownLatch.await();
        } catch (Exception e) {
            log.error("多线程分页查询 member 异常", e);
        } finally {
            if (!Lang.isEmpty(threadPoolTaskExecutor)) {
                threadPoolTaskExecutor.shutdown();
            }
        }

        // 按顺序汇总
        for (int i = 0; i < coreUserIdPageMap.size(); i++) {
            coreUserIdSort.addAll(coreUserIdPageMap.get(i));
        }

        log.info("多线程获取 member {} 条数据, 耗时 {} 秒", coreUserIdSort.size(), System.currentTimeMillis() - methodStartTime);
        return memberDTOMap;
    }

    private Specification<Member> findMemberSpecification(final MemberDto member){
        return (root, query, criteriaBuilder) -> {
            List<Predicate> predicates = new ArrayList<>();

            if(!Lang.isEmpty(member)) {
                if(!Lang.isEmpty(member.getRealName())) {
                    predicates.add(criteriaBuilder.like(root.get("realName"), "%" + member.getRealName() + "%"));
                }
                if(!Lang.isEmpty(member.getCoreUser())) {
                    if(!Lang.isEmpty(member.getCoreUser().getOrganizationId())) {
                        predicates.add(criteriaBuilder.equal(root.get("coreUser").get("organizationId"), member.getCoreUser().getOrganizationId()));
                    }
                    if(!Lang.isEmpty(member.getCoreUser().getUserType())) {
                        predicates.add(criteriaBuilder.equal(root.get("coreUser").get("userType"), member.getCoreUser().getUserType()));
                    }
                    if(!Lang.isEmpty(member.getCoreUser().getPhone())) {
                        predicates.add(criteriaBuilder.like(root.get("coreUser").get("phone"), "%" + member.getCoreUser().getPhone() + "%"));
                    }
                    if(!Lang.isEmpty(member.getCoreUser().getEmail())) {
                        predicates.add(criteriaBuilder.like(root.get("coreUser").get("email"), "%" + member.getCoreUser().getEmail() + "%"));
                    }
                    if(!Lang.isEmpty(member.getCoreUser().getIsDelete())) {
                        predicates.add(criteriaBuilder.equal(root.get("coreUser").get("isDelete"), member.getCoreUser().getIsDelete()));
                    }
                }
            }
            return query.where(predicates.toArray(new Predicate[predicates.size()])).getRestriction();
        };
    }

    @Autowired
    MemberRepos memberRepos;

    @Autowired
    UserRepos userRepos;

    @Autowired
    ScoreAccountRepos scoreAccountRepos;

    @Autowired
    MemberBatchTransactionalService memberBatchTransactionalService;

    @Autowired
    ScoreGiveAwayLogRepos scoreGiveAwayLogRepos;

    @Autowired
    RedisLockUtil redisLockUtil;

    /**
     * 批量查询 积分账户信息
     * @param organizationCode 机构代码
     * @param memberDTOMap Map<String, MemberDto> key = coreUserId, value = memberDTO
     * @return List<ScoreAccountDTO>
     */
    public List<ScoreAccountDTO> queryScoreAccount(String organizationCode, Map<String,MemberDto> memberDTOMap) {
        long methodStartTime = System.currentTimeMillis();
        // Vector 线程安全
        List<ScoreAccountDTO> scoreAccountDTOs = new Vector<>();
        int page = 0;
        int pageSize = 500;
        List<String> coreUserIds = new ArrayList<>(memberDTOMap.keySet());
        int totalPage = coreUserIds.size() / pageSize;
        if (coreUserIds.size() % pageSize > 0) {
            totalPage++;
        }
        log.info("scoreAccount 批查询数量 {} 条, 单批次查询数量 {} 条, 需查询 {} 次", memberDTOMap.size(), pageSize, totalPage);
        log.info("开始多线程查询积 scoreAccount 数据...");
        ThreadPoolTaskExecutor threadPoolTaskExecutor = null;
        try {
            threadPoolTaskExecutor = new ThreadPoolTaskExecutor();
            threadPoolTaskExecutor.setCorePoolSize(10);
            threadPoolTaskExecutor.setMaxPoolSize(100);
            threadPoolTaskExecutor.setQueueCapacity(2000);
            threadPoolTaskExecutor.setKeepAliveSeconds(60);
            threadPoolTaskExecutor.setRejectedExecutionHandler(new ThreadPoolExecutor.AbortPolicy());
            threadPoolTaskExecutor.initialize();
            CountDownLatch countDownLatch = new CountDownLatch(totalPage);
            for (; page < totalPage; page++) {
                int startIndex = pageSize * page;
                int endIndexTemp = pageSize * (page + 1);
                if (endIndexTemp > coreUserIds.size()) {
                    endIndexTemp = coreUserIds.size();
                }
                int endIndex = endIndexTemp;
                threadPoolTaskExecutor.execute(() -> {
                    try {
                        List<ScoreAccount> scoreAccountPOs = scoreAccountRepos.findByOrganizationCodeAndUserIdIn(organizationCode, coreUserIds.subList(startIndex, endIndex));
                        scoreAccountDTOs.addAll(BeanMapper.mapList(scoreAccountPOs, ScoreAccount.class, ScoreAccountDTO.class));
                    } catch (Exception e) {
                        log.error("分页查询 scoreAccount 数据异常", e);
                    } finally {
                        countDownLatch.countDown();
                    }
                });
            }
            // 阻塞线程
            countDownLatch.await();
        } catch (Exception e) {
            log.error("多线程分页查询 scoreAccount 异常", e);
        } finally {
            if (!Lang.isEmpty(threadPoolTaskExecutor)) {
                threadPoolTaskExecutor.shutdown();
            }
        }
        log.info("多线程获取 scoreAccount {} 条数据, 耗时 {} 秒", scoreAccountDTOs.size(), (System.currentTimeMillis() - methodStartTime) / 1000F);
        return scoreAccountDTOs;
    }

    public Map<String, MemberDto> findScoreGiveData(List<String> coreUserIds, List<String> scoreRcptCodes) {
        long methodStart = System.currentTimeMillis();
        int page = 0;
        int pageSize = 1000;
        int totalPage = scoreRcptCodes.size() / pageSize;
        if(scoreRcptCodes.size() % pageSize > 0) {
            totalPage++;
        }
        List<ScoreGiveAwayLog> scoreGiveAwayLogs = new ArrayList<>();
        for (; page < totalPage; page++) {
            int startIndex = pageSize * page;
            int endIndex = pageSize * (page + 1);
            if (endIndex > scoreRcptCodes.size()) {
                endIndex = scoreRcptCodes.size();
            }
            scoreGiveAwayLogs.addAll(scoreGiveAwayLogRepos.findScoreGiveAwayByExpenseCodes(scoreRcptCodes.subList(startIndex, endIndex)));
        }
        coreUserIds.addAll(scoreGiveAwayLogs.stream().map(ScoreGiveAwayLog::getReceiveScoreAccount).map(ScoreAccount::getUserId).collect(Collectors.toList()));
        coreUserIds = coreUserIds.stream().distinct().collect(Collectors.toList());
        Map<String, MemberDto> memberDTOMap = findMembersByCoreUserIds(coreUserIds);
        scoreGiveAwayLogs.forEach(scoreGiveAwayLog -> {
            MemberDto memberDTO = memberDTOMap.get(scoreGiveAwayLog.getReceiveScoreAccount().getUserId());
            memberDTOMap.put(scoreGiveAwayLog.getScoreExpenseRcptCode(), memberDTO);
        });
        log.info("查询转赠积分数据耗时 {} ms", System.currentTimeMillis() - methodStart);
        return memberDTOMap;
    }

    public Map<String, MemberDto> findScoreGiveData(Map<String, ScoreGiveAwayLogDTO> codeLogMap, List<String> scoreRcptCodes) {
        long methodStart = System.currentTimeMillis();
        int page = 0;
        int pageSize = 1000;
        int totalPage = scoreRcptCodes.size() / pageSize;
        if(scoreRcptCodes.size() % pageSize > 0) {
            totalPage++;
        }
        List<ScoreGiveAwayLog> scoreGiveAwayLogs = new ArrayList<>();
        for (; page < totalPage; page++) {
            int startIndex = pageSize * page;
            int endIndex = pageSize * (page + 1);
            if (endIndex > scoreRcptCodes.size()) {
                endIndex = scoreRcptCodes.size();
            }
            scoreGiveAwayLogs.addAll(scoreGiveAwayLogRepos.findScoreGiveAwayByExpenseCodes(scoreRcptCodes.subList(startIndex, endIndex)));
        }

        MapperFactory mapperFactory = new DefaultMapperFactory.Builder().build();
        mapperFactory.classMap(ScoreGiveAwayLog.class, ScoreGiveAwayLogDTO.class).exclude("giveAwayScoreAccount").exclude("receiveScoreAccount").byDefault().register();

        // 匹配 scoreRcptCode 封装 map
        for (ScoreGiveAwayLog scoreGiveAwayLog : scoreGiveAwayLogs) {
            for (String scoreRcptCode : scoreRcptCodes) {
                if(scoreRcptCode.equals(scoreGiveAwayLog.getScoreExpenseRcptCode())) {
                    codeLogMap.put(scoreRcptCode, mapperFactory.getMapperFacade().map(scoreGiveAwayLog, ScoreGiveAwayLogDTO.class));
                    break;
                }
            }
        }

        List<String> coreUserIds = new ArrayList<>();
        coreUserIds.addAll(scoreGiveAwayLogs.stream().map(ScoreGiveAwayLog::getReceiveScoreAccount).map(ScoreAccount::getUserId).collect(Collectors.toList()));
        coreUserIds = coreUserIds.stream().distinct().collect(Collectors.toList());
        Map<String, MemberDto> memberDTOMap = findMembersByCoreUserIds(coreUserIds);
        scoreGiveAwayLogs.forEach(scoreGiveAwayLog -> {
            MemberDto memberDTO = memberDTOMap.get(scoreGiveAwayLog.getReceiveScoreAccount().getUserId());
            memberDTOMap.put(scoreGiveAwayLog.getScoreExpenseRcptCode(), memberDTO);
        });
        log.info("查询转赠积分数据耗时 {} ms", System.currentTimeMillis() - methodStart);
        return memberDTOMap;
    }

    public Map<String, MemberDto> findScoreGetData(List<String> coreUserIds, List<String> scoreRcptCodes) {
        long methodStart = System.currentTimeMillis();
        int page = 0;
        int pageSize = 1000;
        int totalPage = scoreRcptCodes.size() / pageSize;
        if(scoreRcptCodes.size() % pageSize > 0) {
            totalPage++;
        }
        List<ScoreGiveAwayLog> scoreGiveAwayLogs = new ArrayList<>();
        for (; page < totalPage; page++) {
            int startIndex = pageSize * page;
            int endIndex = pageSize * (page + 1);
            if (endIndex > scoreRcptCodes.size()) {
                endIndex = coreUserIds.size();
            }
            scoreGiveAwayLogs.addAll(scoreGiveAwayLogRepos.findScoreGiveAwayByIncomeCodes(scoreRcptCodes.subList(startIndex, endIndex)));
        }
        coreUserIds.addAll(scoreGiveAwayLogs.stream().map(ScoreGiveAwayLog::getGiveAwayScoreAccount).map(ScoreAccount::getUserId).collect(Collectors.toList()));
        coreUserIds = coreUserIds.stream().distinct().collect(Collectors.toList());
        Map<String, MemberDto> memberDTOMap = findMembersByCoreUserIds(coreUserIds);
        scoreGiveAwayLogs.forEach(scoreGiveAwayLog -> {
            MemberDto memberDTO = memberDTOMap.get(scoreGiveAwayLog.getGiveAwayScoreAccount().getUserId());
            memberDTOMap.put(scoreGiveAwayLog.getScoreIncomeRcptCode(), memberDTO);
        });
        log.info("查询获赠积分数据耗时 {} ms", System.currentTimeMillis() - methodStart);
        return memberDTOMap;
    }

    public Map<String, MemberDto> findScoreGetData(Map<String, ScoreGiveAwayLogDTO> codeLogMap, List<String> scoreRcptCodes) {
        long methodStart = System.currentTimeMillis();
        int page = 0;
        int pageSize = 1000;
        int totalPage = scoreRcptCodes.size() / pageSize;
        if(scoreRcptCodes.size() % pageSize > 0) {
            totalPage++;
        }
        List<ScoreGiveAwayLog> scoreGiveAwayLogs = new ArrayList<>();
        for (; page < totalPage; page++) {
            int startIndex = pageSize * page;
            int endIndex = pageSize * (page + 1);
            if (endIndex > scoreRcptCodes.size()) {
                endIndex = scoreRcptCodes.size();
            }
            scoreGiveAwayLogs.addAll(scoreGiveAwayLogRepos.findScoreGiveAwayByIncomeCodes(scoreRcptCodes.subList(startIndex, endIndex)));
        }

        MapperFactory mapperFactory = new DefaultMapperFactory.Builder().build();
        mapperFactory.classMap(ScoreGiveAwayLog.class, ScoreGiveAwayLogDTO.class).exclude("giveAwayScoreAccount").exclude("receiveScoreAccount").byDefault().register();

        // 匹配 scoreRcptCode 封装 map
        for (ScoreGiveAwayLog scoreGiveAwayLog : scoreGiveAwayLogs) {
            for (String scoreRcptCode : scoreRcptCodes) {
                if(scoreRcptCode.equals(scoreGiveAwayLog.getScoreIncomeRcptCode())) {
                    codeLogMap.put(scoreRcptCode, mapperFactory.getMapperFacade().map(scoreGiveAwayLog, ScoreGiveAwayLogDTO.class));
                    break;
                }
            }
        }

        List<String> coreUserIds = new ArrayList<>();
        coreUserIds.addAll(scoreGiveAwayLogs.stream().map(ScoreGiveAwayLog::getGiveAwayScoreAccount).map(ScoreAccount::getUserId).collect(Collectors.toList()));
        coreUserIds = coreUserIds.stream().distinct().collect(Collectors.toList());
        Map<String, MemberDto> memberDTOMap = findMembersByCoreUserIds(coreUserIds);
        scoreGiveAwayLogs.forEach(scoreGiveAwayLog -> {
            MemberDto memberDTO = memberDTOMap.get(scoreGiveAwayLog.getGiveAwayScoreAccount().getUserId());
            memberDTOMap.put(scoreGiveAwayLog.getScoreIncomeRcptCode(), memberDTO);
        });
        log.info("查询转赠积分数据耗时 {} ms", System.currentTimeMillis() - methodStart);
        return memberDTOMap;
    }
}
