package ctscore.web.utils;

import lombok.extern.slf4j.Slf4j;
import member.api.AddressCommonApi;
import member.api.dto.common.AddressCodeDTO;
import utils.GlobalContants;
import utils.Lang;
import utils.ip.IPtoLong;
import utils.web.Webs;

import javax.servlet.http.HttpServletRequest;

/**
 * 诚通商城 地址工具
 *
 * @author Drury
 * @date 2018/4/25
 */
@Slf4j
public class AddressUtil {

    /**
     * 诚通商城 默认省代码
     */
    private final static String DEFAULT_PROVICE_ID = "110000";

    /**
     * 诚通商城 默认市代码
     */
    private final static String DEFAULT_CITY_ID = "110100";

    /**
     * 诚通商城 默认区代码
     */
    private final static String DEFAULT_AREA_ID = "110101";

    /**
     * 诚通商城 默认镇代码
     */
    private final static String DEFAULT_TOWN_ID = "d8a15c70-9c8a-44b6-bc34-06b457ecfaaf";

    /**
     * 获得地址的省, 市, 区, 镇代码
     *
     * @param request HttpServletRequest
     * @param  addressCommonApi AddressCommonApi
     * @return AddressCodeDTO 地址省市区镇代码数据传输对象
     */
    public static AddressCodeDTO getAddressCode(HttpServletRequest request, AddressCommonApi addressCommonApi) {
        long methodStartTime = System.currentTimeMillis();
        log.info("### 开始获取地址省市区镇代码 ... ###");

        log.info("从用户 Session 获取地址数据...");
        String proviceCode = (String) request.getSession().getAttribute(GlobalContants.AREA_LEVEL_PROVINCE_ID);
        String cityCode = (String) request.getSession().getAttribute(GlobalContants.AREA_LEVEL_CITY_ID);
        String areaCode = (String) request.getSession().getAttribute(GlobalContants.AREA_LEVEL_AREA_ID);
        String townCode = (String) request.getSession().getAttribute(GlobalContants.AREA_LEVEL_TOWN_ID);
        log.info("用户 Session 地址代码, 省 {} 市 {} 区 {} 镇 {} ", proviceCode, cityCode, areaCode, townCode);

        AddressCodeDTO addressCodeDTO = null;
        if(Lang.isEmpty(proviceCode) || Lang.isEmpty(areaCode)) {
            log.info("用户 Session 中没有可用地址数据, 尝试根据用户的 IP 获取三级地址...");
            log.info("尝试获取用户 IP ...");
            String ip = Webs.getIp(request);
            if(!Lang.isEmpty(ip)) {
                log.info("用户 IP 为 {}", ip);
                try {
                    log.info("转换 IP 数据...");
                    Long ipLong = IPtoLong.ipToLong(ip);
                    log.info("转换 IP 数据为 {}", ipLong);
                    addressCodeDTO = addressCommonApi.findThreeLevelAddreesCodeByIP(ipLong);
                } catch (Exception e) {
                    log.error("发生异常", e);
                }
                if(Lang.isEmpty(addressCodeDTO)) {
                    if(Lang.isEmpty(proviceCode)) {
                        log.warn("根据用户 IP 获取三级地址失败, 使用默认定位省份代码 {}", DEFAULT_PROVICE_ID);
                    } else {
                        log.warn("根据用户 IP 获取三级地址失败, 使用用户 Session 中省份代码 {}", proviceCode);
                    }
                } else {
                    log.info("根据用户 IP 获取三级地址成功...");
                    updateSessionAddressCode(request, addressCodeDTO);
                }
            } else {
                if(Lang.isEmpty(proviceCode)) {
                    log.warn("获取用户 IP 失败, 使用默认定位省份代码 {}", DEFAULT_PROVICE_ID);
                } else {
                    log.warn("获取用户 IP 失败, 使用用户 Session 中省份代码 {}", proviceCode);
                }
            }
        } else {
            log.info("使用用户 Session 地址代码, 省 {} 市 {} 区 {} 镇 {} ", proviceCode, cityCode, areaCode, townCode );
            addressCodeDTO = new AddressCodeDTO(proviceCode, cityCode, areaCode, townCode);
        }

        if(Lang.isEmpty(addressCodeDTO)) {
            if(Lang.isEmpty(proviceCode)) {
                log.warn("使用默认地址代码, 省 {} 市 {} 区 {} 镇 {} ", DEFAULT_PROVICE_ID, DEFAULT_CITY_ID, DEFAULT_AREA_ID, DEFAULT_TOWN_ID);
                addressCodeDTO = new AddressCodeDTO(DEFAULT_PROVICE_ID, DEFAULT_CITY_ID, DEFAULT_AREA_ID, DEFAULT_TOWN_ID);
                updateSessionAddressCode(request, addressCodeDTO);
            } else {
                log.info("根据默认的省份代码依次获取下级市, 区代码...");
                try {
                    addressCodeDTO = addressCommonApi.findThreeLevelAddreesCodeByProviceCode(proviceCode);
                } catch (Exception e) {
                    log.error("发生异常", e);
                }
                if(Lang.isEmpty(addressCodeDTO)) {
                    log.warn("获取代码失败");
                    log.warn("使用默认地址代码, 省 {} 市 {} 区 {} 镇 {} ", DEFAULT_PROVICE_ID, DEFAULT_CITY_ID, DEFAULT_AREA_ID, null);
                    addressCodeDTO = new AddressCodeDTO(DEFAULT_PROVICE_ID, DEFAULT_CITY_ID, DEFAULT_AREA_ID, null);
                }
                updateSessionAddressCode(request, addressCodeDTO);
            }

        }
        log.info("### 获取地址省市区镇代码总耗时 {} ms ###", System.currentTimeMillis() - methodStartTime);
        return addressCodeDTO;
    }

    /**
     * 更新用户 Session 中地址的省, 市, 区, 镇代码
     *
     * @param request HttpServletRequest
     * @param addressCodeDTO AddressCodeDTO 地址代码数据传输封装对象
     */
    public static void updateSessionAddressCode(HttpServletRequest request, AddressCodeDTO addressCodeDTO) {
        log.info("更新用户 Session 中的地址代码数据, 省 {}, 市 {}, 区 {}, 镇 {}", addressCodeDTO.getProviceCode(), addressCodeDTO.getCityCode(), addressCodeDTO.getAreaCode(), addressCodeDTO.getTownCode());
        request.getSession().setAttribute(GlobalContants.AREA_LEVEL_PROVINCE_ID, addressCodeDTO.getProviceCode());
        request.getSession().setAttribute(GlobalContants.AREA_LEVEL_CITY_ID, addressCodeDTO.getCityCode());
        request.getSession().setAttribute(GlobalContants.AREA_LEVEL_AREA_ID, addressCodeDTO.getAreaCode());
        request.getSession().setAttribute(GlobalContants.AREA_LEVEL_TOWN_ID, addressCodeDTO.getTownCode());
    }

}
