package order.api;

import order.dto.OrderQueryDto;
import order.vo.query.OrdersQueryVo;
import order.exceptions.IllegalAreaCodeException;
import order.exceptions.OrderCreateException;
import order.exceptions.UnAuditableOrderException;
import order.exceptions.temporder.TempOrderCreateException;
import order.vo.*;
import order.vo.response.OrderAuditingResult;
import order.vo.response.OrderPrice;
import order.vo.response.OrderStock;
import sinomall.global.common.response.BaseResponse;
import utils.sql.PageVo;

import java.util.List;

/**
 * 订单操作Api接口，主要定义订单相关操作的业务接口
 * <p><b>Notic:</b>注意！不要定义save、saveAll之类的方法！
 * 此接口定义的方法必须面向准确业务！</p>
 * @author Liang Wenxu
 * @since 2018/7/11
 */
public interface OrderApi {


    /**
     * 根据订单号查询订单详细信息
     * @param  orderNo
     * @return
     */

    public BaseResponse queryProviderOrdersDetails(String orderNo) ;

    /**
     * 9.查询订单列表(分页)
     * @param productQueryDto
     * @return
     */
    BaseResponse<PageVo<OrdersQueryVo>> queryProviderOrdersPage(OrderQueryDto productQueryDto);

    /**
     * 创建临时订单
     * <p>此接口传入所需购买的所有商品（批量下单与普通下单统一入口）</p>
     * <p>目前订单拆分的基本原则是：<br/>
     * 1. 按照店铺拆分<br/>
     * 2. 虚拟商品、实物商品拆分<br/>
     * 3. 每个拆分的子单都将分配临时订单号，理论上每个子单都最终形成一个订单
     * </p>
     * @param tempOrderCreateRequest ，只传入客户勾选后的待购商品
     * @return 临时订单实例，如是批量下单、提交的商品中包含不同店铺等需生成多个订单的场景则临时订单会有父子订单，注意与商城展示用的临时订单Vo区分噢
     * <p><b>订单拆分样例：</b><br />
     *  例如，一个订单中包含如下商品：<br />
     * 1. A1A -- 虚拟商品，A店铺<br />
     * 2. A2B -- 虚拟商品，B店铺<br />
     * 3. B1A -- 实物商品，A店铺<br />
     * 4. B2B -- 实物商品，B店铺<br />
     * 则其拆分后的订单示意如下：<br />
     * <pre>
     * TempOrder{<br/>
     *     orderItems:[<br/>
     *         A1A,A2B,B1A,B2B<br/>
     *     ],<br/>
     *     subOrder:[<br/>
     *         {<br/>
     *             storeCode:A,<br/>
     *             orderItems:[<br/>
     *                 A1A,B1A<br/>
     *             ],<br/>
     *             subOrders:[<br/>
     *                 {<br/>
     *                     storeCode:A,<br/>
     *                     orderItems:[<br/>
     *                         A1A<br/>
     *                     ]<br/>
     *                 }，<br/>
     *                 {<br/>
     *                     storeCode:A,<br/>
     *                     orderItems:[<br/>
     *                         B1A<br/>
     *                     ]<br/>
     *                 }<br/>
     *             ]<br/>
     *         },<br/>
     *         {<br/>
     *             storeCode:B,<br/>
     *             orderItems:[<br/>
     *                 A2B,B2B<br/>
     *             ],<br/>
     *             subOrders:[<br/>
     *                 {
     *                     storeCode:B,<br/>
     *                     orderItems:[<br/>
     *                         A2B<br/>
     *                     ]<br/>
     *                 }，<br/>
     *                 {<br/>
     *                     storeCode:B,<br/>
     *                     orderItems:[<br/>
     *                         B2B<br/>
     *                     ]<br/>
     *                 }<br/>
     *             ]<br/>
     *         }<br/>
     *     ]<br/>
     * }<br/>
     * </pre>
     * </p>
     */
    TempOrderVo newTempOrder(TempOrderCreateRequest tempOrderCreateRequest) throws TempOrderCreateException;

    /**
     * 临时订单库存校验
     * @param tempOrderVo 临时订单对象
     */
    OrderStock tempOrderStock(TempOrderVo tempOrderVo) throws IllegalAreaCodeException;

    /**
     * 临时定案价格重算
     * @param tempOrderVo
     * @return
     */
    OrderPrice tempOrderPrice(TempOrderVo tempOrderVo) throws IllegalAreaCodeException;

    /**
     * 提交订单，将临时订单提交为真实订单并预占商品库存（如需要）
     * <p>在结算确认页面，用户确认操作后调用，将临时订单确认为真实订单</p>
     * @param tempOrder 临时订单
     * @return 返回OrderVo中将包括订单的简略信息，如需详情请调用OrderQueryApi中相关API进行查询
     */
    OrderVo submit(TempOrderVo tempOrder) throws OrderCreateException;

    /**
     * 刷新订单信息，例如
     * @param orderKeys
     * @return
     */
    List<OrderVo> refresh(List<OrderKey> orderKeys);

    /**
     * 订单审核（单个）
     * @param orderKey 需操作的订单索引
     * @param passed true|false（审核通过|审核不通过）
     * @param cancelSupplierOrder
     * @throws UnAuditableOrderException 当订单不可审核时抛出该异常
     */
    OrderAuditingResult audit(OrderKey orderKey, Boolean passed, Boolean cancelSupplierOrder) throws UnAuditableOrderException;

    /**
     * 订单审核（支持批量）
     * @param orderKeys 需操作的订单索引
     * @param passed true|false（审核通过|审核不通过）
     * @param cancelSupplierOrder 审核不通过时是否取消供应商订单 true|false（取消|不取消）
     * @return 返回各订单的处理结果
     */
    List<OrderAuditingResult> audit(List<OrderKey> orderKeys, Boolean passed, Boolean cancelSupplierOrder);

    /**
     * 订单支付确认，确认订单已收款并进入发货流程
     * @param orderKey 其中orderNo、orderId可任选其一
     * @return
     */
    OrderPayConfirmResp payComfirmed(OrderKey orderKey);

    /**
     * 取消指定的订单
     * @param orderKey 其中orderNo、orderId可任选其一
     * @return
     */
    CancelOrderResp cancel(OrderKey orderKey, order.Contants.CancelOrderType cancelType);

    /**
     * 取消指定的一批订单
     * @param orderKey 其中orderNo、orderId可任选其一
     * @return
     */
    CancelOrderResp cancel(List<OrderKey> orderKey, order.Contants.CancelOrderType cancelType);


    /**
     * 订单收货
     * @param orderKey 其中orderNo、orderId可任选其一
     * @param reciptType 订单收货类型，见{@link order.Contants}中ORDER_RECEIPT_TYPE_开头常量
     * @return
     */
    void receipt(OrderKey orderKey, String reciptType);

    /**
     * 订单状态改变
     * @param orderStatus 订单状态
     */
    void statusChange(OrderStatus orderStatus);

    /**
     * 订单支付预处理完毕
     * （客户已开始支付，但未支付成功时调用）
     * @param orderKey 订单key，orderNo或orderId，优先id
     * @param cusPayInfo 客户支付信息，如需在客户支付环节带入新的信息从此参数带入
     */
    void prePayDone(OrderKey orderKey, CusPayInfo cusPayInfo);
}
