package order.model.specification;

import com.alibaba.fastjson.annotation.JSONField;
import order.model.po.OrderMain;
import org.springframework.data.jpa.domain.Specification;
import utils.Lang;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author Drury
 * @date 2018/3/24
 */
public class OrderMainSpecification<T> implements Specification<T> {

    private OrderMain orderMain;

    /**
     * 下单人ID
     */
    private List<String> memberIds;

    /**
     * 订单状态
     */
    private List<String> status;

    /**
     * 订单创建时间开始
     */
    @JSONField(format="yyyy-MM-dd HH:mm:ss")     private Date createTiemStart;

    /**
     * 订单创建时间结束
     */
    @JSONField(format="yyyy-MM-dd HH:mm:ss")     private Date createTiemEnd;

    public OrderMainSpecification(){}

    public OrderMainSpecification(OrderMain orderMain) {
        this.orderMain = orderMain;
    }

    /**
     * Creates a WHERE clause for a query of the referenced entity in form of a {@link Predicate} for the given
     * {@link Root} and {@link CriteriaQuery}.
     *
     * @param root Root
     * @param query CriteriaQuery
     * @param cb CriteriaBuilder
     * @return a {@link Predicate}, must not be {@literal null}.
     */
    @Override
    public Predicate toPredicate(Root root, CriteriaQuery query, CriteriaBuilder cb) {
        List<Predicate> predicates = new ArrayList<>();

        // 空判断
        if(!Lang.isEmpty(orderMain)) {

            // 订单号
            if(!Lang.isEmpty(orderMain.getOrderNo())) {
                predicates.add(cb.equal(root.get("orderNo"), orderMain.getOrderNo()));
            }

            // 下单人ID
            if(!Lang.isEmpty(memberIds)) {
                CriteriaBuilder.In<String> in = cb.in(root.get("memberId"));
                memberIds.forEach(memberId -> in.value(memberId));
                predicates.add(in);
            }

            // 机构ID
            if(!Lang.isEmpty(orderMain.getOrganizationId())) {
                predicates.add(cb.equal(root.get("organizationId"), orderMain.getOrganizationId()));
            }


            // 订单状态
            if(!Lang.isEmpty(status)) {
                CriteriaBuilder.In<String> in = cb.in(root.get("status"));
                status.forEach(s -> in.value(s));
                predicates.add(in);
            }

            // 订单创建时间开始
            if(!Lang.isEmpty(createTiemStart)) {
                predicates.add(cb.greaterThanOrEqualTo(root.get("dateCreated"), createTiemStart));
            }

            // 订单创建时间结束
            if(!Lang.isEmpty(createTiemEnd)) {
                predicates.add(cb.lessThanOrEqualTo(root.get("dateCreated"), createTiemEnd));
            }


            // 订单号
            if(!Lang.isEmpty(orderMain.getParentOrderId())) {
                predicates.add(cb.equal(root.get("parentOrderId"), orderMain.getParentOrderId()));
            }

        }

        // 默认使用false
        predicates.add(cb.equal(root.get("isDelete"), false));

        return query.where(predicates.toArray(new Predicate[predicates.size()])).getRestriction();
    }

    public OrderMain getOrderMain() {
        return orderMain;
    }

    public void setOrderMain(OrderMain orderMain) {
        this.orderMain = orderMain;
    }

    public List<String> getMemberIds() {
        return memberIds;
    }

    public void setMemberIds(List<String> memberIds) {
        this.memberIds = memberIds;
    }

    public List<String> getStatus() {
        return status;
    }

    public void setStatus(List<String> status) {
        this.status = status;
    }

    public Date getCreateTiemStart() {
        return createTiemStart;
    }

    public void setCreateTiemStart(Date createTiemStart) {
        this.createTiemStart = createTiemStart;
    }

    public Date getCreateTiemEnd() {
        return createTiemEnd;
    }

    public void setCreateTiemEnd(Date createTiemEnd) {
        this.createTiemEnd = createTiemEnd;
    }
}
