package order.service.apiimpl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.weibo.api.motan.config.springsupport.annotation.MotanService;
import order.Contants;
import order.api.OrderApi;
import order.dto.OrderQueryDto;
import order.exceptions.IllegalAreaCodeException;
import order.exceptions.OrderCreateException;
import order.exceptions.UnAuditableOrderException;
import order.exceptions.temporder.TempOrderCreateException;
import order.service.service.OrderService;
import order.service.service.query.OrderMainQueryService;
import order.vo.*;
import order.vo.query.OrdersQueryVo;
import order.vo.response.OrderAuditingResult;
import order.vo.response.OrderPrice;
import order.vo.response.OrderStock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import sinomall.global.common.response.BaseResponse;
import utils.sql.PageVo;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Liang Wenxu
 * @since 2018/7/18
 */
@MotanService(basicService = "motanServerBasicConfig")
@Transactional(propagation = Propagation.NOT_SUPPORTED)
public class OrderApiImpl implements OrderApi {

    @Autowired
    private OrderService orderService;
    @Autowired
    private OrderMainQueryService orderMainQueryService;
    private static final Logger logger = LoggerFactory.getLogger(OrderApiImpl.class);

    /**
     * 根据订单号查询订单详细信息
     * @param  orderNo
     * @return
     */
    @Override
    public BaseResponse queryProviderOrdersDetails(String orderNo) {
        logger.info("第三方订单对接接口，查询订单详细信息接口输入参数：{}", JSON.toJSONString(orderNo, SerializerFeature.WriteMapNullValue));
        BaseResponse baseResponse = orderMainQueryService.queryProviderOrdersDetails(orderNo);
        logger.info("第三方订单对接接口，查询订单详细信息接口输出参数：{}",JSON.toJSONString(baseResponse,SerializerFeature.WriteMapNullValue));
        return baseResponse;
    }

    @Override
    public BaseResponse<PageVo<OrdersQueryVo>> queryProviderOrdersPage(OrderQueryDto orderQueryDto) {
        logger.info("第三方订单对接接口，查询订单列表接口输入参数：{}", JSON.toJSONString(orderQueryDto, SerializerFeature.WriteMapNullValue));
        BaseResponse baseResponse = orderMainQueryService.queryProviderOrdersPage(orderQueryDto);
        logger.info("第三方订单对接接口，查询订单列表接口输出参数：{}",JSON.toJSONString(baseResponse,SerializerFeature.WriteMapNullValue));
        return baseResponse;
    }

    /**
     * 创建临时订单<br />
     * <p>此接口传入所需购买的所有商品（批量下单与普通下单统一入口）</p>
     * <p>拆分的基本原则是：</p>
     *
     * <p><b>1. 先按照店铺拆分<br />
     * 2. 再按虚拟商品、实物商品拆分<br />
     * 3. 每个拆分的子单都将分配临时订单号，理论上每个子单都最终形成一个订单</b><br />
     * 例如，一个订单中包含如下商品：<br />
     *
     * A1A -- 虚拟商品，A店铺<br />
     * A2B -- 虚拟商品，B店铺<br />
     * B1A -- 实物商品，A店铺<br />
     * B2B -- 实物商品，B店铺 <br />
     * 则其拆分后的订单示意如下：
     * <pre>TempOrder{
     *     orderItems:[
     *         A1A,A2B,B1A,B2B
     *     ],
     *     subOrder:[
     *         {
     *             storeCode:A,
     *             orderItems:[
     *                 A1A,B1A
     *             ],
     *             subOrders:[
     *                 {
     *                     storeCode:A,
     *                     orderItems:[
     *                         A1A
     *                     ]
     *                 }，
     *                 {
     *                     storeCode:A,
     *                     orderItems:[
     *                         B1A
     *                     ]
     *                 }
     *             ]
     *         },
     *         {
     *             storeCode:B,
     *             orderItems:[
     *                 A2B,B2B
     *             ],
     *             subOrders:[
     *                 {
     *                     storeCode:B,
     *                     orderItems:[
     *                         A2B
     *                     ]
     *                 }，
     *                 {
     *                     storeCode:B,
     *                     orderItems:[
     *                         B2B
     *                     ]
     *                 }
     *             ]
     *         }
     *     ]
     * }</pre></p>
     * @param tempOrderCreateRequest ，只传入客户勾选后的待购商品
     * @return 临时订单实例，如是批量下单、提交的商品中包含不同店铺等需生成多个订单的场景则临时订单会有父子订单，注意与商城展示用的临时订单Vo区分噢
     */
    @Override
    public TempOrderVo newTempOrder(TempOrderCreateRequest tempOrderCreateRequest) throws TempOrderCreateException {
        return orderService.newTempOrder(tempOrderCreateRequest);
    }

    /**
     * 临时订单库存校验
     *
     * @param tempOrderVo 临时订单对象
     */
    @Override
    public OrderStock tempOrderStock(TempOrderVo tempOrderVo) throws IllegalAreaCodeException {
        return orderService.tempOrderStock(tempOrderVo);
    }

    /**
     * 临时定案价格重算
     *
     * @param tempOrderVo
     * @return
     */
    @Override
    public OrderPrice tempOrderPrice(TempOrderVo tempOrderVo) throws IllegalAreaCodeException {
        return orderService.tempOrderPrice(tempOrderVo);
    }

    /**
     * 提交订单，将临时订单提交为真实订单并预占商品库存（如需要）
     * <p>在结算确认页面，用户确认操作后调用，将临时订单确认为真实订单</p>
     *
     * @param tempOrder 临时订单
     * @return 返回OrderVo中将包括订单的简略信息，如需详情请调用OrderQueryApi中相关API进行查询
     */
    @Override
    public OrderVo submit(TempOrderVo tempOrder) throws OrderCreateException {
        return orderService.submit(tempOrder);
    }

    /**
     * 刷新订单信息，例如
     *
     * @param orderKeys
     * @return
     */
    @Override
    public List<OrderVo> refresh(List<OrderKey> orderKeys) {
        return orderService.refresh(orderKeys);
    }

    /**
     * 订单审核（单个）
     *
     * @param orderKey 需操作的订单索引
     * @param passed   true|false（审核通过|审核不通过）
     * @param cancelSupplierOrder
     * @throws UnAuditableOrderException 当订单不可审核时抛出该异常
     */
    @Override
    public OrderAuditingResult audit(OrderKey orderKey, Boolean passed, Boolean cancelSupplierOrder) throws UnAuditableOrderException {
        return orderService.audit(orderKey, passed, cancelSupplierOrder);
    }

    /**
     * 订单审核（支持批量）
     *
     * @param orderKeys 需操作的订单索引
     * @param passed    true|false（审核通过|审核不通过）
     * @param cancelSupplierOrder
     * @return 返回各订单的处理结果
     */
    @Override
    public List<OrderAuditingResult> audit(List<OrderKey> orderKeys, Boolean passed, Boolean cancelSupplierOrder) {
        return orderService.audit(orderKeys, passed, cancelSupplierOrder);
    }

    /**
     * 订单支付确认，确认订单已收款并进入发货流程
     *
     * @param orderKey 其中orderNo、orderId可任选其一
     * @return
     */
    @Override
    public OrderPayConfirmResp payComfirmed(OrderKey orderKey) {
        return orderService.payComfirmed(orderKey);
    }

    /**
     * 取消指定的订单
     *
     * @param orderKey 其中orderNo、orderId可任选其一
     * @return
     */
    @Override
    public CancelOrderResp cancel(OrderKey orderKey, Contants.CancelOrderType cancelType) {
        List<OrderKey> orderKeys = new ArrayList<>();
        orderKeys.add(orderKey);
        return this.cancel(orderKeys, cancelType);
    }

    /**
     * 取消指定的一批订单
     *
     * @param orderKey   其中orderNo、orderId可任选其一
     * @param cancelType 取消类型
     * @return
     */
    @Override
    public CancelOrderResp cancel(List<OrderKey> orderKey, Contants.CancelOrderType cancelType) {
        return orderService.cancel(orderKey, cancelType);
    }

    /**
     * 订单收货
     *
     * @param orderKey 其中orderNo、orderId可任选其一
     * @param receiptType 订单收货类型，见{@link Contants}中ORDER_RECEIPT_TYPE_开头常量
     * @return
     */
    @Override
    public void receipt(OrderKey orderKey, String receiptType) {
        orderService.receipt(orderKey, receiptType);
    }

    /**
     * 订单状态改变
     *
     * @param orderStatus 订单状态
     */
    @Override
    public void statusChange(OrderStatus orderStatus) {
        orderService.statusChange(orderStatus);
    }

    /**
     * 订单支付预处理完毕
     * （客户已开始支付，但未支付成功时调用）
     *
     * @param orderKey   订单key，orderNo或orderId，优先id
     * @param cusPayInfo 客户支付信息，如需在客户支付环节带入新的信息从此参数带入
     */
    @Override
    public void prePayDone(OrderKey orderKey, CusPayInfo cusPayInfo) {
        orderService.prePayDone(orderKey, cusPayInfo);
    }
}
