package order.service.events.listener;

import order.model.po.OrderMain;
import fiance.messages.bills.BillsQueueOrderRequest;
import fiance.messages.bills.BillsQueueRequest;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFacade;
import order.service.events.OrderEvent;
import order.service.events.OrderPassedEvent;
import order.service.jms.JmsHelper;
import order.service.service.OrderMainService;
import order.service.service.OrderService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.event.SmartApplicationListener;
import org.springframework.stereotype.Component;
import store.api.dto.modeldto.core.StoreDto;
import utils.Lang;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 协议采购订单审核通过事件监听
 *
 * @author Liang Wenxu
 * @since 2018-01-31
 */
@Component
@Slf4j
public class MallDefaultOrderPassedEventListener implements SmartApplicationListener {
    private static final String orgCode = "protocol";

    @Value("${activemq.queue.orderEvents:}")
    private String orderEventQueue;

    @Value("${finance.bills.message.queue.billscreate:}")
    private String billsCreateQueue;
    @Autowired
    MapperFacade mapperFacade;

    @Override
    public boolean supportsEventType(Class<? extends ApplicationEvent> eventType) {
        return eventType == OrderPassedEvent.class;
    }

    @Override
    public boolean supportsSourceType(Class<?> sourceType) {
        return true;
    }

    @Override
    public void onApplicationEvent(ApplicationEvent event) {
        OrderPassedEvent orderEvent = (OrderPassedEvent) event;
        OrderMain orderMain = (OrderMain) orderEvent.getSource();
        log.info("订单审核通过事件orderMain：{}", orderMain);
        if (!Lang.isEmpty(orderMain)) {
            /**
             * 增加审核通过时间
             */
            orderMainService.updateOrderAuditEndTime(orderMain.getId());
            if (!StoreDto.STORE_TYPE_PROTOCOL.equals(orderMain.getStoreType())) {
                log.info("orderNo: {}, orgCode: {} ,非协议采购订单审核通过事件", orderMain.getOrderNo(), ((OrderEvent) event).getOrgCode());
                SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
                // 提交账单申请
                BillsQueueRequest billsQueueRequest = new BillsQueueRequest();
                // 直购订单审核通过时间即为账单生成时间
                billsQueueRequest.setBillsTime(new Date());
                // 账单生成请求ID
                String requestId = Lang.isEmpty(orderMain.getBillsRequestId()) ? BillsQueueRequest.SOURCE_CODE_DIRECT + "_" + orderMain.getOrderNo() + "_" + sdf.format(new Date()) : orderMain.getBillsRequestId();
                billsQueueRequest.setRequestId(requestId);
                billsQueueRequest.setInvoiceFlag(true);
                billsQueueRequest.setVoucherFlag(true);
                billsQueueRequest.setSouceCode(BillsQueueRequest.SOURCE_CODE_DIRECT);
                billsQueueRequest.setPrice(orderMain.getSumPrice());
                List<BillsQueueOrderRequest> orders = new ArrayList<>();
                orders.add(new BillsQueueOrderRequest(orderMain.getId(), orderMain.getOrderNo()));
                billsQueueRequest.setBillsOrder(orders);

                jmsHelper.sendQueuedObjMessage(billsQueueRequest, billsCreateQueue);
                orderMain.setBillsCreated(true);
                orderMain.setBillsRequestId(requestId);
                orderMain.setAuditEndTime(new Date());
                orderMainService.saveAndFlush(orderMain);
                log.info("orderNo: {}, orgCode: {} ,非协议采购订单审核通过事件-已发送账单申请MQ消息", orderMain.getOrderNo(), ((OrderEvent) event).getOrgCode());
            }
        }
    }

    @Override
    public int getOrder() {
        return 0;
    }

    @Autowired
    OrderMainService orderMainService;

    @Autowired
    JmsHelper jmsHelper;

    @Autowired
    OrderService orderService;
}
