package order.service.factory.helper;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

/**
 * @author Liang Wenxu
 * @since 2018/7/11
 */
@Slf4j
@Component
public class OrderTemplateSpringBeanHelper implements ApplicationContextAware {

    private ApplicationContext applicationContext;

    /**
     * 获取Bean，如果获取的Bean不存在，会尝试将该Bean初始化并注册到Spring容器，处理失败时返回null
     * @param beanName 注意全局唯一
     * @param className springBean的类名，全名
     * @param targetClass 返回实例的类型，请使用接口或最高基类的类型，如非可转换类型，会抛出异常
     * @param <T> 返回值会被转换为targetClass指定类型的实例，转换失败抛出ClassCastException
     * @return 返回对应的Bean实例，失败时为null
     */
    public <T> T getOrRegisterBean(String beanName, String className, Class<T> targetClass ) throws ClassCastException {
        Object obj = this.getOrRegisterBean(beanName, className);
        return targetClass.cast(obj);
    }

    /**
     * 获取Bean，如果获取的Bean不存在，会尝试将该Bean初始化并注册到Spring容器，处理失败时返回null
     * <p>这个做法有点装逼，主要用途是可以在运行时将处理模板的bean加入容器，可扩展性比使用注解会灵活一些，而且也能实现动态加载Class生成bean（非常少用）</p>
     * @param beanName 注意全局唯一
     * @param className springBean的类名，全名
     * @return 返回对应的Bean实例，失败时为null
     */
    public Object getOrRegisterBean(String beanName, String className) {
        Object beanInstance;
        try {
            // 先看看Spring容器里有没有
            beanInstance = applicationContext.getBean(beanName);
        } catch (NoSuchBeanDefinitionException nsbde) {
            beanInstance = null;
        }
        if(beanInstance == null) {
            // 实例化template类，并加入Spring容器
            DefaultListableBeanFactory beanFactory = (DefaultListableBeanFactory)applicationContext.getParentBeanFactory();
            if(beanFactory == null) {
                beanFactory = (DefaultListableBeanFactory)applicationContext.getAutowireCapableBeanFactory();
            }
            try {
                Class clz = Class.forName(className);
                BeanDefinitionBuilder beanDefinitionBuilder = BeanDefinitionBuilder
                        .genericBeanDefinition(clz)
                        .setScope(BeanDefinition.SCOPE_SINGLETON)
                        .setLazyInit(false)
                        .setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_NAME);
                BeanDefinition beanDefinition = beanDefinitionBuilder.getBeanDefinition();
                beanDefinition.setAutowireCandidate(true);
//                beanFactory.applyBeanPostProcessorsAfterInitialization(beanInstance, beanName);
                beanFactory.registerBeanDefinition(beanName, beanDefinitionBuilder.getBeanDefinition());
                return applicationContext.getBean(beanName);
            } catch (ClassNotFoundException e) {
                log.error("加载OrderTemplate类失败!", e);
            }
        }

        return beanInstance;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
