package order.service.operator.impl;

import lombok.extern.slf4j.Slf4j;
import member.api.dto.shop.MemberAddressDto;
import order.service.operator.OrderFreightOperator;
import order.service.stubs.MemberAddressApiStub;
import order.service.stubs.OutsideProductApiStub;
import order.vo.TempOrderItemVo;
import order.vo.TempOrderVo;
import order.vo.TempSubOrderVo;
import order.vo.response.OrderFreight;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import outsideapi.vo.FashionNumsVo;
import outsideapi.vo.HandlerRespVo;
import outsideapi.vo.ShippingFeeRequetVo;
import utils.Lang;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.*;

/**
 * @author Liang Wenxu
 * @since 2018/7/27
 */
@Slf4j
@Component("orderFreightOperator")
public class DefaultOrderFreightOperator implements OrderFreightOperator {
    /**
     * REQUEST_MAP的KEY{@value}: 店铺ID
     */
    public static final String REQUEST_MAP_KEY_STORE_ID = "storeId";
    /**
     * REQUEST_MAP的KEY{@value}: 订单号
     */
    public static final String REQUEST_MAP_KEY_ORDER_NO = "orderNo";
    /**
     * REQUEST_MAP的KEY{@value}: 订单号
     */
    public static final String REQUEST_MAP_KEY_REQUEST_DATA = "request";

    @Autowired
    private MemberAddressApiStub memberAddressApi;

    @Autowired
    OutsideProductApiStub outsideProductApi;

    /**
     * 查询运费
     * 目前的临时订单拆分结构为 主单 -> 店铺订单 -> 细分订单（目前分实物商品订单和虚拟商品订单），查询运费只会到细分订单一级
     *
     * @param tempOrderVo 临时订单实体，请求中的areaCode为最末级地区代码
     * @return 返回的临时订单对象包括了父子结构，父子结构将与tempOrder中的保持一致
     */
    @Override
    public OrderFreight get(TempOrderVo tempOrderVo) {

        Map<String, Object> request;
        List<Map<String, Object>> requestList = new ArrayList<>();

        if (tempOrderVo.getSubOrders() != null) {
            // API调用线程池
            ExecutorService pool = new ThreadPoolExecutor(0, 3,
                    10L, TimeUnit.MILLISECONDS,
                    new LinkedBlockingQueue<Runnable>());
            List<Future<Map<String, Object>>> outsideFreightFutures = new ArrayList<>();


            // 如果包含子订单的，需处理所有子订单
            TempSubOrderVo curChild = null;
            Vector<TempSubOrderVo> tempSubOrders = new Vector<>();
            for (TempSubOrderVo so : tempOrderVo.getSubOrders()) {
                //TODO: 目前临时订单的拆分，SubOrder必然是店铺：总单->店铺订单->细分订单，如果有更深的嵌套（这场景可能不存在）则需无级处理代码逻辑更复杂
                request = new HashMap<>();
                if (so.getSubOrders() != null && so.getSubOrders().size() > 0) {
                    for (TempSubOrderVo sso : so.getSubOrders()) {
                        request.put(REQUEST_MAP_KEY_STORE_ID, so.getStoreId());
                        request.put(REQUEST_MAP_KEY_ORDER_NO, sso.getTmpOrderNo());
                        request.put(REQUEST_MAP_KEY_REQUEST_DATA, createShippingFeeRequest(sso, tempOrderVo.getMemberAddrId(), tempOrderVo.getAreaCodeList()));
                        requestList.add(request);
                    }
                } else {
                    request.put(REQUEST_MAP_KEY_STORE_ID, so.getStoreId());
                    request.put(REQUEST_MAP_KEY_ORDER_NO, so.getTmpOrderNo());
                    request.put(REQUEST_MAP_KEY_REQUEST_DATA, createShippingFeeRequest(so, tempOrderVo.getMemberAddrId(), tempOrderVo.getAreaCodeList()));
                    requestList.add(request);
                }
            }
            for (Map m : requestList) {
                // 异步方式调用API提高效率
                outsideFreightFutures.add(pool.submit(() -> {
                    long queryFashionPriceTime = System.currentTimeMillis();
                    HandlerRespVo<Map<String, Object>> outsideApiRes = outsideProductApi.queryShippingFee((String) m.get(REQUEST_MAP_KEY_STORE_ID), (ShippingFeeRequetVo) m.get(REQUEST_MAP_KEY_REQUEST_DATA));
                    log.info("第三方订单运费查询（单个线程）耗时 {} ms", System.currentTimeMillis() - queryFashionPriceTime);
                    if (HandlerRespVo.RESPONSE_STATUS_SUCCESS.equals(outsideApiRes.getStatus())) {
                        Map<String, Object> respMap = new HashMap<>();
                        respMap.put(REQUEST_MAP_KEY_ORDER_NO, m.get(REQUEST_MAP_KEY_ORDER_NO));
                        respMap.put(REQUEST_MAP_KEY_REQUEST_DATA, outsideApiRes.getData());
                        return respMap;
                    }
                    return null;
                }));
            }
            pool.shutdown();

            // 初始化返回结果
            OrderFreight orderFreight = new OrderFreight();
            orderFreight.setOrderNo(tempOrderVo.getTmpOrderNo());
            orderFreight.setBaseFreight(new BigDecimal(-1));
            orderFreight.setSubOrderFreights(new ArrayList<>());
            for (TempSubOrderVo so : tempOrderVo.getSubOrders()) {
                OrderFreight sof = new OrderFreight();
                sof.setOrderNo(so.getTmpOrderNo());
                sof.setBaseFreight(new BigDecimal(-1));
                if (so.getSubOrders() != null && so.getSubOrders().size() > 0) {
                    sof.setSubOrderFreights(new ArrayList<>());
                    for (TempSubOrderVo sso : so.getSubOrders()) {
                        OrderFreight ssof = new OrderFreight();
                        ssof.setOrderNo(sso.getTmpOrderNo());
                        ssof.setBaseFreight(new BigDecimal(-1));
                        sof.getSubOrderFreights().add(ssof);
                    }
                }
                orderFreight.getSubOrderFreights().add(sof);
            }

            // 获取并回填运费
            for (Future<Map<String, Object>> f : outsideFreightFutures) {
                try {
                    Map<String, Object> fm = f.get();
                    String orderNo = (String) fm.get(REQUEST_MAP_KEY_ORDER_NO);
                    Map rsMap = (Map) fm.get(REQUEST_MAP_KEY_REQUEST_DATA);
                    BigDecimal freight = (BigDecimal) rsMap.get("freight");
                    // 回填子单运费
                    orderFreight.getSubOrderFreights().stream().filter(o ->
                            (o.getSubOrderFreights() == null || o.getSubOrderFreights().size() <= 0) && orderNo.equals(o.getOrderNo())
                    ).forEach(o -> o.setBaseFreight(freight));
                    // 回填二级子单运费
                    orderFreight.getSubOrderFreights().stream().filter(o -> o.getSubOrderFreights() != null)
                            .forEach(i -> {
                                // 回填运费
                                i.getSubOrderFreights().stream()
                                    .filter(j -> j.getOrderNo().equals(orderNo))
                                    .forEach(so -> so.setBaseFreight(freight));
                                // 计算合计
                                i.setBaseFreight(i.getSubOrderFreights().stream()
                                    .map(OrderFreight::getBaseFreight)
                                    .reduce(BigDecimal.ZERO, BigDecimal::add));
                            });
                    // 计算总运费
                    orderFreight.setBaseFreight(
                            orderFreight.getSubOrderFreights().stream()
                                    .map(OrderFreight::getBaseFreight)
                                    .reduce(BigDecimal.ZERO, BigDecimal::add)
                    );

                } catch (InterruptedException e) {
                    e.printStackTrace();
                } catch (ExecutionException e) {
                    e.printStackTrace();
                }
            }

            return orderFreight;

        }
        return null;
    }


    public void update(TempOrderVo tempOrderVo) {
        OrderFreight freight = this.get(tempOrderVo);
        tempOrderVo.setSumShippingFee(freight.getBaseFreight());
        // 处理子单运费
        freight.getSubOrderFreights().forEach(i -> {
            tempOrderVo.getSubOrders().stream().filter(j -> j.getTmpOrderNo().equals(i.getOrderNo())).forEach(
                    k -> k.setSumShippingFee(i.getBaseFreight())
            );
        });

        // 处理二级子单
        freight.getSubOrderFreights().stream()
                .filter( i -> i.getSubOrderFreights() != null).forEach(
                        i -> i.getSubOrderFreights().forEach(j -> {
                            tempOrderVo.getSubOrders().stream().filter(k -> k.getSubOrders() != null).forEach(k -> {
                                k.getSubOrders().stream().filter(l -> l.getTmpOrderNo().equals(j.getOrderNo())).forEach( m -> {
                                    m.setSumShippingFee(j.getBaseFreight());
                                });
                            });
                        })
        );
    }

    /**
     * 临时子订单转换成请求对象
     *
     * @param tempOrderVo
     * @param memberAddrId
     * @param areaCodeList
     * @return
     */
    private ShippingFeeRequetVo createShippingFeeRequest(TempSubOrderVo tempOrderVo, String memberAddrId, List<String> areaCodeList) {
        ShippingFeeRequetVo shippingFeeRequetVo = new ShippingFeeRequetVo();
        List<FashionNumsVo> fashionNums = new ArrayList<>();
        FashionNumsVo fashionNumsVo;
        for (TempOrderItemVo itemVo : tempOrderVo.getOrderItems()) {
            fashionNumsVo = new FashionNumsVo(
                    itemVo.getProductFashionId(),
                    itemVo.getProductCode(),
                    new Long(itemVo.getCount())
            );
            fashionNums.add(fashionNumsVo);
        }
        shippingFeeRequetVo.setFashionNums(fashionNums);

        MemberAddressDto address = null;
        if (!Lang.isEmpty(memberAddrId)) {
            address = memberAddressApi.findById(memberAddrId);
        }

        if (address != null && !Lang.isEmpty(address.getId())) {
            shippingFeeRequetVo.setProvinceCode(address.getProvinceCode());
            shippingFeeRequetVo.setCityCode(address.getCityCode());
            shippingFeeRequetVo.setCountyCode(address.getAreaCode());
            shippingFeeRequetVo.setTownCode(address.getTownCode());
            shippingFeeRequetVo.setAddrDetail(address.getAddressDetail());

        } else {
            if (areaCodeList.size() > 0) {
                shippingFeeRequetVo.setProvinceCode(areaCodeList.get(0));
            }
            if (areaCodeList.size() > 1) {
                shippingFeeRequetVo.setCityCode(areaCodeList.get(1));
            }
            if (areaCodeList.size() > 2) {
                shippingFeeRequetVo.setCountyCode(areaCodeList.get(2));
            }
            if (areaCodeList.size() > 3) {
                shippingFeeRequetVo.setTownCode(areaCodeList.get(3));
            }
            shippingFeeRequetVo.setAddrDetail("默认地址");
        }
        return shippingFeeRequetVo;
    }
}
