package order.service.order.helper;

import lombok.extern.slf4j.Slf4j;
import order.service.stubs.SalesPriceRateApiStub;
import order.vo.OrderSumItemPrices;
import order.vo.TempOrderItemVo;
import order.vo.TempOrderVo;
import order.vo.TempSubOrderVo;
import order.vo.response.OrderItemPrice;
import order.vo.response.OrderPrice;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import utils.Lang;
import utils.lang.Copys;
import utils.spring.SequenceGenerator;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * 临时订单相关Helper
 * @author Liang Wenxu
 * @since 2018/7/16
 */
@Component
@Slf4j
public class TempOrderHelper {
    @Value("${spring.profiles.active}")
    private String activeEnv;

    @Value("${order.temporderno.prefix:}")
    private String TEMP_ORDER_NO_PREFIX;

    @Value("${order.temporderno.dbkey:}")
    private String TEMP_ORDER_NO_KEY;

    @Value("${order.temporderno.size:10}")
    private Integer TEMP_ORDER_NO_SIZE;

    @Autowired
    private SequenceGenerator sequenceGenerator;

    @Autowired
    SalesPriceRateApiStub salesPriceRateApi;

    /**
     * 将多个临时订单合并为一个临时订单
     * 主要用在一次提交的商品走了不同的OrderTemplate之后的数据梳理
     * @param tempOrderVoList
     * @return
     */
    public TempOrderVo combime(List<TempOrderVo> tempOrderVoList) {
        if(tempOrderVoList != null && !tempOrderVoList.isEmpty()) {
            if(tempOrderVoList.size() > 1) {
                // 此时需要组合的订单是这样的：
                // 虚拟订单 01
                // -- 店铺A子单 03
                // -- 店铺B子单 04
                // 实物订单 02
                // -- 店铺A子单 05
                // -- 店铺C子单 06

                // 抽取所有子订单，然后按店铺分组
                Map<String, List<TempSubOrderVo>> groupedByStoreOrders = tempOrderVoList.stream()
                        .flatMap(o -> o.getSubOrders().stream())
                        .collect(Collectors.groupingBy(TempSubOrderVo::getStoreId))
                ;

                TempOrderVo tempOrderVo = new TempOrderVo();
                Copys copys = Copys.create();
                copys.from(tempOrderVoList.get(0))
                        .excludes("orderItems", "subOrders", "tmpOrderGifts", "orderAttrs", "orderCouponVos", "tempInvoiceVos")
                        .to(tempOrderVo);
                // 分配一个父订单号
                tempOrderVo.setTmpOrderNo(this.genTempOrderNo());

                // 按店铺合并订单
                List<TempSubOrderVo> subOrderVoList = new ArrayList<>();
                List<TempOrderItemVo> combimedOrderItems = new ArrayList<>();
                for(Map.Entry<String, List<TempSubOrderVo>> e : groupedByStoreOrders.entrySet()) {
                    List<TempSubOrderVo> tempSubOrderList = e.getValue();
                    if(tempSubOrderList != null && tempSubOrderList.size() > 0) {
                        // 合并子单的Item到主单
                        for(TempSubOrderVo subOrderVo : e.getValue()) {
                            combimedOrderItems.addAll(subOrderVo.getOrderItems());
                        }

                        // 店铺子单
                        TempSubOrderVo tempStoreOrder = new TempSubOrderVo();
                        Copys copys2 = Copys.create();
                        copys.from(tempSubOrderList.get(0))
                                .excludes("orderItems", "subOrders", "tmpOrderGifts", "orderAttrs", "orderCouponVos", "tempInvoiceVos")
                                .to(tempStoreOrder);
                        tempStoreOrder.setTmpOrderNo(this.genTempOrderNo());
                        // 将二级拆分的子单归到店铺子单下
                        tempSubOrderList.forEach(it -> it.setPOrderNo(tempStoreOrder.getTmpOrderNo()));
                        tempStoreOrder.setSubOrders(tempSubOrderList);
                        tempStoreOrder.setOrderItems(
                                tempSubOrderList.stream()
                                        .flatMap(o -> o.getOrderItems().stream())
                                        .collect(Collectors.toList())
                        );
                        OrderSumItemPrices orderSumItemPrices = calculateSumItemPrices(tempStoreOrder.getOrderItems());
                        tempStoreOrder.setSumPrice(orderSumItemPrices.getSumPrice());
                        tempStoreOrder.setSumOrginalPrice(orderSumItemPrices.getSumOrginalPrice());
                        tempStoreOrder.setSumCostPrice(orderSumItemPrices.getSumCostPrice());
                        tempStoreOrder.setSumMarketPrice(orderSumItemPrices.getSumMarketPrice());


                        // 将店铺子单归入主单
                        tempStoreOrder.setPOrderNo(tempOrderVo.getTmpOrderNo());
                        subOrderVoList.add(tempStoreOrder);

                    }

                }

                tempOrderVo.setOrderItems(combimedOrderItems);
                // 计算父订单的总价
                OrderSumItemPrices orderSumItemPrices = calculateSumItemPrices(combimedOrderItems);
                tempOrderVo.setSumPrice(orderSumItemPrices.getSumPrice());
                tempOrderVo.setSumOrginalPrice(orderSumItemPrices.getSumOrginalPrice());
                tempOrderVo.setSumCostPrice(orderSumItemPrices.getSumCostPrice());
                tempOrderVo.setSumMarketPrice(orderSumItemPrices.getSumMarketPrice());

                tempOrderVo.setSubOrders(subOrderVoList);
            } else {
                return tempOrderVoList.get(0);
            }

        }



        return null;
    }

    /**
     * 更新临时订单中指定商品的SalesPrice
     * <p>例如在临时订单中，把GoodsId为1的商品价格替换为指定价格：</p>
     * <p>TempOrderHelper.updateOrderItemPrice(tempOrderVo
     *                         , tempOrderItemVo -> "1".equals(tempOrderItemVo.getGoodsId())
     *                         , BigDecimal.valueOf(998));</p>
     * @param tempOrderVo 临时订单
     * @param filter 过滤器，指定item的搜索条件，返回true表示符合条件，false为不符合
     * @param newSalesPirce 新的销售价格（单价）
     */
    public static void updateOrderItemPrice(TempOrderVo tempOrderVo, Predicate<? super TempOrderItemVo> filter, BigDecimal newSalesPirce) {
        tempOrderVo.getOrderItems().stream()
                .filter(filter)
                .forEach(oi -> {
                    oi.setSalePrice(newSalesPirce);
                    oi.recalateSumPrices();
                });

        // 更新子订单里的售价
        if(tempOrderVo.getSubOrders() != null && tempOrderVo.getSubOrders().size() > 0) {
            updateOrderItemPrice(tempOrderVo.getSubOrders(), filter, newSalesPirce);
        }


        OrderSumItemPrices orderSumItemPrices = calculateSumItemPrices(tempOrderVo.getOrderItems());
        tempOrderVo.setSumPrice(orderSumItemPrices.getSumPrice());
        tempOrderVo.setSumOrginalPrice(orderSumItemPrices.getSumOrginalPrice());
        tempOrderVo.setSumCostPrice(orderSumItemPrices.getSumCostPrice());
        tempOrderVo.setSumMarketPrice(orderSumItemPrices.getSumMarketPrice());
    }

    /**
     * 更新子订单中的销售价格
     * @see TempOrderHelper#updateOrderItemPrice(TempOrderVo, Predicate, BigDecimal)
     * @param subOrderVoList 子订单列表
     * @param filter 过滤器，指定item的搜索条件
     * @param newSalesPrice 新指定的价格
     */
    public static void updateOrderItemPrice(List<TempSubOrderVo> subOrderVoList, Predicate<? super TempOrderItemVo> filter, BigDecimal newSalesPrice) {
        for(TempSubOrderVo tso : subOrderVoList) {
            tso.getOrderItems().stream()
                    .filter(filter)
                    .forEach(oi -> {
                        oi.setSalePrice(newSalesPrice);
                        oi.recalateSumPrices();
                    });

            if(tso.getSubOrders() != null && tso.getSubOrders().size() > 0) {
                updateOrderItemPrice(tso.getSubOrders(), filter, newSalesPrice);
            }

            OrderSumItemPrices orderSumItemPrices = calculateSumItemPrices(tso.getOrderItems());
            tso.setSumPrice(orderSumItemPrices.getSumPrice());
            tso.setSumOrginalPrice(orderSumItemPrices.getSumOrginalPrice());
            tso.setSumCostPrice(orderSumItemPrices.getSumCostPrice());
            tso.setSumMarketPrice(orderSumItemPrices.getSumMarketPrice());
        }
    }

    /**
     * 在子订单中查找指定的item
     * @param subOrderVos 子订单
     * @param filter 过滤器，指定item的搜索条件
     * @return
     */
    public static List<TempOrderItemVo> findItemInSubOrders(List<TempSubOrderVo> subOrderVos, Predicate<? super TempOrderItemVo> filter) {
        List<TempOrderItemVo> result = new ArrayList<>();
        subOrderVos.forEach(i -> {
            result.addAll(i.getOrderItems().stream().filter(filter).collect(Collectors.toList()));
            if(i.getSubOrders() != null && i.getSubOrders().size() > 0) {
                result.addAll(findItemInSubOrders(i.getSubOrders(), filter));
            }
        });

        return result;
    }

    /**
     * 根据订单商品计算订单的价格
     * @param tempOrderItemList 订单包含商品列表
     * @return
     */
    public static OrderSumItemPrices calculateSumItemPrices(List<TempOrderItemVo> tempOrderItemList) {
        OrderSumItemPrices orderSumItemPrices = new OrderSumItemPrices();
        BigDecimal orderSumPrice = new BigDecimal(0);
        BigDecimal orderSumCostPrice = new BigDecimal(0);
        BigDecimal orderSumMarketPrice = new BigDecimal(0);
        for(TempOrderItemVo orderItem : tempOrderItemList) {
            if(orderItem.getSumSalePrice() == null || orderItem.getSumSalePrice().equals(BigDecimal.ZERO)) {
                orderSumPrice = orderSumPrice.add(orderItem.getSalePrice().multiply(BigDecimal.valueOf(orderItem.getCount())));
            } else {
                orderSumPrice = orderSumPrice.add(orderItem.getSumSalePrice());
            }

            if(orderItem.getSumCostPrice() == null || orderItem.getSumCostPrice().equals(BigDecimal.ZERO)) {
                orderSumCostPrice = orderSumCostPrice.add(orderItem.getCostPrice().multiply(BigDecimal.valueOf(orderItem.getCount())));
            } else {
                orderSumCostPrice = orderSumCostPrice.add(orderItem.getSumCostPrice());
            }

            if(orderItem.getSumMarketPrice() == null || orderItem.getSumMarketPrice().equals(BigDecimal.ZERO)) {
                orderSumMarketPrice = orderSumMarketPrice.add(orderItem.getMarketPrice().multiply(BigDecimal.valueOf(orderItem.getCount())));
            } else {
                orderSumMarketPrice = orderSumMarketPrice.add(orderItem.getSumMarketPrice());
            }
        }
        orderSumItemPrices.setSumPrice(orderSumPrice);
        orderSumItemPrices.setSumOrginalPrice(orderSumPrice);
        orderSumItemPrices.setSumCostPrice(orderSumCostPrice);
        orderSumItemPrices.setSumMarketPrice(orderSumMarketPrice);

        return orderSumItemPrices;
    }

    /**
     * 生成临时订单号
     * @return
     */
    public String genTempOrderNo() {
        int fullYear = Calendar.getInstance().get(Calendar.YEAR);
        String envPrefix = this.getEnvPrefix();
        return TEMP_ORDER_NO_PREFIX + (envPrefix == null ? "" : envPrefix) + fullYear + sequenceGenerator.generateSequence(TEMP_ORDER_NO_KEY, TEMP_ORDER_NO_SIZE);
    }

    /**
     * 根据当前配置文件环境获取部署环境前缀
     * @return
     */
    private String getEnvPrefix() {
        String envStr = "";
        /** 根据不同环境配置定义订单前缀 */
        if (!Lang.isEmpty(this.activeEnv)) {
            String[] envArr = this.activeEnv.split(",");
            for (String env : envArr) {
                if (!env.equals("adaptive")) {
                    envStr = env;
                    break;
                }
            }
        }

        return envStr;
    }

    /**
     * 将订单中的所有子单以平铺列表方式返回，只获取最末级的订单
     * @param tempOrderVo
     * @return
     */
    public static List<TempSubOrderVo> flatSubOrders(TempOrderVo tempOrderVo) {
        List<TempSubOrderVo> flat = new ArrayList<>();
        tempOrderVo.getSubOrders().forEach( it -> {
            if(it.getSubOrders() != null && it.getSubOrders().size() > 0) {
                flat.addAll(flatSubOrders(it));
            } else {
                flat.add(it);
            }
        });

        return flat;
    }

    /**
     * 将订单中的所有子单以平铺列表方式返回
     * @param tempOrderVo
     * @return
     */
    public static List<TempSubOrderVo> flatSubOrders(TempSubOrderVo tempOrderVo) {
        List<TempSubOrderVo> flat = new ArrayList<>();
        tempOrderVo.getSubOrders().forEach( it -> {
            if(it.getSubOrders() != null && it.getSubOrders().size() > 0) {
                it.getSubOrders().forEach( sso -> {
                    flat.addAll(flatSubOrders(sso));
                });
            } else {
                flat.add(it);
            }
        });

        return flat;
    }

    /**
     * 更新临时订单的价格信息
     * @param tempOrderVo 临时订单
     * @param orderPrice 使用OrderPriceOperator获取的订单金额
     */
    public static void updateTempOrderPrices(TempOrderVo tempOrderVo, OrderPrice orderPrice) {
        // 先拉出所有子订单
        List<TempSubOrderVo> flatSubOrders = TempOrderHelper.flatSubOrders(tempOrderVo);
        for(TempSubOrderVo subOrder : flatSubOrders) {
            OrderPrice fp = orderPrice.findChildFirst(subOrder.getTmpOrderNo());
            if(fp != null) {
                for(OrderItemPrice oip : fp.getOrderItemPrices()) {
                    subOrder.getOrderItems().stream()
                            .filter(i -> i.getProductFashionId().equals(oip.getFashionId()))
                            .forEach(i -> {
                                i.setCostPrice(oip.getCostPrice());
                                i.setSalePrice(oip.getSalesPrice());
                                i.setMarketPrice(oip.getMarketPrice());
                                i.setOrginSalePrice(i.getSalePrice());
                                i.recalateSumPrices();
                            });

                    subOrder.recalatePrices();
                }
            }
        }
        // 更新主订单里的item金额
        orderPrice.getOrderItemPrices().forEach(oip -> {
            tempOrderVo.getOrderItems().stream()
                    .filter(i -> i.getProductFashionId().equals(oip.getFashionId()))
                    .forEach(i -> {
                        i.setCostPrice(oip.getCostPrice());
                        i.setSalePrice(oip.getSalesPrice());
                        i.setMarketPrice(oip.getMarketPrice());
                        i.setOrginSalePrice(i.getSalePrice());
                        i.recalateSumPrices();
                    });
        });

        tempOrderVo.recalatePrices();
    }
}
