package order.service.order.templates;

import activities.dto.goods.GoodsDto;
import order.model.po.OrderMain;
import lombok.extern.slf4j.Slf4j;
import member.api.dto.core.CoreCompanyDto;
import order.Contants;
import order.exceptions.NoStockException;
import order.exceptions.OrderCreateException;
import order.exceptions.temporder.TempOrderCreateException;
import order.service.operator.OrderPriceOperator;
import order.service.order.helper.TempOrderHelper;
import order.service.service.JicaiOrderChecker;
import order.service.service.OrderService;
import order.service.stubs.*;
import order.vo.*;
import order.vo.response.OrderAuditingResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import sinomall.global.common.response.BaseResponse;
import utils.GlobalContants;
import utils.Lang;
import utils.data.BeanMapper;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author Liang Wenxu
 * @since 2018/7/10
 */
@Slf4j
public class MallJicaiOrderTemplate extends MallEntityGoodsOrderTemplate {
    /**
     * 集采临时订单创建
     *
     * @param tempOrderCreateRequest
     * @return
     */
    @Override
    public TempOrderVo createTempOrder(TempOrderCreateRequest tempOrderCreateRequest) throws TempOrderCreateException {
        // 通过覆盖父类方法，对父类方法进行扩展
        TempOrderVo tempOrderVo = super.createTempOrder(tempOrderCreateRequest);
        // 再此加入集采特性 :
        return tempOrderVo;
    }

    /**
     * 创建订单，统一默认实现，在其子类覆盖此方法以实现个性化
     *
     * @param request
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public OrderVo createOrder(CreateOrderRequest request) throws OrderCreateException, NoStockException {
        OrderVo orderVo = super.createOrder(request);
        // 集采个性
        OrderMain orderMain = orderMainQueryService.getOrder(orderVo.getOrderId());
        long getBuyerCompanyTime = System.currentTimeMillis();
        CoreCompanyDto coreCompanyDto = coreCompDepartUserApi.getBuyerCompany(request.getUserId());
        log.info("获取买家机构耗时 {} ms", System.currentTimeMillis() - getBuyerCompanyTime);

        if (!Lang.isEmpty(coreCompanyDto)) {
            long getParentCompanyTime = System.currentTimeMillis();
            CoreCompanyDto compLv2 = coreCompanyApi.getParentCompany(coreCompanyDto.getId(), 2);
            log.info("获取父级机构耗时 {} ms", System.currentTimeMillis() - getParentCompanyTime);

            if (compLv2 != null) {
                orderMain.setCompanyId(compLv2.getId());
            }
        }


        // 集采订单创建订单后立即改为待审核
        OrderStatus orderStatus = new OrderStatus();
        orderStatus.setOrderId(orderVo.getOrderId());
        orderStatus.setOrderNo(orderVo.getOrderNo());
        orderStatus.setStatus(GlobalContants.ORDER_STATUS.IN_CHECK);
        orderService.statusChange(orderStatus);
        // 不进入支付页面
        orderVo.setGotoPay(false);
        return orderVo;
    }

    /**
     * 集采个性下单校验
     *
     * @param tempOrderVo
     * @return
     */
    @Override
    public BaseResponse<Boolean> orderPreCheck(TempOrderVo tempOrderVo) {
        // 先调用父类方法
        BaseResponse<Boolean> response = super.orderPreCheck(tempOrderVo);
        // 父级校验不通过时，个性化校验不需要进行
        if (!response.isSuccess()) {
            return response;
        }
        if (response.isSuccess()) {
            // 集采支付限额控制
            jicaiOrderChecker.checkOrderPriceLimit(tempOrderVo);
        }
        return response;
    }

    @Override
    public BaseResponse<Boolean> marketingDone(TempOrderVo tempOrderVo) {
        // 集采活动商品处理
        List<String> allGoodsIds = tempOrderVo.getOrderItems().stream()
                .filter(it -> !Lang.isEmpty(it.getGoodsId()))
                .map(TempOrderItemVo::getGoodsId)
                .collect(Collectors.toList());

        List<goods.dto.goods.GoodsDto> goodsDtos = goodsApi.findByIdIn(allGoodsIds);
        List<GoodsDto> actGoodsDtos = BeanMapper.mapList(goodsDtos, BeanMapper.getType(goods.dto.goods.GoodsDto.class), BeanMapper.getType(GoodsDto.class));
        List<Map<String, Object>> actRs = activityGoodsApi.isActivityGoodsAndGetActivityPriceBatch(actGoodsDtos, tempOrderVo.getOrganizationCode());
        // 活动价回写
        if (actRs != null) {
            actRs.forEach(m -> {
                if ((Boolean) m.get("isActivityGoods")) {
                    TempOrderHelper.updateOrderItemPrice(tempOrderVo
                            , tempOrderItemVo -> m.get("goodsId").equals(tempOrderItemVo.getGoodsId())
                            , (BigDecimal) m.get("price"));
                }
            });
        }

        // 营销活动处理完成
        return new BaseResponse<>(true, "SUC", true);
    }

    @Override
    public BaseResponse<Boolean> orderPayPreSubmitDone(OrderVo orderVo) {
        BaseResponse<Boolean> superRs = super.orderPayPreSubmitDone(orderVo);
        // 集采支付提交处理
        return superRs;
    }

    /**
     * 订单审核（批量），审核通过订单可进入生产状态（待审核），审核不通过则取消订单
     *
     * @param orderKeys 需操作的订单索引
     * @param passed    true|false（审核通过|审核不通过）
     * @return 返回中包括各订单的审核处理情况
     */
    @Override
    public List<OrderAuditingResult> audit(List<OrderKey> orderKeys, Boolean passed) {
//        List<OrderAuditingResult> r = super.audit(orderKeys, passed);
        List<OrderAuditingResult> orderAuditingResults = new ArrayList<>();
        // 集采个性化处理，下单后进入待审核状态
        if (passed) {
            orderKeys.forEach(orderKey -> {
                OrderAuditingResult orderAuditingResult=new OrderAuditingResult();
                orderAuditingResult.setSuccess(true);
                orderAuditingResult.setOrderKey(orderKey);
                orderAuditingResults.add(orderAuditingResult);
            });
//            // 通过审核
//            orderMainService.updateBatchStatusByOrderNo(
//                    orderKeys.stream().map(OrderKey::getOrderNo).collect(Collectors.toList())
//                    , GlobalContants.ORDER_STATUS.IN_CHECK);
        }
        return orderAuditingResults;
    }

    @Override
    public BaseResponse<CancelOrderResp> cancelOrder(List<OrderKey> orderKeys, Contants.CancelOrderType cancelType) {
        return super.cancelOrder(orderKeys, cancelType);
    }

    /**
     * 订单收货
     *
     * @param orderKey    订单ID或orderNo
     * @param receiptType 订单收货类型，见{@link Contants}中ORDER_RECEIPT_TYPE_开头
     */
    @Override
    public void orderReceipt(OrderKey orderKey, String receiptType) {

    }

    @Autowired
    SalesPriceRateApiStub salesPriceRateApiStub;

    @Autowired
    TempOrderHelper tempOrderHelper;

    @Autowired
    OrderService orderService;

    @Autowired
    ActivityGoodsApiStub activityGoodsApi;

    @Autowired
    ProductFashionApiStub productFashionApi;

    @Autowired
    GoodsApiStub goodsApi;

    @Autowired
    OrderPriceOperator orderPriceOperator;

    @Autowired
    BiddingMainApiStub biddingMainApi;

    @Autowired
    JicaiOrderChecker jicaiOrderChecker;
}
