package order.service.order.templates;

import order.Contants;
import order.exceptions.IllegalAreaCodeException;
import order.exceptions.NoStockException;
import order.exceptions.OrderCreateException;
import order.exceptions.temporder.TempOrderCreateException;
import order.vo.*;
import order.vo.response.OrderAuditingResult;
import order.vo.response.OrderFreight;
import order.vo.response.OrderStock;
import outsideapi.exceptions.SupplierInterfaceInvokeException;
import outsideapi.exceptions.SupplierOrderComfirmFalidException;
import sinomall.global.common.response.BaseResponse;

import java.util.List;

/**
 * 订单处理模型接口，定义订单相关操作
 * @author Liang Wenxu
 * @since 2018/7/9
 */
public interface OrderTemplate {

    /**
     * 创建临时订单
     * @param tempOrderCreateRequest
     * @return
     */
    TempOrderVo createTempOrder(final TempOrderCreateRequest tempOrderCreateRequest) throws TempOrderCreateException;

    /**
     * 库存校验
     * @param tempOrderVo
     * @param areaCode
     * @param organizationId
     * @return
     */
    BaseResponse<OrderStock> validateStock(final TempSubOrderVo tempOrderVo, String areaCode, String organizationId) throws IllegalAreaCodeException;

    /**
     * 获取运费
     * @param tempOrderVo
     * @return
     */
    BaseResponse<OrderFreight> getFreight(final TempOrderVo tempOrderVo);

    /**
     * 订单预校验
     * @param tempOrderVo
     * @return
     */
    BaseResponse<Boolean> orderPreCheck(TempOrderVo tempOrderVo);

    /**
     * 营销活动
     * @param tempOrderVo
     * @return
     */
    BaseResponse<Boolean> marketingDone(TempOrderVo tempOrderVo);

    /**
     * 创建订单
     * @param request
     * @return
     */
    OrderVo createOrder(final CreateOrderRequest request) throws OrderCreateException, NoStockException;

    /**
     * 预占库存
     * @param orderVo
     * @return
     */
    BaseResponse<Boolean> subscribeOrder(OrderVo orderVo);

    /**
     * 订单审核（批量），审核通过订单可进入生产状态（如待支付），审核不通过则取消订单
     * <p>目前订单流程中，是没有审核流程的（不是集采的审核），引入审核的目的是为了适应以后可能出现的从创建订单到允许支付之间的业务流程</p>
     * @param orderKeys 需操作的订单索引
     * @param passed true|false（审核通过|审核不通过）
     * @return 返回中包括各订单的审核处理情况
     */
    List<OrderAuditingResult> audit(List<OrderKey> orderKeys, Boolean passed);

    /**
     * 订单支付前回调
     * @param orderVo
     * @return
     */
    BaseResponse<Boolean> orderPayPreSubmitDone(OrderVo orderVo);

    /**
     * 订单支付回调
     * @param orderVo
     * @return
     */
    OrderPayConfirmResp paySubmit(OrderVo orderVo) throws SupplierInterfaceInvokeException, SupplierOrderComfirmFalidException;

    /**
     * 取消订单
     * @param orderKeys 订单主键，其中orderNo和orderId可任选其一
     * @param cancelType
     * @return
     */
    BaseResponse<CancelOrderResp> cancelOrder(List<OrderKey> orderKeys, Contants.CancelOrderType cancelType);

    /**
     * 订单收货
     * @param orderKey 订单ID或orderNo
     * @param receiptType 订单收货类型，见{@link Contants}中ORDER_RECEIPT_TYPE_开头常量
     */
    void orderReceipt(OrderKey orderKey, String receiptType);

    /**
     * 刷新订单状态
     * @param orderKeys
     * @return
     */
    BaseResponse<List<OrderVo>> refresh(List<OrderKey> orderKeys);
}
