package order.service.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import payment.model.po.PayLog;
import payment.model.po.Payment;
import payment.model.po.PaymentConfig;
import payment.model.repository.PayLogRepos;
import payment.model.repository.PaymentConfigRepos;
import payment.model.repository.PaymentRepos;
import utils.lang.Copys;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * @author Liang Wenxu
 * @since 2018/8/8
 */
@Slf4j
@Service
@Transactional(readOnly = true, rollbackFor = Exception.class)
public class PaymentService {
    @Autowired
    PaymentConfigRepos paymentConfigRepos;
    @Autowired
    PaymentRepos paymentRepos;
    @Autowired
    JdbcTemplate jdbcTemplate;
    @Autowired
    PayLogRepos payLogRepos;

    @Transactional(readOnly = false)
    public Payment saveOrUpdate(Payment payment, Boolean flush) {
        Copys copys = Copys.create();
        PaymentConfig config = paymentConfigRepos.findById(payment.getConfig().getId());
        if(config == null) {
            config = new PaymentConfig();
            copys.from(payment.getConfig()).to(config).clear();
            paymentConfigRepos.save(config);
        }

        Payment p = new Payment();
        copys.from(payment).excludes("config").to(p).clear();
        p.setConfig(config);
        if(flush) {
            return paymentRepos.saveAndFlush(p);
        } else {
            return paymentRepos.save(p);
        }
    }

    @Transactional(readOnly = true)
    public Payment findByid(String id) {
        Payment payment = paymentRepos.findById(id);
        payment.setConfig(paymentConfigRepos.findById(payment.getConfig().getId()));
        return payment;
    }

    @Transactional(readOnly = true)
    public List<Payment> findByIds(List<String> ids) {
        List<PaymentConfig> configList = new ArrayList<>();
        PaymentConfig newConfig;
        List<Payment> paymentList = paymentRepos.findByIdIn(ids);
        Copys copys = Copys.create();
        for(Payment payment : paymentList) {
            if(payment != null) {
                newConfig = new PaymentConfig();
                copys.clear().excludes("payment").from(payment.getConfig()).to(newConfig);
                payment.setConfig(newConfig);
            }
        }
        return paymentRepos.findByIdIn(ids);
    }

    /**
     * 深拷贝
     * @param payment
     * @return
     */
    public Payment cpyPayment(Payment payment) {
        Copys copys = Copys.create();
        PaymentConfig config = paymentConfigRepos.findById(payment.getConfig().getId());
        if(config == null) {
            config = new PaymentConfig();
            copys.from(payment.getConfig()).to(config).clear();
        }

        Payment p = new Payment();
        copys.from(payment).excludes("config").to(p).clear();
        p.setConfig(config);

        return p;
    }

    public int updatePaymentPayLogId(String paymentId, Integer payLogId) {
        String sql = "update payment set pay_log_id = ? where id = ?";
        int number = jdbcTemplate.update(sql, payLogId, paymentId);
        return number;
    }


    public String getPayStatus(Integer payLogId) {
        PayLog payLog = payLogRepos.findOne(payLogId);
        if (payLog == null) throw new RuntimeException("支付成功时,检测到payLog不存在");
        else
            return payLog.getStatus();
    }

    @Transactional(readOnly = false)
    public int updatePayLogStatus(Integer payLogId, String status) {
        String sql = "update pay_log set status =? where id=?";
        //修改payLog的状态
        int number = jdbcTemplate.update(sql, status, payLogId);
        return number;
    }

    @Transactional(readOnly = false)
    public int updatePaymentStatus(String paymentId, String status) {
        String sql = "update payment set status =?,last_updated=? where id=?";
        //修改payment的状态
        int number = jdbcTemplate.update(sql, status,new Date(), paymentId);
        return number;
    }
}
