package order.service.stubs;

import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import lombok.extern.slf4j.Slf4j;
import member.api.MemberApi;
import member.api.dto.core.CoreUserDto;
import member.api.dto.shop.MemberDto;
import member.api.dto.shop.MemberGroupDto;
import member.api.vo.MemberVo;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import utils.sql.PageVo;

import java.util.Date;
import java.util.List;

/**
 * @author Liang Wenxu
 * @since 2018/8/3
 */
@Slf4j
@Component
@Transactional(propagation = Propagation.NOT_SUPPORTED)
public class MemberApiStub {
    /**
     * 取用户信息
     * @param coreUserId 用户id
     * @return CoreUser, 根据传入的用户id，返回相应的用户信息
     */
    public CoreUserDto getUserInfo(String coreUserId) {
        return memberApi.getUserInfo(coreUserId);
    }

    /**
     * 根据会员id取会员信息
     * @param memberId 会员id
     * @return MemberVo，根据传入的会员id，返回相应的会员信息VO,具体返回VO请查询->
     * @see MemberVo
     */
    public MemberVo getMemberInfo(String memberId) {
        return memberApi.getMemberInfo(memberId);
    }

    /**
     * 根据用户id取会员信息
     * @param userId 用户id
     * @return MemberVo, 根据传入的用户id，返回相应的会员信息VO,具体返回VO请查询->
     * @see MemberVo
     */
    public MemberVo getMemberInfoByUserId(String userId) {
        return memberApi.getMemberInfoByUserId(userId);
    }

    /**
     * 根据手机号码取会员信息
     * @param phone 手机号码
     * @return MemberVo, 根据传入的用户手机号，返回相应的会员信息VO,具体返回VO请查询->
     * @see MemberVo
     */
    public MemberVo getMemberInfoByUserPhone(String phone) {
        return memberApi.getMemberInfoByUserPhone(phone);
    }

    /**
     * 获取SSO统一登录导入用户，只需提供某个参数即可，其余参数传null，多个参数时查询条件优先级 uid > email > mobile > name
     * @param uid 第三方登录关联表openId
     * @param email 邮箱
     * @param mobile 手机号码
     * @param name 真实姓名
     * @return MemberVo, 根据传入的参数，返回相应的会员信息VO,具体返回VO请查询->
     * @see MemberVo
     */
    public MemberVo getSsoMemberInfo(String uid, String email, String mobile, String name) {
        return memberApi.getSsoMemberInfo(uid, email, mobile, name);
    }

    /**
     * 查询系统中定义的所有的有效会员组
     * @return
     */
    public List<MemberGroupDto> listAllMemberGroup() {
        return memberApi.listAllMemberGroup();
    }

    /**
     * 根据CoreUser取会员信息
     * @param coreUser 用户信息，具体实体请查询->
     * @return Member, 根据传入的用户信息，返回相应的会员信息
     */
    public MemberDto getMemberInfoByCoreUser(CoreUserDto coreUser) {
        return memberApi.getMemberInfoByCoreUser(coreUser);
    }

    /**
     * 分页取会员信息
     * @param pageRequest 请求参数，具体请求实体请查询->
     * @see PageRequest
     * @return Page<Member>,根据传入的参数，返回相应的具有分页功能的会员信息
     */
    public Page<MemberDto> getMemberPage(PageRequest pageRequest) {
        return memberApi.getMemberPage(pageRequest);
    }

    /**
     * 分页获取会员信息
     * @param startDate 创建起始时间
     * @param endDate 创建终止时间
     * @param name 用户名称
     * @param type 用户类型
     * @param company 机构id
     * @param pageable 分页信息
     * @return Page<Member>，返回具有分页功能的会员信息
     */
    public Page<MemberDto> listMembers(Date startDate, Date endDate, String name, String type, String company, Pageable pageable) {
        return memberApi.listMembers(startDate, endDate, name, type, company, pageable);
    }

    /**
     * 根据organizationId查询member
     * @param organizationId 合作伙伴id
     * @return List<Member>，根据传入的合作伙伴id，返回该合作机构下的会员信息列表
     */
    public List<MemberDto> findMemberByOrganizationId(String organizationId) {
        return memberApi.findMemberByOrganizationId(organizationId);
    }

    /**
     * 根据id查询member
     * @param id 会员id
     * @return Member, 根据传入的会员id，返回相应的会员信息
     */
    public MemberDto findMemberById(String id) {
        return memberApi.findMemberById(id);
    }

    /**
     * 根据用户信息列表，查询相应的会员信息列表
     * @param coreUser 用户信息列表，具体请求实体请查询->
     * @return List<Member>，根据传入的用户信息列表，顺序返回相应的会员信息列表
     */
    public List<MemberDto> findMembersByCoreUsers(List<CoreUserDto> coreUser) {
        return memberApi.findMembersByCoreUsers(coreUser);
    }

    /**
     * 分页查询会员信息
     * @param username 用户名，为空，则忽略此查询条件
     * @param companyId 机构id
     * @param organizationId 合作伙伴id
     * @param pageable 分页信息
     * @return Page<Member>，根据传入的参数，返回相应的带有分页功能的会员信息
     */
    public Page<MemberDto> findMemberByParams(String username, String companyId, String organizationId, Pageable pageable) {
        return memberApi.findMemberByParams(username, companyId, organizationId, pageable);
    }

    /**
     * 分页查询会员信息
     * @param pageSize 一页记录数
     * @param pageNum 当前页码
     * @param userId 用户id
     * @param searchLoginId 邮箱或手机，可为空
     * @param memberName 会员真实姓名
     * @param organizationId
     * @return PageVo，根据参数，返回相应的会员信息VO,带分页
     */
    public PageVo findMemberPageByUserIdAndName(int pageSize, int pageNum, String userId, String searchLoginId, String memberName, String organizationId) {
        return memberApi.findMemberPageByUserIdAndName(pageSize, pageNum, userId, searchLoginId, memberName, organizationId);
    }

    /**
     * 根据Example实体查找会员信息
     * @param example 匹配条件，具体请求参数请查询->
     * @return List<Member>，根据请求条件，返回相应的会员信息列表
     */
    public List<MemberDto> findMemberByExample(MemberDto example) {
        return memberApi.findMemberByExample(example);
    }

    /**
     * 根据Example实体查找(结果分页)
     * @param example 匹配条件
     * @param pageable 分页参数
     * @return Page<Member>，根据传入的请求参数，返回相应的会员信息。带分页
     */
    public Page<MemberDto> findMemberByExample(MemberDto example, Pageable pageable) {
        return memberApi.findMemberByExample(example, pageable);
    }

    /**
     * 根据用户信息列表，查询相应的会员信息
     * @param users 用户信息列表，具体实体请查询->
     * @return List<Member>，根据传入的用户信息列表，顺序返回相应的会员信息列表
     */
    public List<MemberDto> findByUsers(List<CoreUserDto> users) {
        return memberApi.findByUsers(users);
    }

    /**
     * 根据用户id查询会员信息
     * @param user_id 用户id
     * @return List<Member>，根据传入的用户id，返回相应的会员信息列表
     */
    public List<MemberDto> findByCoreUserId(String user_id) {
        return memberApi.findByCoreUserId(user_id);
    }

    public List<MemberDto> findByRealNameLikeAndOrganization(String orderName, String organizationId) {
        return memberApi.findByRealNameLikeAndOrganization(orderName, organizationId);
    }

    /**
     * 通过coreuserIds查找member
     * @param coreUserIds
     * @return
     */
    public List<MemberDto> findByCoreUserIds(List<String> coreUserIds) {
        return memberApi.findByCoreUserIds(coreUserIds);
    }

    /**
     * 获取 CoreUserId
     *
     * @param id MemberId
     * @return CoreUserId
     */
    public String findCoreUserIdById(String id) {
        return memberApi.findCoreUserIdById(id);
    }

    /**
     * 获取机构ID
     *
     * @param id MemberId
     * @return 机构ID
     */
    public String findOrganizationIdById(String id) {
        return memberApi.findOrganizationIdById(id);
    }

    /**
     * 获取用户姓名
     * @param userId coreUserId
     * @return String
     */
    public String findRealNameByCoreUserId(String userId) {
        return memberApi.findRealNameByCoreUserId(userId);
    }

    /**
     * 获取用户姓名
     * @param id coreUserId
     * @return String
     */
    public String findRealNameById(String id) {
        return memberApi.findRealNameById(id);
    }

    @MotanReferer
    MemberApi memberApi;


}
