package order.service.stubs;

import cms.api.OrganizationApi;
import cms.api.dto.OrganizationDto;
import cms.api.dto.SalesPriceRateDto;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import utils.Lang;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author Liang Wenxu
 * @since 2018/8/3
 */
@Slf4j
@Component
@Transactional(propagation = Propagation.NOT_SUPPORTED)
public class OrganizationApiStub {
    private static Map<String, String> orgIdCodeMap = new ConcurrentHashMap<>();


    /**
     * 根据机构的主键ID查询机构
     *
     * @param id 机构的主键ID
     * @return Organization 查询得到的机构
     */
    public OrganizationDto findOrganizationById(String id) {
        return organizationApi.findOrganizationById(id);
    }

    /**
     * 根据机构的名称查询全部机构
     *
     * @param name 机构的名称
     * @return List<Organization> 查询得到的机构集合
     */
    public List<OrganizationDto> findOrganizationByOrgName(String name) {
        return organizationApi.findOrganizationByOrgName(name);
    }

    /**
     * 根据机构的名称查询未删除的机构
     *
     * @param name 机构的名称
     * @return List<Organization> 查询得到的机构集合
     */
    public List<OrganizationDto> findOrganizationByOrgNameAndIsDelete(String name) {
        return organizationApi.findOrganizationByOrgNameAndIsDelete(name);
    }

    /**
     * 根据机构的名称查询机构
     *
     * @param orgName  机构的名称
     * @param pageable
     * @return Page<Organization> 查询得到的机构列表
     */
    public Page<OrganizationDto> findOrganizationByOrgName(String orgName, Pageable pageable) {
        return organizationApi.findOrganizationByOrgName(orgName, pageable);
    }

    /**
     * 查找所有未删除的机构
     *
     * @return List<Organization> 所有的机构的集合
     */
    public List<OrganizationDto> findOrganizationAll() {
        return organizationApi.findOrganizationAll();
    }

    /**
     * 查找所有的机构, 并以机构的ID为KEY, 名称为VALUE构成键值对, 添加到MAP中
     *
     * @return Map(orgId, orgName)
     */
    public Map listToMap() {
        return organizationApi.listToMap();
    }

    /**
     * 根据机构的企业代码查找所有未删除的机构
     *
     * @param organizationCode 机构的企业代码
     * @return Organization 查询得到的机构
     */
    public OrganizationDto findOrganizationByOrgCode(String organizationCode) {
        return organizationApi.findOrganizationByOrgCode(organizationCode);
    }

    /**
     * 将销售价格费率表中的机构的ID替换为机构的名称, 用于页面展示
     *
     * @param salesPriceRates 销售价格费率列表
     * @return List<SalesPriceRate> 销售价格费率集合
     */
    public List<SalesPriceRateDto> getOrgNameList(Page<SalesPriceRateDto> salesPriceRates) {
        return organizationApi.getOrgNameList(salesPriceRates);
    }

    public void insPectPrepaidDeposit() {
        organizationApi.insPectPrepaidDeposit();
    }

    /**
     * 获取机构ID
     *
     * @param organizationCode 机构代码
     * @return 机构ID
     */
    public String findOrganizationIdByOrgCode(String organizationCode) {
        Optional<Map.Entry<String, String>> me = orgIdCodeMap.entrySet().stream()
                .filter(e -> e.getValue().equals(organizationCode)).findFirst();
        String id;
        if(me.isPresent()) {
            id = me.get().getKey();
        } else {
            id = organizationApi.findOrganizationIdByOrgCode(organizationCode);
            orgIdCodeMap.put(id, organizationCode);
        }

       return id;
    }

    /**
     * 获取机构名称
     *
     * @param organizationCode 机构代码
     * @return 机构ID
     */
    public String findOrganizationNameByOrgCode(String organizationCode) {
        return organizationApi.findOrganizationNameByOrgCode(organizationCode);
    }

    /**
     * 获取机构代码
     * @param id 机构ID
     * @return 机构代码
     */
    public String findOrganizationCodeById(String id) {
        String code = orgIdCodeMap.get(id);
        if(Lang.isEmpty(code)) {
            code = organizationApi.findOrganizationCodeById(id);
            orgIdCodeMap.put(id, code);
        }

        return code;
    }

    @MotanReferer
    OrganizationApi organizationApi;
}
