package order.service.stubs;

import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import goods.api.ProductApi;
import goods.dto.product.ProductCategoryDto;
import goods.dto.product.ProductDto;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * ProductApi的本地代理
 * @author Liang Wenxu
 * @since 2018/7/17
 */
@Component
@Transactional(propagation = Propagation.NOT_SUPPORTED)
public class ProductApiStub {
    /**
     * 根据id获取产品
     * @param id
     * @return
     */
    public ProductDto findById(String id) {
        return productApi.findById(id);
    }

    public ProductDto findProductDtoById(String id) {
        return productApi.findProductDtoById(id);
    }

    /**
     * 根据产品名称模糊查找产品
     * @param productName
     * @param pageable
     * @return
     */
    public Page<ProductDto> findProductPage(String productName, Pageable pageable) {
        return productApi.findProductPage(productName, pageable);
    }

    /**
     * 根据分类、产品名模糊查找产品
     * @param categoryId
     * @param productName
     * @param pageable
     * @return
     */
    public Page<ProductDto> findProductPage(String categoryId, String productName, Pageable pageable) {
        return productApi.findProductPage(categoryId, productName, pageable);
    }

    /**
     * 根据产品名称、分类、是否删除查找产品
     * @param productName
     * @param categoryId
     * @param isDelete
     * @param pageable
     * @return
     */
    public Page<ProductDto> findProductByParamPage(String productName, String categoryId, Boolean isDelete, Pageable pageable) {
        return productApi.findProductByParamPage(productName, categoryId, isDelete, pageable);
    }

    /**
     * 根据产品名、供应商、分类查找产品
     * @param productName
     * @param storeId
     * @param productCategory
     * @param pageable
     * @return
     */
    public Page<ProductDto> getProductPage(String productName, String storeId, ProductCategoryDto productCategory, Pageable pageable) {
        return productApi.getProductPage(productName, storeId, productCategory, pageable);
    }

    /**
     * 根据多个id查找产品
     * @param ids
     * @return
     */
    public List<ProductDto> findByIds(Iterable<String> ids) {
        return productApi.findByIds(ids);
    }

    /**
     * 产品信息查询（分页）
     * @param queryRequest 请求数据，特别说明：
     *                     <p>1. 精确匹配，DTO的属性不为null即为进行该属性的全匹配查询</p>
     *                     <p>2. 模糊匹配，DTO的属性不为null且值中包含了'%'即为进行该属性的模糊匹配查询</p>
     *                     <p>3. in查询，使用inQuerys字段组织查询</p>
     *                     <p>4. range范围查询，使用rangeQueries字段组织查询</p>
     * @param pageable 分页
     * @return {@link ApiResponseVo}:
     *      1. 本API专用的错误代码在此API接口处定义相关常量
     *
     */
    public Page<ProductDto> query(ProductDto queryRequest, Pageable pageable) {
        return productApi.query(queryRequest, pageable);
    }

    /**
     * 产品信息查询（不分页）
     * @param queryRequest 请求数据，特别说明：
     *                     <p>1. 精确匹配，DTO的属性不为null即为进行该属性的全匹配查询</p>
     *                     <p>2. 模糊匹配，DTO的属性不为null且值中包含了'%'即为进行该属性的模糊匹配查询</p>
     *                     <p>3. in查询，使用inQuerys字段组织查询</p>
     *                     <p>4. range范围查询，使用rangeQueries字段组织查询</p>
     * @return {@link ApiResponseVo}:
     *      1. 本API专用的错误代码在此API接口处定义相关常量
     *
     */
    public List<ProductDto> query(ProductDto queryRequest) {
        return productApi.query(queryRequest);
    }

    /**
     * 根据产品ID获取产品
     * @param productIds 产品ID，可多个
     * @return
     */
    public List<ProductDto> findById(List<String> productIds) {
        return productApi.findById(productIds);
    }


    /**
     * 校验产品编码是否重复，同个店铺中，productCode不能重复
     * @param productCode 产品编码（sku），第三方商品编码
     * @param storeId 店铺ID（供应商ID）
     * @return
     */
    public Boolean productCodeExists(String productCode, String storeId) {
        return productApi.productCodeExists(productCode, storeId);
    }


    public Page<ProductDto> findByProductCategory(ProductCategoryDto productCategory, Pageable pageable) {
        return productApi.findByProductCategory(productCategory, pageable);
    }

    public Page<ProductDto> findByProductCategoryAndIsDeleteAndStoreIdIn(ProductCategoryDto productCategory, boolean b, String[] storeIds, Pageable pageable) {
        return productApi.findByProductCategoryAndIsDeleteAndStoreIdIn(productCategory, b, storeIds, pageable);
    }

    /**
     * @author taofeng
     * @date 2018/6/8
     *
     * 根据productId获取productCategoryId和layerIndex
     * List.get(0)是categoryId,List.get(1)是Layer
     * @param productId
     */
    public List<String> findCategoryIdAndLayerByProductId(String productId) {
        return productApi.findCategoryIdAndLayerByProductId(productId);
    }

    @MotanReferer
    ProductApi productApi;

}
