package order.service.stubs;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONException;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import goods.api.ProductFashionApi;
import goods.dto.product.ProductDto;
import goods.dto.product.ProductFashionDto;
import goods.vo.FashionIdAndCountVo;
import goods.vo.FashionStockRequest;
import goods.vo.FashionStockResult;
import goods.vo.ProductFashionVo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import sinomall.global.common.response.BaseResponse;
import utils.Lang;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author Liang Wenxu
 * @since 2018/7/17
 */
@Component
@Slf4j
@Transactional(propagation = Propagation.NOT_SUPPORTED)
public class ProductFashionApiStub {
    public static final String PRODUCT_FASHION_STUB_CACHING_KEY_PREFIX = "order-service:stub:productfashion:cache";
    public static final Integer PRODUCT_FASHION_STUB_CACHING_TIMEOUT = 5000;


    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    /**
     * 获取缓存的Map
     * @return
     */
    private Map getCachingMap() {
        BoundValueOperations<String, String>  operations = redisTemplate.boundValueOps(PRODUCT_FASHION_STUB_CACHING_KEY_PREFIX);
        String mapJson = operations.get();
        if(!Lang.isEmpty(mapJson)) {
            try {
                HashMap cachingMap = JSON.parseObject(mapJson, HashMap.class);
                return cachingMap;
            } catch (JSONException e) {
                log.error("解析缓存JSON报错，JSON原文:{}", mapJson, e);
            }
        }

        return null;
    }

    /**
     * 获取单个缓存对象
     * @param id ProductFashionId
     * @return
     */
    private ProductFashionDto getInCache(String id) {
        try {
            Map cachingMap = getCachingMap();
            if(cachingMap != null) {
                ProductFashionDto bean = (ProductFashionDto) cachingMap.get(id);
                return bean;
            }
        } catch (ClassCastException ce) {
            log.error("缓存对象类型转换失败", ce);
        }
        return null;
    }

    /**
     * 将实体加入缓存中
     * @param entitys
     */
    private void addToCache(List<ProductFashionDto> entitys) {
        BoundValueOperations<String, String>  operations =
                redisTemplate.boundValueOps(PRODUCT_FASHION_STUB_CACHING_KEY_PREFIX);
        Map cachingMap = getCachingMap();

        entitys.forEach(e -> {
            if(e != null) {
                cachingMap.put(e.getId(), e);
            }
        });
        operations.set(JSON.toJSONString(cachingMap), PRODUCT_FASHION_STUB_CACHING_TIMEOUT, TimeUnit.MILLISECONDS);
    }

    /**
     * 按ID获取商品规格信息
     *
     * @param id
     * @param useCache
     * @return
     */
    public ProductFashionDto findById(String id, Boolean useCache) {
        ProductFashionDto bean = getInCache(id);
        if(!useCache || bean == null) {
            bean = productFashionApi.findById(id);
        }
        return bean;
    }

    public ProductFashionDto findDtoById(String id, Boolean useCache) {
        ProductFashionDto bean = getInCache(id);
        if(!useCache || bean == null) {
            bean = productFashionApi.findDtoById(id);
        }
        return bean;
    }

    /**
     * 获取简单的返回，除了图片之外所有关联都不做处理
     * @param id
     * @return
     */
    public ProductFashionDto findSimple(String id) {
        return productFashionApi.findSimple(id);
    }

    public List<ProductFashionDto> findByProduct(ProductDto product) {
        return productFashionApi.findByProduct(product);
    }

    public ProductFashionVo findByGoodsId(String goodsId) {
        return productFashionApi.findByGoodsId(goodsId);
    }

    /**
     * 根据SKU查询
     * @param skus sku对应ProductFashion中的productCode
     * @return
     */
    public List<ProductFashionDto> findAllBySku(List<String> skus) {
        return productFashionApi.findAllBySku(skus);
    }

    /**
     * 根据sku查询productFashionId
     * @param sku
     * @return
     */
    public List<String> findProductFashionIdBySku(String sku) {
        return productFashionApi.findProductFashionIdBySku(sku);
    }

    /**
     * 根据productFashionId查sku
     * @param productFashionId
     * @return
     */
    public String findSkuByProductFashionId(String productFashionId) {
        return productFashionApi.findSkuByProductFashionId(productFashionId);
    }

    /**
     * 获取指定产品规格的库存（获取京东等第三方平台实时库存）
     *
     * @param fashionIds 产品规格ID
     * @param areaCode   地区代码
     * @return
     */
    public Map<String, Object> getStockCounts(List<String> fashionIds, String areaCode) {
        return productFashionApi.getStockCounts(fashionIds, areaCode);
    }

    public ProductFashionDto findOne(String id) {
        return productFashionApi.findOne(id);
    }

    /**
     * 按id列表查找productFashion
     *
     * @param ids
     * @return
     */
    public List<ProductFashionDto> findByIds(List<String> ids) {
        return productFashionApi.findByIds(ids);
    }

    /**
     * 查询产品规格的库存（规范化）
     * @param request 请求对象，其中的所有属性都是必须参数
     * @return 详情看 {@link FashionStockResult}
     */
    public BaseResponse<FashionStockResult> queryFashionStockStatus(FashionStockRequest request) {
        return productFashionApi.queryFashionStockStatus(request);
    }

    /**
     * 查询产品规格的库存（规范化）
     *
     * @param fashionIdAndCountVoList 产品规格及购买数量Vo列表
     * @param areaCode
     * @param orgId
     * @return 返回Map: {
     * status: 状态
     * data: [
     * noStockFlag: 包含无库存商品标识，true: 包含有无库存商品 | false: 不包含无库存商品
     * details: ArrayList<Map> -> [
     * {fashionId: "1111", onSale: true|false, stockCount: int}
     * ]  // 详细数据
     * ] // 返回数据
     * }
     */
    public Map queryFashionStockStatus(List<FashionIdAndCountVo> fashionIdAndCountVoList, String areaCode, String orgId) {
        return productFashionApi.queryFashionStockStatus(fashionIdAndCountVoList, areaCode, orgId);
    }

    public List<ProductFashionVo> findByProductId(String productId) {
        return productFashionApi.findByProductId(productId);
    }

    public List<String> findIdsByProductId(String productId) {
        return productFashionApi.findIdsByProductId(productId);
    }

    public String findIdByGoodsId(String goodsId) {
        return productFashionApi.findIdByGoodsId(goodsId);
    }

    @MotanReferer
    ProductFashionApi productFashionApi;
}
