package order.web.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import member.api.CoreUserApi;
import member.api.LoginApi;
import member.api.vo.MemberVo;
import okhttp3.*;
import order.web.service.OrderWebService;
import order.web.vo.TempOrderCreateWebRequest;
import order.web.vo.WebTempOrderVo;
import order.web.vo.response.WebOrderPrice;
import order.web.vo.response.WebOrderStock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.bind.annotation.RequestBody;
import sinomall.global.dto.WebResponse;
import utils.GlobalContants;
import utils.Lang;
import utils.security.MD5Utils;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 商城订单WEB-API
 *
 * @author Liang Wenxu
 * @since 2018/7/20
 */
@Slf4j
@Api(value = "mallorder", description = "订单相关API", tags = "OrderService")
@RestController
@RequestMapping("api/mallorder")
public class MallOrderWebApiController {
    @Autowired
    OrderWebService orderWebService;

    @MotanReferer
    CoreUserApi coreUserApi;

    @MotanReferer
    LoginApi loginApi;
    @Autowired
    RedisTemplate redisTemplate;
    final String templateOrderKey = "order:temporder:key:";

    @ApiOperation(value = "创建临时订单"
            , httpMethod = "POST"
            , notes = "该API需用户登录后才可使用，接口支持【从指定商品】或【指定购物车项目】创建临时订单，从指定商品创建时，products参数不能为空，从购物车创建时cartItemIds不为空" +
            "<p><b>返回代码说明：</b><br/>" +
            "0001: 未提交任何商品<br/>" +
            "0002: 获取登录用户失败<br/>" +
            "0003: 指定的临时订单获取失败<br/>" +
            "1001: 未提供商品数据<br/>" +
            "1999: 未知参数异常<br/>" +
            "2001: 订单中包含了无货商品<br/>" +
            "2002: 下单地区areaCode不合法<br/>" +
            "9999: 未定义异常<br/>" +
            "</p>")
    @RequestMapping(value = "/temporder", method = RequestMethod.POST)
    public WebResponse<WebTempOrderVo> tempOrder(@RequestBody TempOrderCreateWebRequest tempOrderCreateRequest, HttpServletRequest request) {
        WebResponse<WebTempOrderVo> response;

        if ((tempOrderCreateRequest.getCartItemIds() == null || Lang.isEmpty(tempOrderCreateRequest.getCartItemIds()))
                && (tempOrderCreateRequest.getProducts() == null || Lang.isEmpty(tempOrderCreateRequest.getProducts()))) {
            /** 错误1，未提交任何商品 */
            response = WebResponse.ERROR("0001", "Required Paramter cartItemIds Is Empty!");
        } else {
            String userAgent = request.getHeader("USER-AGENT").toLowerCase();
            MemberVo memberVo = orderWebService.findMemberVo(request);

            if (memberVo != null) {
                BoundValueOperations tempOrderLock = redisTemplate.boundValueOps(templateOrderKey + memberVo.getMember().getId() + ":" + tempOrderCreateRequest.getRequestId());
                if (tempOrderLock.setIfAbsent(true)) {
                    tempOrderCreateRequest.setUserAgent(userAgent);
                    response = orderWebService.createTempOrder(tempOrderCreateRequest, memberVo, request.getSession());
                } else {
                    response = WebResponse.ERROR("0005", "request repeat");
                }
            } else {
                /** 错误2，获取登录用户失败 */
                response = WebResponse.ERROR("0002", "Member Not Found!");

            }
        }

        return response;
    }

    @ApiOperation(value = "临时订单详情获取接口"
            , httpMethod = "GET"
            , notes = "通过此接口获取临时订单的信息" +
            "<p><b>返回代码说明：</b><br/>" +
            "0001: 临时订单号不存在<br/>" +
            "</p>"
    )
    @ApiImplicitParams(value = {
            @ApiImplicitParam(value = "临时订单号", name = "orderNo", paramType = "path")
    })
    @RequestMapping(value = "/temporder/{orderNo}", method = RequestMethod.GET)
    public WebResponse<WebTempOrderVo> getTempOrder(
            @PathVariable("orderNo") String tempOrderNo,
            HttpServletRequest request) {
        WebResponse<WebTempOrderVo> response = null;
        WebTempOrderVo tempOrderVo = orderWebService.getTempOrder(tempOrderNo, request.getSession());
        if (tempOrderVo != null) {
            response = WebResponse.SUCCESS(tempOrderVo);
        } else {
            response = WebResponse.ERROR("0001", "该订单不存在");

        }
        return response;
    }

    @ApiOperation(value = "临时订单库存获取接口"
            , httpMethod = "GET"
            , notes = "通过此接口获取最新的库存状态" +
            "<p><b>返回代码说明：</b><br/>" +
            "0001: 临时订单号不存在<br/>" +
            "0002: areaCode未指定<br/>" +
            "0003: 指定的临时订单获取失败<br/>" +
            "1001: areaCode代码不合法<br/>" +
            "</p>"
    )
    @ApiImplicitParams(value = {
            @ApiImplicitParam(value = "临时订单号", name = "orderNo", paramType = "path"),
            @ApiImplicitParam(value = "最末级地区编码", name = "areaCode", paramType = "path")
    })
    @RequestMapping(value = "/temporder/stock/{orderNo}/{areaCode}", method = RequestMethod.GET)
    public WebResponse<WebOrderStock> temporderStock(
            @PathVariable("orderNo") String tempOrderNo,
            @PathVariable("areaCode") String areaCode,
            HttpServletRequest request
    ) {
        WebResponse<WebOrderStock> response = new WebResponse<>();
        if (!Lang.isEmpty(areaCode)) {
            // 临时订单是否存在校验
            if (orderWebService.tmpOrderIdExists(tempOrderNo, request.getSession())) {
                WebTempOrderVo tempOrderVo = orderWebService.getTempOrder(tempOrderNo, request.getSession());
                if (tempOrderVo != null) {
                    tempOrderVo.setAreaCode(areaCode);
                    WebResponse<WebOrderStock> orderStockResp = orderWebService.getOrderStock(tempOrderVo);
                    if (orderStockResp.getSuccess()) {
                        response = WebResponse.SUCCESS(orderStockResp.getData());
                    } else {
                        response = WebResponse.ERROR(orderStockResp.getErrorCode(), orderStockResp.getMessage());
                    }
                } else {
                    response = WebResponse.ERROR("0003", "指定的订单不存在");
                }

            } else {
                response = WebResponse.ERROR("0001", "没有找到订单号" + tempOrderNo + "对应的临时订单");
            }
        } else {
            response = WebResponse.ERROR("0002", "缺少必要参数areaCode");

        }

        return response;
    }

    @ApiOperation(value = "重新获取临时订单的价格"
            , httpMethod = "GET"
            , notes = "通过此接口获取最新的库存状态" +
            "<p><b>返回代码说明：</b><br/>" +
            "0001: 临时订单号不存在<br/>" +
            "0002: areaCode未指定<br/>" +
            "0003: 指定的临时订单获取失败<br/>" +
            "1001: areaCode代码不合法<br/>" +
            "</p>"
    )
    @ApiImplicitParams(value = {
            @ApiImplicitParam(value = "临时订单号", name = "orderNo", paramType = "path"),
            @ApiImplicitParam(value = "最末级地区编码", name = "areaCode", paramType = "path")
    })
    @RequestMapping(value = "/temporder/price/{orderNo}/{areaCode}", method = RequestMethod.GET)
    public WebResponse<WebOrderPrice> tempOrderPrice(
            @PathVariable("orderNo") String tempOrderNo,
            @PathVariable("areaCode") String areaCode,
            HttpServletRequest request
    ) {
        WebResponse<WebOrderPrice> response = new WebResponse<>();
        if (!Lang.isEmpty(areaCode)) {
            // 临时订单是否存在校验
            if (orderWebService.tmpOrderIdExists(tempOrderNo, request.getSession())) {
                WebTempOrderVo tempOrderVo = orderWebService.getTempOrder(tempOrderNo, request.getSession());
                if (tempOrderVo != null) {
                    tempOrderVo.setAreaCode(areaCode);
                    response = orderWebService.getOrderPrice(tempOrderVo);
                } else {
                    response = WebResponse.ERROR("0003", "指定的订单不存在");
                }

            } else {
                response = WebResponse.ERROR("0001", "没有找到订单号" + tempOrderNo + "对应的临时订单");
            }
        } else {
            response = WebResponse.ERROR("0002", "缺少必要参数areaCode");

        }

        return response;
    }

    @ApiOperation(value = "测试登陆用接口，生产环境关闭！")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userName", value = "用户邮箱", required = true, dataType = "string", paramType = "query", defaultValue = "feijin@ec.cntaiping.com"),
            @ApiImplicitParam(name = "pwd", value = "密码", required = true, dataType = "string", paramType = "query", defaultValue = "qwer1234"),
            @ApiImplicitParam(name = "orgId", value = "渠道代码", required = true, dataType = "string", paramType = "query", defaultValue = "09577ac9-10db-41c9-8aad-35a75b379310")
    })
    @RequestMapping(value = "/testLogin", method = RequestMethod.GET)
    public WebResponse testLogin(
            @RequestParam("userName") String userName,
            @RequestParam("pwd") String pwd,
            @RequestParam("orgId") String orgId,
            HttpServletRequest request) {

        MemberVo memberVo = loginApi.checkByEmail(userName, pwd, request.getRemoteHost(), orgId);
        request.getSession().setAttribute(GlobalContants.SESSION_MEMBER_ID, memberVo.getMember().getId());
        request.getSession().setAttribute(GlobalContants.SESSION_USER_ID, memberVo.getUser().getId());
        request.getSession().setAttribute(GlobalContants.SESSION_USER_NAME, memberVo.getUser().getUsername());
        request.getSession().setAttribute(GlobalContants.SESSION_MEMBER_VO_JSON, JSON.toJSONString(memberVo));
        request.getSession().setAttribute(GlobalContants.SESSION_USER_TYPE, memberVo.getUser().getUserType());
        return WebResponse.SUCCESS(memberVo);
    }

    @RequestMapping(value = "/testTrdApis", method = RequestMethod.GET)
    public Object testTrdApis() throws Exception {
        String username = "deli";
//        String password = MD5Utils.encrypt("deli_user_secret");
        String password = "deli_user_secret";
        String timestamp = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date());
        String clientSecret = "deli_secret";
        String clientId = "deli";
        String accessToken = "password";
        String sign = clientSecret + timestamp + clientId + username + password + accessToken + clientSecret;
        System.out.println("sign=" + sign);
        sign = MD5Utils.encrypt(sign).toUpperCase();
        System.out.println("sign=" + sign);
        String url = "http://localhost/api/oauth/token";
        String data =
                "grant_type=access_token" +
                        "&client_id=" + clientId +
                        "&username=" + username +
                        "&password=" + password +
                        "&timestamp=" + timestamp +
                        "&sign=" + sign;

        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        OkHttpClient okHttpClient = builder.addInterceptor(new Interceptor() {
            @Override
            public Response intercept(Chain chain) throws IOException {
                Request request = chain.request();

                long t1 = System.nanoTime();
                log.info("Sending request {} on {}{}",
                        request.url(), chain.connection());
                log.info("Sending request body {} ",
                        JSON.toJSONString(request.body()));


                Response response = chain.proceed(request);

                long t2 = System.nanoTime();
                log.info("Received response for {} in {}ms{}{}",
                        response.request().url(), (t2 - t1) / 1e6d,
                        response.headers());
                okhttp3.MediaType mediaType = response.body().contentType();

                String content = response.body().string();
                log.info("content:{}", content);
                System.out.println("response====================>" + content);
                //response = chain.proceed(request);
                return response
                        .newBuilder()
                        .body(okhttp3.ResponseBody.create(mediaType,
                                content)).build();
            }
        }).build();
        FormBody formBody = new FormBody
                .Builder()
                //设置参数名称和参数值
                .add("grant_type", accessToken)
                .add("client_id", clientId)
                .add("username", username)
                .add("password", password)
                .add("timestamp", timestamp)
                .add("sign", sign)
                .add("client_secret", clientSecret)
                .build();

        Request request = new Request.Builder().url(url).post(formBody).build();
        Response response = okHttpClient.newCall(request).execute();

        JSONObject object = JSON.parseObject(response.body().string());
        String access_token = object.getString("access_token");


        JSONObject jsonObject = new JSONObject();
        jsonObject.put("subOrderNo", "8000132575");
        okhttp3.RequestBody requestBody = okhttp3.RequestBody.create(MediaType.parse("application/json"), jsonObject.toString());

        Request request2 = new Request.Builder()
                .addHeader("Content-Type", "application/json; charset=utf-8")
                .url("http://localhost/api/modify/orders/received?access_token=" + access_token + "&timestamp=" + timestamp + "&version=1.0")
                .post(requestBody).build();
        Response response2 = okHttpClient.newCall(request2).execute();
        return response2.body().string();
    }
}
