/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dubbo.common.utils;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MulticastSocket;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.regex.Pattern;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.config.ConfigurationUtils;
import org.apache.dubbo.common.logger.Logger;
import org.apache.dubbo.common.logger.LoggerFactory;
import org.apache.dubbo.common.utils.CIDRUtils;
import org.apache.dubbo.common.utils.LRUCache;

public class NetUtils {
    private static final Logger logger = LoggerFactory.getLogger(NetUtils.class);
    private static final int RND_PORT_START = 30000;
    private static final int RND_PORT_RANGE = 10000;
    private static final int MIN_PORT = 0;
    private static final int MAX_PORT = 65535;
    private static final Pattern ADDRESS_PATTERN = Pattern.compile("^\\d{1,3}(\\.\\d{1,3}){3}\\:\\d{1,5}$");
    private static final Pattern LOCAL_IP_PATTERN = Pattern.compile("127(\\.\\d{1,3}){3}$");
    private static final Pattern IP_PATTERN = Pattern.compile("\\d{1,3}(\\.\\d{1,3}){3,5}$");
    private static final Map<String, String> HOST_NAME_CACHE = new LRUCache<String, String>(1000);
    private static volatile InetAddress LOCAL_ADDRESS = null;
    private static final String SPLIT_IPV4_CHARECTER = "\\.";
    private static final String SPLIT_IPV6_CHARECTER = ":";

    public static int getRandomPort() {
        return 30000 + ThreadLocalRandom.current().nextInt(10000);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static int getAvailablePort() {
        try (ServerSocket ss = new ServerSocket();){
            ss.bind(null);
            int n = ss.getLocalPort();
            return n;
        }
        catch (IOException e) {
            return NetUtils.getRandomPort();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static int getAvailablePort(int port) {
        if (port <= 0) {
            return NetUtils.getAvailablePort();
        }
        int i = port;
        while (i < 65535) {
            try (ServerSocket ss = new ServerSocket(i);){
                int n = i;
                return n;
            }
            catch (IOException iOException) {
                ++i;
            }
        }
        return port;
    }

    public static boolean isInvalidPort(int port) {
        return port <= 0 || port > 65535;
    }

    public static boolean isValidAddress(String address) {
        return ADDRESS_PATTERN.matcher(address).matches();
    }

    public static boolean isLocalHost(String host) {
        return host != null && (LOCAL_IP_PATTERN.matcher(host).matches() || host.equalsIgnoreCase("localhost"));
    }

    public static boolean isAnyHost(String host) {
        return "0.0.0.0".equals(host);
    }

    public static boolean isInvalidLocalHost(String host) {
        return host == null || host.length() == 0 || host.equalsIgnoreCase("localhost") || host.equals("0.0.0.0") || LOCAL_IP_PATTERN.matcher(host).matches();
    }

    public static boolean isValidLocalHost(String host) {
        return !NetUtils.isInvalidLocalHost(host);
    }

    public static InetSocketAddress getLocalSocketAddress(String host, int port) {
        return NetUtils.isInvalidLocalHost(host) ? new InetSocketAddress(port) : new InetSocketAddress(host, port);
    }

    static boolean isValidV4Address(InetAddress address) {
        if (address == null || address.isLoopbackAddress()) {
            return false;
        }
        String name = address.getHostAddress();
        boolean result = name != null && IP_PATTERN.matcher(name).matches() && !"0.0.0.0".equals(name) && !"127.0.0.1".equals(name);
        return result;
    }

    static boolean isPreferIPV6Address() {
        boolean preferIpv6 = Boolean.getBoolean("java.net.preferIPv6Addresses");
        if (!preferIpv6) {
            return false;
        }
        return false;
    }

    static InetAddress normalizeV6Address(Inet6Address address) {
        String addr = address.getHostAddress();
        int i = addr.lastIndexOf(37);
        if (i > 0) {
            try {
                return InetAddress.getByName(addr.substring(0, i) + '%' + address.getScopeId());
            }
            catch (UnknownHostException e) {
                logger.debug("Unknown IPV6 address: ", e);
            }
        }
        return address;
    }

    public static String getLocalHost() {
        InetAddress address = NetUtils.getLocalAddress();
        return address == null ? "127.0.0.1" : address.getHostAddress();
    }

    public static String filterLocalHost(String host) {
        if (host == null || host.length() == 0) {
            return host;
        }
        if (host.contains("://")) {
            URL u = URL.valueOf(host);
            if (NetUtils.isInvalidLocalHost(u.getHost())) {
                return u.setHost(NetUtils.getLocalHost()).toFullString();
            }
        } else if (host.contains(SPLIT_IPV6_CHARECTER)) {
            int i = host.lastIndexOf(58);
            if (NetUtils.isInvalidLocalHost(host.substring(0, i))) {
                return NetUtils.getLocalHost() + host.substring(i);
            }
        } else if (NetUtils.isInvalidLocalHost(host)) {
            return NetUtils.getLocalHost();
        }
        return host;
    }

    public static String getIpByConfig() {
        String configIp = ConfigurationUtils.getProperty("DUBBO_IP_TO_BIND");
        if (configIp != null) {
            return configIp;
        }
        return NetUtils.getIpByHost(NetUtils.getLocalAddress().getHostName());
    }

    public static InetAddress getLocalAddress() {
        InetAddress localAddress;
        if (LOCAL_ADDRESS != null) {
            return LOCAL_ADDRESS;
        }
        LOCAL_ADDRESS = localAddress = NetUtils.getLocalAddress0();
        return localAddress;
    }

    private static Optional<InetAddress> toValidAddress(InetAddress address) {
        if (address instanceof Inet6Address) {
            Inet6Address v6Address = (Inet6Address)address;
            if (NetUtils.isPreferIPV6Address()) {
                return Optional.ofNullable(NetUtils.normalizeV6Address(v6Address));
            }
        }
        if (NetUtils.isValidV4Address(address)) {
            return Optional.of(address);
        }
        return Optional.empty();
    }

    private static InetAddress getLocalAddress0() {
        InetAddress localAddress = null;
        try {
            localAddress = InetAddress.getLocalHost();
            Optional<InetAddress> addressOp = NetUtils.toValidAddress(localAddress);
            if (addressOp.isPresent()) {
                return addressOp.get();
            }
        }
        catch (Throwable e) {
            logger.warn(e);
        }
        try {
            Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
            if (null == interfaces) {
                return localAddress;
            }
            while (interfaces.hasMoreElements()) {
                try {
                    NetworkInterface network = interfaces.nextElement();
                    if (network.isLoopback() || network.isVirtual() || !network.isUp()) continue;
                    Enumeration<InetAddress> addresses = network.getInetAddresses();
                    while (addresses.hasMoreElements()) {
                        try {
                            Optional<InetAddress> addressOp = NetUtils.toValidAddress(addresses.nextElement());
                            if (!addressOp.isPresent()) continue;
                            try {
                                if (!addressOp.get().isReachable(100)) continue;
                                return addressOp.get();
                            }
                            catch (IOException iOException) {
                            }
                        }
                        catch (Throwable e) {
                            logger.warn(e);
                        }
                    }
                }
                catch (Throwable e) {
                    logger.warn(e);
                }
            }
        }
        catch (Throwable e) {
            logger.warn(e);
        }
        return localAddress;
    }

    public static String getHostName(String address) {
        try {
            String hostname;
            int i = address.indexOf(58);
            if (i > -1) {
                address = address.substring(0, i);
            }
            if ((hostname = HOST_NAME_CACHE.get(address)) != null && hostname.length() > 0) {
                return hostname;
            }
            InetAddress inetAddress = InetAddress.getByName(address);
            if (inetAddress != null) {
                hostname = inetAddress.getHostName();
                HOST_NAME_CACHE.put(address, hostname);
                return hostname;
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        return address;
    }

    public static String getIpByHost(String hostName) {
        try {
            return InetAddress.getByName(hostName).getHostAddress();
        }
        catch (UnknownHostException e) {
            return hostName;
        }
    }

    public static String toAddressString(InetSocketAddress address) {
        return address.getAddress().getHostAddress() + SPLIT_IPV6_CHARECTER + address.getPort();
    }

    public static InetSocketAddress toAddress(String address) {
        int port;
        String host;
        int i = address.indexOf(58);
        if (i > -1) {
            host = address.substring(0, i);
            port = Integer.parseInt(address.substring(i + 1));
        } else {
            host = address;
            port = 0;
        }
        return new InetSocketAddress(host, port);
    }

    public static String toURL(String protocol, String host, int port, String path) {
        StringBuilder sb = new StringBuilder();
        sb.append(protocol).append("://");
        sb.append(host).append(':').append(port);
        if (path.charAt(0) != '/') {
            sb.append('/');
        }
        sb.append(path);
        return sb.toString();
    }

    public static void joinMulticastGroup(MulticastSocket multicastSocket, InetAddress multicastAddress) throws IOException {
        NetUtils.setInterface(multicastSocket, multicastAddress instanceof Inet6Address);
        multicastSocket.setLoopbackMode(false);
        multicastSocket.joinGroup(multicastAddress);
    }

    public static void setInterface(MulticastSocket multicastSocket, boolean preferIpv6) throws IOException {
        boolean interfaceSet = false;
        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        while (interfaces.hasMoreElements()) {
            NetworkInterface i = interfaces.nextElement();
            Enumeration<InetAddress> addresses = i.getInetAddresses();
            while (addresses.hasMoreElements()) {
                InetAddress address = addresses.nextElement();
                if (preferIpv6 && address instanceof Inet6Address) {
                    multicastSocket.setInterface(address);
                    interfaceSet = true;
                    break;
                }
                if (preferIpv6 || !(address instanceof Inet4Address)) continue;
                multicastSocket.setInterface(address);
                interfaceSet = true;
                break;
            }
            if (!interfaceSet) continue;
            break;
        }
    }

    public static boolean matchIpExpression(String pattern, String host, int port) throws UnknownHostException {
        if (pattern.contains("/")) {
            CIDRUtils utils = new CIDRUtils(pattern);
            return utils.isInRange(host);
        }
        return NetUtils.matchIpRange(pattern, host, port);
    }

    public static boolean matchIpRange(String pattern, String host, int port) throws UnknownHostException {
        if (pattern == null || host == null) {
            throw new IllegalArgumentException("Illegal Argument pattern or hostName. Pattern:" + pattern + ", Host:" + host);
        }
        if ((pattern = pattern.trim()).equals("*.*.*.*") || pattern.equals("*")) {
            return true;
        }
        InetAddress inetAddress = InetAddress.getByName(host);
        boolean isIpv4 = NetUtils.isValidV4Address(inetAddress);
        String[] hostAndPort = NetUtils.getPatternHostAndPort(pattern, isIpv4);
        if (hostAndPort[1] != null && !hostAndPort[1].equals(String.valueOf(port))) {
            return false;
        }
        pattern = hostAndPort[0];
        String splitCharacter = SPLIT_IPV4_CHARECTER;
        if (!isIpv4) {
            splitCharacter = SPLIT_IPV6_CHARECTER;
        }
        String[] mask = pattern.split(splitCharacter);
        NetUtils.checkHostPattern(pattern, mask, isIpv4);
        host = inetAddress.getHostAddress();
        String[] ipAddress = host.split(splitCharacter);
        if (pattern.equals(host)) {
            return true;
        }
        if (!NetUtils.ipPatternContainExpression(pattern)) {
            InetAddress patternAddress = InetAddress.getByName(pattern);
            return patternAddress.getHostAddress().equals(host);
        }
        for (int i = 0; i < mask.length; ++i) {
            if (mask[i].equals("*") || mask[i].equals(ipAddress[i])) continue;
            if (mask[i].contains("-")) {
                String[] rangeNumStrs = mask[i].split("-");
                if (rangeNumStrs.length != 2) {
                    throw new IllegalArgumentException("There is wrong format of ip Address: " + mask[i]);
                }
                Integer min = NetUtils.getNumOfIpSegment(rangeNumStrs[0], isIpv4);
                Integer max = NetUtils.getNumOfIpSegment(rangeNumStrs[1], isIpv4);
                Integer ip = NetUtils.getNumOfIpSegment(ipAddress[i], isIpv4);
                if (ip >= min && ip <= max) continue;
                return false;
            }
            if ("0".equals(ipAddress[i]) && ("0".equals(mask[i]) || "00".equals(mask[i]) || "000".equals(mask[i]) || "0000".equals(mask[i])) || mask[i].equals(ipAddress[i])) continue;
            return false;
        }
        return true;
    }

    private static boolean ipPatternContainExpression(String pattern) {
        return pattern.contains("*") || pattern.contains("-");
    }

    private static void checkHostPattern(String pattern, String[] mask, boolean isIpv4) {
        if (!isIpv4) {
            if (mask.length != 8 && NetUtils.ipPatternContainExpression(pattern)) {
                throw new IllegalArgumentException("If you config ip expression that contains '*' or '-', please fill qulified ip pattern like 234e:0:4567:0:0:0:3d:*. ");
            }
            if (mask.length != 8 && !pattern.contains("::")) {
                throw new IllegalArgumentException("The host is ipv6, but the pattern is not ipv6 pattern : " + pattern);
            }
        } else if (mask.length != 4) {
            throw new IllegalArgumentException("The host is ipv4, but the pattern is not ipv4 pattern : " + pattern);
        }
    }

    private static String[] getPatternHostAndPort(String pattern, boolean isIpv4) {
        String[] result = new String[2];
        if (pattern.startsWith("[") && pattern.contains("]:")) {
            int end = pattern.indexOf("]:");
            result[0] = pattern.substring(1, end);
            result[1] = pattern.substring(end + 2);
            return result;
        }
        if (pattern.startsWith("[") && pattern.endsWith("]")) {
            result[0] = pattern.substring(1, pattern.length() - 1);
            result[1] = null;
            return result;
        }
        if (isIpv4 && pattern.contains(SPLIT_IPV6_CHARECTER)) {
            int end = pattern.indexOf(SPLIT_IPV6_CHARECTER);
            result[0] = pattern.substring(0, end);
            result[1] = pattern.substring(end + 1);
            return result;
        }
        result[0] = pattern;
        return result;
    }

    private static Integer getNumOfIpSegment(String ipSegment, boolean isIpv4) {
        if (isIpv4) {
            return Integer.parseInt(ipSegment);
        }
        return Integer.parseInt(ipSegment, 16);
    }
}

