/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Random;
import net.jpountz.xxhash.XXHashFactory;
import org.apache.kafka.common.record.BufferSupplier;
import org.apache.kafka.common.record.KafkaLZ4BlockInputStream;
import org.apache.kafka.common.record.KafkaLZ4BlockOutputStream;
import org.hamcrest.CoreMatchers;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class KafkaLZ4Test {
    private static final Random RANDOM = new Random(0L);
    private final boolean useBrokenFlagDescriptorChecksum;
    private final boolean ignoreFlagDescriptorChecksum;
    private final byte[] payload;
    private final boolean close;
    private final boolean blockChecksum;
    @Rule
    public ExpectedException thrown = ExpectedException.none();

    @Parameterized.Parameters(name="{index} useBrokenFlagDescriptorChecksum={0}, ignoreFlagDescriptorChecksum={1}, blockChecksum={2}, close={3}, payload={4}")
    public static Collection<Object[]> data() {
        ArrayList<Payload> payloads = new ArrayList<Payload>();
        payloads.add(new Payload("empty", new byte[0]));
        payloads.add(new Payload("onebyte", new byte[]{1}));
        for (int size : Arrays.asList(1000, 65536, 98304)) {
            byte[] random = new byte[size];
            RANDOM.nextBytes(random);
            payloads.add(new Payload("random", random));
            byte[] ones = new byte[size];
            Arrays.fill(ones, (byte)1);
            payloads.add(new Payload("ones", ones));
        }
        ArrayList<Object[]> values = new ArrayList<Object[]>();
        for (Payload payload : payloads) {
            for (boolean broken : Arrays.asList(false, true)) {
                for (boolean ignore : Arrays.asList(false, true)) {
                    for (boolean blockChecksum : Arrays.asList(false, true)) {
                        for (boolean close : Arrays.asList(false, true)) {
                            values.add(new Object[]{broken, ignore, blockChecksum, close, payload});
                        }
                    }
                }
            }
        }
        return values;
    }

    public KafkaLZ4Test(boolean useBrokenFlagDescriptorChecksum, boolean ignoreFlagDescriptorChecksum, boolean blockChecksum, boolean close, Payload payload) {
        this.useBrokenFlagDescriptorChecksum = useBrokenFlagDescriptorChecksum;
        this.ignoreFlagDescriptorChecksum = ignoreFlagDescriptorChecksum;
        this.payload = payload.payload;
        this.close = close;
        this.blockChecksum = blockChecksum;
    }

    @Test
    public void testHeaderPrematureEnd() throws Exception {
        this.thrown.expect(IOException.class);
        this.thrown.expectMessage("Stream ended prematurely");
        ByteBuffer buffer = ByteBuffer.allocate(2);
        this.makeInputStream(buffer);
    }

    private KafkaLZ4BlockInputStream makeInputStream(ByteBuffer buffer) throws IOException {
        return new KafkaLZ4BlockInputStream(buffer, BufferSupplier.create(), this.ignoreFlagDescriptorChecksum);
    }

    @Test
    public void testNotSupported() throws Exception {
        this.thrown.expect(IOException.class);
        this.thrown.expectMessage("Stream unsupported (invalid magic bytes)");
        byte[] compressed = this.compressedBytes();
        compressed[0] = 0;
        this.makeInputStream(ByteBuffer.wrap(compressed));
    }

    @Test
    public void testBadFrameChecksum() throws Exception {
        if (!this.ignoreFlagDescriptorChecksum) {
            this.thrown.expect(IOException.class);
            this.thrown.expectMessage("Stream frame descriptor corrupted");
        }
        byte[] compressed = this.compressedBytes();
        compressed[6] = -1;
        this.makeInputStream(ByteBuffer.wrap(compressed));
    }

    @Test
    public void testBadBlockSize() throws Exception {
        if (!this.close || this.useBrokenFlagDescriptorChecksum && !this.ignoreFlagDescriptorChecksum) {
            return;
        }
        this.thrown.expect(IOException.class);
        this.thrown.expectMessage(CoreMatchers.containsString((String)"exceeded max"));
        byte[] compressed = this.compressedBytes();
        ByteBuffer buffer = ByteBuffer.wrap(compressed).order(ByteOrder.LITTLE_ENDIAN);
        int blockSize = buffer.getInt(7);
        blockSize = blockSize & Integer.MIN_VALUE | 0x1000000;
        buffer.putInt(7, blockSize);
        this.testDecompression(buffer);
    }

    @Test
    public void testCompression() throws Exception {
        boolean contentSize;
        byte[] compressed = this.compressedBytes();
        int offset = 0;
        Assert.assertEquals((long)4L, (long)compressed[offset++]);
        Assert.assertEquals((long)34L, (long)compressed[offset++]);
        Assert.assertEquals((long)77L, (long)compressed[offset++]);
        Assert.assertEquals((long)24L, (long)compressed[offset++]);
        byte flg = compressed[offset++];
        int version = flg >>> 6 & 3;
        Assert.assertEquals((long)1L, (long)version);
        int reserved = flg & 3;
        Assert.assertEquals((long)0L, (long)reserved);
        byte bd = compressed[offset++];
        int blockMaxSize = bd >>> 4 & 7;
        Assert.assertTrue((blockMaxSize >= 4 ? 1 : 0) != 0);
        Assert.assertTrue((blockMaxSize <= 7 ? 1 : 0) != 0);
        reserved = bd & 0xF;
        Assert.assertEquals((long)0L, (long)reserved);
        reserved = bd >>> 7 & 1;
        Assert.assertEquals((long)0L, (long)reserved);
        boolean bl = contentSize = (flg >>> 3 & 1) != 0;
        if (contentSize) {
            offset += 8;
        }
        int off = 4;
        int len = offset - 4;
        if (this.useBrokenFlagDescriptorChecksum) {
            off = 0;
            len = offset;
        }
        int hash = XXHashFactory.fastestInstance().hash32().hash(compressed, off, len, 0);
        byte hc = compressed[offset++];
        Assert.assertEquals((long)((byte)(hash >> 8 & 0xFF)), (long)hc);
        if (this.close) {
            offset = compressed.length - 4;
            Assert.assertEquals((long)0L, (long)compressed[offset++]);
            Assert.assertEquals((long)0L, (long)compressed[offset++]);
            Assert.assertEquals((long)0L, (long)compressed[offset++]);
            Assert.assertEquals((long)0L, (long)compressed[offset++]);
        }
    }

    @Test
    public void testArrayBackedBuffer() throws IOException {
        byte[] compressed = this.compressedBytes();
        this.testDecompression(ByteBuffer.wrap(compressed));
    }

    @Test
    public void testArrayBackedBufferSlice() throws IOException {
        byte[] compressed = this.compressedBytes();
        int sliceOffset = 12;
        ByteBuffer buffer = ByteBuffer.allocate(compressed.length + sliceOffset + 123);
        buffer.position(sliceOffset);
        buffer.put(compressed).flip();
        buffer.position(sliceOffset);
        ByteBuffer slice = buffer.slice();
        this.testDecompression(slice);
        int offset = 42;
        buffer = ByteBuffer.allocate(compressed.length + sliceOffset + offset);
        buffer.position(sliceOffset + offset);
        buffer.put(compressed).flip();
        buffer.position(sliceOffset);
        slice = buffer.slice();
        slice.position(offset);
        this.testDecompression(slice);
    }

    @Test
    public void testDirectBuffer() throws IOException {
        byte[] compressed = this.compressedBytes();
        ByteBuffer buffer = ByteBuffer.allocateDirect(compressed.length);
        buffer.put(compressed).flip();
        this.testDecompression(buffer);
        int offset = 42;
        buffer = ByteBuffer.allocateDirect(compressed.length + offset + 123);
        buffer.position(offset);
        buffer.put(compressed).flip();
        buffer.position(offset);
        this.testDecompression(buffer);
    }

    @Test
    public void testSkip() throws Exception {
        if (!this.close || this.useBrokenFlagDescriptorChecksum && !this.ignoreFlagDescriptorChecksum) {
            return;
        }
        KafkaLZ4BlockInputStream in = this.makeInputStream(ByteBuffer.wrap(this.compressedBytes()));
        int n = 100;
        int remaining = this.payload.length;
        long skipped = in.skip((long)n);
        Assert.assertEquals((long)Math.min(n, remaining), (long)skipped);
        n = 10000;
        remaining = (int)((long)remaining - skipped);
        skipped = in.skip((long)n);
        Assert.assertEquals((long)Math.min(n, remaining), (long)skipped);
    }

    private void testDecompression(ByteBuffer buffer) throws IOException {
        IOException error = null;
        try {
            int n;
            KafkaLZ4BlockInputStream decompressed = this.makeInputStream(buffer);
            byte[] testPayload = new byte[this.payload.length];
            byte[] tmp = new byte[1024];
            int pos = 0;
            int i = 0;
            while ((n = decompressed.read(tmp, i, tmp.length - i)) != -1) {
                if ((i += n) != tmp.length) continue;
                System.arraycopy(tmp, 0, testPayload, pos, i);
                pos += i;
                i = 0;
            }
            System.arraycopy(tmp, 0, testPayload, pos, i);
            Assert.assertEquals((long)-1L, (long)decompressed.read(tmp, 0, tmp.length));
            Assert.assertEquals((long)this.payload.length, (long)(pos += i));
            Assert.assertArrayEquals((byte[])this.payload, (byte[])testPayload);
        }
        catch (IOException e) {
            if (!this.ignoreFlagDescriptorChecksum && this.useBrokenFlagDescriptorChecksum) {
                Assert.assertEquals((Object)"Stream frame descriptor corrupted", (Object)e.getMessage());
                error = e;
            }
            if (!this.close) {
                Assert.assertEquals((Object)"Stream ended prematurely", (Object)e.getMessage());
                error = e;
            }
            throw e;
        }
        if (!this.ignoreFlagDescriptorChecksum && this.useBrokenFlagDescriptorChecksum) {
            Assert.assertNotNull((Object)error);
        }
        if (!this.close) {
            Assert.assertNotNull((Object)error);
        }
    }

    private byte[] compressedBytes() throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        KafkaLZ4BlockOutputStream lz4 = new KafkaLZ4BlockOutputStream((OutputStream)output, 4, this.blockChecksum, this.useBrokenFlagDescriptorChecksum);
        lz4.write(this.payload, 0, this.payload.length);
        if (this.close) {
            lz4.close();
        } else {
            lz4.flush();
        }
        return output.toByteArray();
    }

    static class Payload {
        String name;
        byte[] payload;

        Payload(String name, byte[] payload) {
            this.name = name;
            this.payload = payload;
        }

        public String toString() {
            return "Payload{size=" + this.payload.length + ", name='" + this.name + '\'' + '}';
        }
    }
}

