package org.apache.velocity.context;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.    
 */

import java.util.HashMap;

import org.apache.velocity.runtime.RuntimeServices;
import org.apache.velocity.runtime.RuntimeConstants;
import org.apache.velocity.runtime.directive.VMProxyArg;
import org.apache.velocity.util.introspection.IntrospectionCacheData;
import org.apache.velocity.runtime.resource.Resource;
import org.apache.velocity.app.event.EventCartridge;
import org.apache.velocity.exception.MethodInvocationException;

/**
 *  This is a special, internal-use-only context implementation to be
 *  used for the new Velocimacro implementation.
 *
 *  The main distinguishing feature is the management of the VMProxyArg objects
 *  in the put() and get() methods.
 *
 *  Further, this context also supports the 'VM local context' mode, where
 *  any put() of references that aren't args to the VM are considered
 *  local to the vm, protecting the global context.
 *
 *  @author <a href="mailto:geirm@optonline.net">Geir Magnusson Jr.</a>
 *  @version $Id: VMContext.java 471908 2006-11-06 22:39:28Z henning $
 */
public class VMContext implements InternalContextAdapter
{
    /** container for our VMProxy Objects */
    HashMap vmproxyhash = new HashMap();

    /** container for any local or constant VMProxy items */
    HashMap localcontext = new HashMap();

    /** the base context store.  This is the 'global' context */
    InternalContextAdapter innerContext = null;

    /** context that we are wrapping */
    InternalContextAdapter wrappedContext = null;

    /** support for local context scope feature, where all references are local */
    private  boolean localcontextscope = false;

     /**
     *  CTOR, wraps an ICA
     * @param inner
     * @param rsvc
     */
    public VMContext( InternalContextAdapter  inner, RuntimeServices rsvc )
    {
        localcontextscope = rsvc.getBoolean( RuntimeConstants.VM_CONTEXT_LOCALSCOPE, false );

        wrappedContext = inner;
        innerContext = inner.getBaseContext();
    }

    /**
     *  Return the inner / user context.
     * @return The inner / user context.
     */
    public Context getInternalUserContext()
    {
        return innerContext.getInternalUserContext();
    }

    /**
     * @see org.apache.velocity.context.InternalWrapperContext#getBaseContext()
     */
    public InternalContextAdapter getBaseContext()
    {
        return innerContext.getBaseContext();
    }

    /**
     *  Used to put VMProxyArgs into this context.  It separates
     *  the VMProxyArgs into constant and non-constant types
     *  pulling out the value of the constant types so they can
     *  be modified w/o damaging the VMProxyArg, and leaving the
     *  dynamic ones, as they modify context rather than their own
     *  state
     *  @param  vmpa VMProxyArg to add
     * @throws MethodInvocationException
     */
    public void addVMProxyArg(  VMProxyArg vmpa ) throws MethodInvocationException
    {
        /*
         *  ask if it's a constant : if so, get the value and put into the
         *  local context, otherwise, put the vmpa in our vmproxyhash
         */

        String key = vmpa.getContextReference();

        if ( vmpa.isConstant() )
        {
            localcontext.put( key, vmpa.getObject( wrappedContext ) );
        }
        else
        {
            vmproxyhash.put( key, vmpa );
        }
    }

    /**
     *  Impl of the Context.put() method.
     *
     *  @param key name of item to set
     *  @param value object to set to key
     *  @return old stored object
     */
    public Object put(final String key, final Object value)
    {
        return put(key, value, localcontextscope);
    }

    /**
     * Allows callers to explicitly put objects in the local context,
     * no matter what the velocimacro.context.local setting says. Needed
     * e.g. for loop variables in foreach.
     *
     *  @param key name of item to set.
     *  @param value object to set to key.
     *  @return old stored object
     */
    public Object localPut(final String key, final Object value)
    {
        return put(key, value, true);
    }

    /**
     *  Internal put method to select between local and global scope.
     *
     *  @param key name of item to set
     *  @param value object to set to key
     *  @param forceLocal True forces the object into the local scope.
     *  @return old stored object
     */
    protected Object put(final String key, final Object value, final boolean forceLocal)
    {
        /*
         *  first see if this is a vmpa
         */

        VMProxyArg vmpa = (VMProxyArg) vmproxyhash.get( key );

        if( vmpa != null)
        {
            return vmpa.setObject( wrappedContext, value );
        }
        else
        {
            if(forceLocal)
            {
                /*
                 *  just put in the local context
                 */
                return localcontext.put(key, value);
            }
            else
            {
                /*
                 *  ok, how about the local context?
                 */

                if (localcontext.containsKey(key))
                {
                    return localcontext.put(key, value);
                }
                else
                {
                    /*
                     * otherwise, let them push it into the 'global' context
                     */

                    return innerContext.put(key, value);
                }
            }
        }
    }

    /**
     *  Impl of the Context.gut() method.
     *
     *  @param key name of item to get
     *  @return  stored object or null
     */
    public Object get( String key )
    {
        /*
         * first, see if it's a VMPA
         */

        Object o = null;

        VMProxyArg vmpa = (VMProxyArg) vmproxyhash.get( key );

        if( vmpa != null )
        {
            o = vmpa.getObject( wrappedContext );
        }
        else
        {
            /*
             *  always try the local context then innerContext--even if  localcontextscope
             */

            o = localcontext.get( key );

            if ( o == null)
            {
                /*
                 * last chance
                 */

                o = innerContext.get( key );
            }
        }

        return o;
    }

    /**
     * @see org.apache.velocity.context.Context#containsKey(java.lang.Object)
     */
    public boolean containsKey(Object key)
    {
        return false;
    }

    /**
     * @see org.apache.velocity.context.Context#getKeys()
     */
    public Object[] getKeys()
    {
        return vmproxyhash.keySet().toArray();
    }

    /**
     * @see org.apache.velocity.context.Context#remove(java.lang.Object)
     */
    public Object remove(Object key)
    {
        return vmproxyhash.remove( key );
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#pushCurrentTemplateName(java.lang.String)
     */
    public void pushCurrentTemplateName( String s )
    {
        innerContext.pushCurrentTemplateName( s );
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#popCurrentTemplateName()
     */
    public void popCurrentTemplateName()
    {
        innerContext.popCurrentTemplateName();
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#getCurrentTemplateName()
     */
    public String getCurrentTemplateName()
    {
        return innerContext.getCurrentTemplateName();
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#getTemplateNameStack()
     */
    public Object[] getTemplateNameStack()
    {
        return innerContext.getTemplateNameStack();
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#icacheGet(java.lang.Object)
     */
    public IntrospectionCacheData icacheGet( Object key )
    {
        return innerContext.icacheGet( key );
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#icachePut(java.lang.Object, org.apache.velocity.util.introspection.IntrospectionCacheData)
     */
    public void icachePut( Object key, IntrospectionCacheData o )
    {
        innerContext.icachePut( key, o );
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#getAllowRendering()
     */
    public boolean getAllowRendering()
    {
       return innerContext.getAllowRendering();
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#setAllowRendering(boolean)
     */
    public void setAllowRendering(boolean v)
    {
        innerContext.setAllowRendering(v);
    }

    /**
     * @see org.apache.velocity.context.InternalEventContext#attachEventCartridge(org.apache.velocity.app.event.EventCartridge)
     */
    public EventCartridge attachEventCartridge( EventCartridge ec )
    {
        EventCartridge cartridge = innerContext.attachEventCartridge( ec );
        return cartridge;
    }

    /**
     * @see org.apache.velocity.context.InternalEventContext#getEventCartridge()
     */
    public EventCartridge getEventCartridge()
    {
        return innerContext.getEventCartridge();
    }


    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#setCurrentResource(org.apache.velocity.runtime.resource.Resource)
     */
    public void setCurrentResource( Resource r )
    {
        innerContext.setCurrentResource( r );
    }

    /**
     * @see org.apache.velocity.context.InternalHousekeepingContext#getCurrentResource()
     */
    public Resource getCurrentResource()
    {
        return innerContext.getCurrentResource();
    }
}



