/*
 * Decompiled with CFR 0.152.
 */
package org.jfaster.mango.transaction;

import java.sql.Connection;
import java.sql.SQLException;
import javax.sql.DataSource;
import org.jfaster.mango.transaction.ConnectionHolder;
import org.jfaster.mango.transaction.DataSourceMonitor;
import org.jfaster.mango.transaction.DataSourceUtils;
import org.jfaster.mango.transaction.Transaction;
import org.jfaster.mango.transaction.TransactionSynchronizationManager;
import org.jfaster.mango.transaction.exception.IllegalTransactionStateException;
import org.jfaster.mango.transaction.exception.TransactionSystemException;
import org.jfaster.mango.util.logging.InternalLogger;
import org.jfaster.mango.util.logging.InternalLoggerFactory;

public class TransactionImpl
implements Transaction {
    private static final InternalLogger logger = InternalLoggerFactory.getInstance(TransactionImpl.class);
    private final boolean newTransaction;
    private final DataSource dataSource;
    private final Integer previousLevel;
    private final boolean mustRestoreAutoCommit;
    private boolean completed = false;
    private boolean rollbackOnly = false;

    public TransactionImpl(boolean newTransaction, DataSource dataSource) {
        this(newTransaction, dataSource, null, true);
    }

    public TransactionImpl(boolean newTransaction, DataSource dataSource, Integer previousLevel, boolean mustRestoreAutoCommit) {
        this.newTransaction = newTransaction;
        this.dataSource = dataSource;
        this.previousLevel = previousLevel;
        this.mustRestoreAutoCommit = mustRestoreAutoCommit;
    }

    @Override
    public void commit() {
        if (this.completed) {
            throw new IllegalTransactionStateException("Transaction is already completed - do not call commit or rollback more than once per transaction");
        }
        ConnectionHolder connHolder = TransactionSynchronizationManager.getConnectionHolder(this.dataSource);
        if (connHolder == null) {
            throw new IllegalStateException("No ConnectionHolder bind to DataSource [" + this.dataSource + "]");
        }
        if (!this.newTransaction) {
            if (logger.isDebugEnabled()) {
                logger.debug("Commit transaction is not new");
            }
            if (this.rollbackOnly) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Marking transaction as rollback-only");
                }
                connHolder.setRollbackOnly(true);
            }
            return;
        }
        if (this.rollbackOnly || connHolder.isRollbackOnly()) {
            if (logger.isDebugEnabled()) {
                logger.debug("Transaction is marked as rollback-only, so will rollback");
            }
            this.processRollback(connHolder.getConnection());
            return;
        }
        this.processCommit(connHolder.getConnection());
    }

    @Override
    public void rollback() {
        if (this.completed) {
            throw new IllegalTransactionStateException("Transaction is already completed - do not call commit or rollback more than once per transaction");
        }
        ConnectionHolder connHolder = TransactionSynchronizationManager.getConnectionHolder(this.dataSource);
        if (connHolder == null) {
            throw new IllegalStateException("No ConnectionHolder bind to DataSource [" + this.dataSource + "]");
        }
        if (!this.newTransaction) {
            if (logger.isDebugEnabled()) {
                logger.debug("Rollback transaction is not new, marking transaction as rollback-only");
            }
            connHolder.setRollbackOnly(true);
            return;
        }
        this.processRollback(connHolder.getConnection());
    }

    @Override
    public boolean isRollbackOnly() {
        return this.rollbackOnly;
    }

    @Override
    public void setRollbackOnly(boolean rollbackOnly) {
        this.rollbackOnly = rollbackOnly;
    }

    private void processCommit(Connection conn) {
        try {
            this.doCommit(conn);
        }
        catch (Exception e) {
            this.doRollbackOnCommitException(conn, e);
        }
        finally {
            this.cleanup(conn);
        }
    }

    private void doCommit(Connection conn) {
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("Committing JDBC transaction on Connection [" + conn + "]");
            }
            conn.commit();
        }
        catch (SQLException e) {
            throw new TransactionSystemException("Could not commit JDBC transaction", e);
        }
    }

    private void doRollbackOnCommitException(Connection conn, Exception e) {
        try {
            this.doRollback(conn);
        }
        catch (TransactionSystemException tes) {
            logger.error("Commit exception overridden by rollback exception", e);
            throw tes;
        }
    }

    private void processRollback(Connection conn) {
        try {
            this.doRollback(conn);
        }
        finally {
            this.cleanup(conn);
        }
    }

    private void doRollback(Connection conn) {
        try {
            if (logger.isDebugEnabled()) {
                logger.debug("Rolling back JDBC transaction on Connection [" + conn + "]");
            }
            conn.rollback();
        }
        catch (SQLException e) {
            throw new TransactionSystemException("Could not roll back JDBC transaction", e);
        }
    }

    private void cleanup(Connection conn) {
        TransactionSynchronizationManager.unbindConnectionHolder(this.dataSource);
        this.resetConnectionAfterTransaction(conn);
        DataSourceUtils.releaseConnection(conn, this.dataSource);
        this.completed = true;
    }

    private void resetConnectionAfterTransaction(Connection conn) {
        try {
            if (this.mustRestoreAutoCommit) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Switching JDBC Connection to auto commit");
                }
                conn.setAutoCommit(true);
            }
        }
        catch (SQLException e) {
            DataSourceMonitor.resetAutoCommitFail(this.dataSource);
            logger.error("Could not reset autoCommit of JDBC Connection after transaction", e);
        }
        catch (Throwable e) {
            DataSourceMonitor.resetAutoCommitFail(this.dataSource);
            logger.error("Unexpected exception on resetting autoCommit of JDBC Connection after transaction", e);
        }
        try {
            if (this.previousLevel != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Resetting isolation level of JDBC Connection to " + this.previousLevel);
                }
                conn.setTransactionIsolation(this.previousLevel);
            }
        }
        catch (SQLException e) {
            logger.error("Could not reset isolation level of JDBC Connection after transaction", e);
        }
        catch (Throwable e) {
            logger.error("Unexpected exception on resetting isolation level of JDBC Connection after transaction", e);
        }
    }
}

