/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ConnectionDescription;
import com.mongodb.connection.ConnectionId;
import com.mongodb.connection.ConnectionPoolSettings;
import com.mongodb.connection.ServerDescription;
import com.mongodb.connection.ServerId;
import com.mongodb.diagnostics.logging.Logger;
import com.mongodb.diagnostics.logging.Loggers;
import com.mongodb.event.ConnectionAddedEvent;
import com.mongodb.event.ConnectionCheckOutFailedEvent;
import com.mongodb.event.ConnectionCheckOutStartedEvent;
import com.mongodb.event.ConnectionCheckedInEvent;
import com.mongodb.event.ConnectionCheckedOutEvent;
import com.mongodb.event.ConnectionClosedEvent;
import com.mongodb.event.ConnectionCreatedEvent;
import com.mongodb.event.ConnectionPoolClearedEvent;
import com.mongodb.event.ConnectionPoolClosedEvent;
import com.mongodb.event.ConnectionPoolCreatedEvent;
import com.mongodb.event.ConnectionPoolListener;
import com.mongodb.event.ConnectionPoolOpenedEvent;
import com.mongodb.event.ConnectionReadyEvent;
import com.mongodb.event.ConnectionRemovedEvent;
import com.mongodb.internal.async.ErrorHandlingResultCallback;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.connection.CommandMessage;
import com.mongodb.internal.connection.ConcurrentPool;
import com.mongodb.internal.connection.ConnectionPool;
import com.mongodb.internal.connection.InternalConnection;
import com.mongodb.internal.connection.InternalConnectionFactory;
import com.mongodb.internal.connection.ResponseBuffers;
import com.mongodb.internal.connection.UsageTrackingInternalConnection;
import com.mongodb.internal.event.EventListenerHelper;
import com.mongodb.internal.session.SessionContext;
import com.mongodb.internal.thread.DaemonThreadFactory;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.bson.ByteBuf;
import org.bson.codecs.Decoder;

class DefaultConnectionPool
implements ConnectionPool {
    private static final Logger LOGGER = Loggers.getLogger("connection");
    private final ConcurrentPool<UsageTrackingInternalConnection> pool;
    private final ConnectionPoolSettings settings;
    private final AtomicInteger generation = new AtomicInteger(0);
    private final AtomicInteger lastPrunedGeneration = new AtomicInteger(0);
    private final ScheduledExecutorService sizeMaintenanceTimer;
    private ExecutorService asyncGetter;
    private final Runnable maintenanceTask;
    private final ConnectionPoolListener connectionPoolListener;
    private final ServerId serverId;
    private volatile boolean closed;

    DefaultConnectionPool(ServerId serverId, InternalConnectionFactory internalConnectionFactory, ConnectionPoolSettings settings) {
        this.serverId = Assertions.notNull("serverId", serverId);
        this.settings = Assertions.notNull("settings", settings);
        UsageTrackingInternalConnectionItemFactory connectionItemFactory = new UsageTrackingInternalConnectionItemFactory(internalConnectionFactory);
        this.pool = new ConcurrentPool<UsageTrackingInternalConnection>(settings.getMaxSize(), connectionItemFactory);
        this.connectionPoolListener = EventListenerHelper.getConnectionPoolListener(settings);
        this.maintenanceTask = this.createMaintenanceTask();
        this.sizeMaintenanceTimer = this.createMaintenanceTimer();
        this.connectionPoolCreated(this.connectionPoolListener, serverId, settings);
    }

    @Override
    public void start() {
        if (this.sizeMaintenanceTimer != null) {
            this.sizeMaintenanceTimer.scheduleAtFixedRate(this.maintenanceTask, this.settings.getMaintenanceInitialDelay(TimeUnit.MILLISECONDS), this.settings.getMaintenanceFrequency(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
        }
    }

    @Override
    public InternalConnection get() {
        return this.get(this.settings.getMaxWaitTime(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
    }

    @Override
    public InternalConnection get(long timeout, TimeUnit timeUnit) {
        PooledConnection pooledConnection;
        try {
            this.connectionPoolListener.connectionCheckOutStarted(new ConnectionCheckOutStartedEvent(this.serverId));
            pooledConnection = this.getPooledConnection(timeout, timeUnit);
        }
        catch (Throwable t) {
            this.emitCheckOutFailedEvent(t);
            throw t;
        }
        if (!pooledConnection.opened()) {
            try {
                pooledConnection.open();
            }
            catch (Throwable t) {
                this.pool.release(pooledConnection.wrapped, true);
                this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR));
                throw t;
            }
        }
        this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(pooledConnection.getDescription().getConnectionId()));
        return pooledConnection;
    }

    @Override
    public void getAsync(SingleResultCallback<InternalConnection> callback) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(String.format("Asynchronously getting a connection from the pool for server %s", this.serverId));
        }
        final SingleResultCallback<InternalConnection> errHandlingCallback = ErrorHandlingResultCallback.errorHandlingCallback(callback, LOGGER);
        PooledConnection connection = null;
        try {
            this.connectionPoolListener.connectionCheckOutStarted(new ConnectionCheckOutStartedEvent(this.serverId));
            connection = this.getPooledConnection(0L, TimeUnit.MILLISECONDS);
        }
        catch (MongoTimeoutException mongoTimeoutException) {
        }
        catch (Throwable t) {
            this.emitCheckOutFailedEvent(t);
            callback.onResult(null, t);
            return;
        }
        if (connection != null) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace(String.format("Asynchronously opening pooled connection %s to server %s", connection.getDescription().getConnectionId(), this.serverId));
            }
            this.openAsync(connection, errHandlingCallback);
        } else {
            final long startTimeMillis = System.currentTimeMillis();
            this.getAsyncGetter().submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        if (this.getRemainingWaitTime() <= 0L) {
                            errHandlingCallback.onResult(null, DefaultConnectionPool.this.createTimeoutException());
                        } else {
                            PooledConnection connection = DefaultConnectionPool.this.getPooledConnection(this.getRemainingWaitTime(), TimeUnit.MILLISECONDS);
                            DefaultConnectionPool.this.openAsync(connection, errHandlingCallback);
                        }
                    }
                    catch (MongoTimeoutException e) {
                        MongoTimeoutException exception = new MongoTimeoutException(String.format("Timeout waiting for a pooled connection after %d %s", new Object[]{DefaultConnectionPool.this.settings.getMaxWaitTime(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS}));
                        DefaultConnectionPool.this.emitCheckOutFailedEvent(exception);
                        errHandlingCallback.onResult(null, exception);
                    }
                    catch (Throwable t) {
                        DefaultConnectionPool.this.emitCheckOutFailedEvent(t);
                        errHandlingCallback.onResult(null, t);
                    }
                }

                private long getRemainingWaitTime() {
                    return startTimeMillis + DefaultConnectionPool.this.settings.getMaxWaitTime(TimeUnit.MILLISECONDS) - System.currentTimeMillis();
                }
            });
        }
    }

    private void emitCheckOutFailedEvent(Throwable t) {
        if (t instanceof MongoTimeoutException) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.TIMEOUT));
        } else if (t instanceof IllegalStateException && t.getMessage().equals("The pool is closed")) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.POOL_CLOSED));
        } else {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.UNKNOWN));
        }
    }

    private void openAsync(final PooledConnection pooledConnection, final SingleResultCallback<InternalConnection> callback) {
        if (pooledConnection.opened()) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace(String.format("Pooled connection %s to server %s is already open", pooledConnection.getDescription().getConnectionId(), this.serverId));
            }
            this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(pooledConnection.getDescription().getConnectionId()));
            callback.onResult(pooledConnection, null);
        } else {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace(String.format("Pooled connection %s to server %s is not yet open", pooledConnection.getDescription().getConnectionId(), this.serverId));
            }
            pooledConnection.openAsync(new SingleResultCallback<Void>(){

                @Override
                public void onResult(Void result, Throwable t) {
                    if (t != null) {
                        if (LOGGER.isTraceEnabled()) {
                            LOGGER.trace(String.format("Pooled connection %s to server %s failed to open", pooledConnection.getDescription().getConnectionId(), DefaultConnectionPool.this.serverId));
                        }
                        DefaultConnectionPool.this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(DefaultConnectionPool.this.serverId, ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR));
                        callback.onResult(null, t);
                        DefaultConnectionPool.this.pool.release(pooledConnection.wrapped, true);
                    } else {
                        if (LOGGER.isTraceEnabled()) {
                            LOGGER.trace(String.format("Pooled connection %s to server %s is now open", pooledConnection.getDescription().getConnectionId(), DefaultConnectionPool.this.serverId));
                        }
                        DefaultConnectionPool.this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(pooledConnection.getDescription().getConnectionId()));
                        callback.onResult(pooledConnection, null);
                    }
                }
            });
        }
    }

    private synchronized ExecutorService getAsyncGetter() {
        if (this.asyncGetter == null) {
            this.asyncGetter = Executors.newSingleThreadExecutor(new DaemonThreadFactory("AsyncGetter"));
        }
        return this.asyncGetter;
    }

    private synchronized void shutdownAsyncGetter() {
        if (this.asyncGetter != null) {
            this.asyncGetter.shutdownNow();
        }
    }

    @Override
    public void invalidate() {
        LOGGER.debug("Invalidating the connection pool");
        this.generation.incrementAndGet();
        this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(this.serverId));
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.pool.close();
            if (this.sizeMaintenanceTimer != null) {
                this.sizeMaintenanceTimer.shutdownNow();
            }
            this.shutdownAsyncGetter();
            this.closed = true;
            this.connectionPoolListener.connectionPoolClosed(new ConnectionPoolClosedEvent(this.serverId));
        }
    }

    @Override
    public int getGeneration() {
        return this.generation.get();
    }

    public void doMaintenance() {
        if (this.maintenanceTask != null) {
            this.maintenanceTask.run();
        }
    }

    private PooledConnection getPooledConnection(long timeout, TimeUnit timeUnit) {
        UsageTrackingInternalConnection internalConnection = this.pool.get(timeout, timeUnit);
        while (this.shouldPrune(internalConnection)) {
            this.pool.release(internalConnection, true);
            internalConnection = this.pool.get(timeout, timeUnit);
        }
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(String.format("Checked out connection [%s] to server %s", this.getId(internalConnection), this.serverId.getAddress()));
        }
        return new PooledConnection(internalConnection);
    }

    private MongoTimeoutException createTimeoutException() {
        return new MongoTimeoutException(String.format("Timed out after %d ms while waiting for a connection to server %s.", this.settings.getMaxWaitTime(TimeUnit.MILLISECONDS), this.serverId.getAddress()));
    }

    ConcurrentPool<UsageTrackingInternalConnection> getPool() {
        return this.pool;
    }

    private Runnable createMaintenanceTask() {
        Runnable newMaintenanceTask = null;
        if (this.shouldPrune() || this.shouldEnsureMinSize()) {
            newMaintenanceTask = new Runnable(){

                @Override
                public synchronized void run() {
                    try {
                        int curGeneration = DefaultConnectionPool.this.generation.get();
                        if (DefaultConnectionPool.this.shouldPrune() || curGeneration > DefaultConnectionPool.this.lastPrunedGeneration.get()) {
                            if (LOGGER.isDebugEnabled()) {
                                LOGGER.debug(String.format("Pruning pooled connections to %s", DefaultConnectionPool.this.serverId.getAddress()));
                            }
                            DefaultConnectionPool.this.pool.prune();
                        }
                        DefaultConnectionPool.this.lastPrunedGeneration.set(curGeneration);
                        if (DefaultConnectionPool.this.shouldEnsureMinSize()) {
                            if (LOGGER.isDebugEnabled()) {
                                LOGGER.debug(String.format("Ensuring minimum pooled connections to %s", DefaultConnectionPool.this.serverId.getAddress()));
                            }
                            DefaultConnectionPool.this.pool.ensureMinSize(DefaultConnectionPool.this.settings.getMinSize(), true);
                        }
                    }
                    catch (MongoInterruptedException curGeneration) {
                    }
                    catch (Exception e) {
                        LOGGER.warn("Exception thrown during connection pool background maintenance task", e);
                    }
                }
            };
        }
        return newMaintenanceTask;
    }

    private ScheduledExecutorService createMaintenanceTimer() {
        if (this.maintenanceTask == null) {
            return null;
        }
        return Executors.newSingleThreadScheduledExecutor(new DaemonThreadFactory("MaintenanceTimer"));
    }

    private boolean shouldEnsureMinSize() {
        return this.settings.getMinSize() > 0;
    }

    private boolean shouldPrune() {
        return this.settings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS) > 0L || this.settings.getMaxConnectionLifeTime(TimeUnit.MILLISECONDS) > 0L;
    }

    private boolean shouldPrune(UsageTrackingInternalConnection connection) {
        return this.fromPreviousGeneration(connection) || this.pastMaxLifeTime(connection) || this.pastMaxIdleTime(connection);
    }

    private boolean pastMaxIdleTime(UsageTrackingInternalConnection connection) {
        return this.expired(connection.getLastUsedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS));
    }

    private boolean pastMaxLifeTime(UsageTrackingInternalConnection connection) {
        return this.expired(connection.getOpenedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionLifeTime(TimeUnit.MILLISECONDS));
    }

    private boolean fromPreviousGeneration(UsageTrackingInternalConnection connection) {
        return this.generation.get() > connection.getGeneration();
    }

    private boolean expired(long startTime, long curTime, long maxTime) {
        return maxTime != 0L && curTime - startTime > maxTime;
    }

    private void connectionPoolCreated(ConnectionPoolListener connectionPoolListener, ServerId serverId, ConnectionPoolSettings settings) {
        connectionPoolListener.connectionPoolCreated(new ConnectionPoolCreatedEvent(serverId, settings));
        connectionPoolListener.connectionPoolOpened(new ConnectionPoolOpenedEvent(serverId, settings));
    }

    private void connectionCreated(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId) {
        connectionPoolListener.connectionAdded(new ConnectionAddedEvent(connectionId));
        connectionPoolListener.connectionCreated(new ConnectionCreatedEvent(connectionId));
    }

    private void connectionClosed(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId, ConnectionClosedEvent.Reason reason) {
        connectionPoolListener.connectionRemoved(new ConnectionRemovedEvent(connectionId, this.getReasonForRemoved(reason)));
        connectionPoolListener.connectionClosed(new ConnectionClosedEvent(connectionId, reason));
    }

    private ConnectionRemovedEvent.Reason getReasonForRemoved(ConnectionClosedEvent.Reason reason) {
        ConnectionRemovedEvent.Reason removedReason = ConnectionRemovedEvent.Reason.UNKNOWN;
        switch (reason) {
            case STALE: {
                removedReason = ConnectionRemovedEvent.Reason.STALE;
                break;
            }
            case IDLE: {
                removedReason = ConnectionRemovedEvent.Reason.MAX_IDLE_TIME_EXCEEDED;
                break;
            }
            case ERROR: {
                removedReason = ConnectionRemovedEvent.Reason.ERROR;
                break;
            }
            case POOL_CLOSED: {
                removedReason = ConnectionRemovedEvent.Reason.POOL_CLOSED;
                break;
            }
        }
        return removedReason;
    }

    private ConnectionId getId(InternalConnection internalConnection) {
        return internalConnection.getDescription().getConnectionId();
    }

    private class UsageTrackingInternalConnectionItemFactory
    implements ConcurrentPool.ItemFactory<UsageTrackingInternalConnection> {
        private final InternalConnectionFactory internalConnectionFactory;

        UsageTrackingInternalConnectionItemFactory(InternalConnectionFactory internalConnectionFactory) {
            this.internalConnectionFactory = internalConnectionFactory;
        }

        @Override
        public UsageTrackingInternalConnection create(boolean initialize) {
            UsageTrackingInternalConnection internalConnection = new UsageTrackingInternalConnection(this.internalConnectionFactory.create(DefaultConnectionPool.this.serverId), DefaultConnectionPool.this.generation.get());
            if (initialize) {
                internalConnection.open();
            }
            DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, DefaultConnectionPool.this.getId(internalConnection));
            return internalConnection;
        }

        @Override
        public void close(UsageTrackingInternalConnection connection) {
            DefaultConnectionPool.this.connectionClosed(DefaultConnectionPool.this.connectionPoolListener, DefaultConnectionPool.this.getId(connection), this.getReasonForClosing(connection));
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info(String.format("Closed connection [%s] to %s because %s.", DefaultConnectionPool.this.getId(connection), DefaultConnectionPool.this.serverId.getAddress(), this.getReasonStringForClosing(connection)));
            }
            connection.close();
        }

        private String getReasonStringForClosing(UsageTrackingInternalConnection connection) {
            String reason = connection.isClosed() ? "there was a socket exception raised by this connection" : (DefaultConnectionPool.this.fromPreviousGeneration(connection) ? "there was a socket exception raised on another connection from this pool" : (DefaultConnectionPool.this.pastMaxLifeTime(connection) ? "it is past its maximum allowed life time" : (DefaultConnectionPool.this.pastMaxIdleTime(connection) ? "it is past its maximum allowed idle time" : "the pool has been closed")));
            return reason;
        }

        private ConnectionClosedEvent.Reason getReasonForClosing(UsageTrackingInternalConnection connection) {
            ConnectionClosedEvent.Reason reason = connection.isClosed() ? ConnectionClosedEvent.Reason.ERROR : (DefaultConnectionPool.this.fromPreviousGeneration(connection) ? ConnectionClosedEvent.Reason.STALE : (DefaultConnectionPool.this.pastMaxIdleTime(connection) ? ConnectionClosedEvent.Reason.IDLE : ConnectionClosedEvent.Reason.POOL_CLOSED));
            return reason;
        }

        @Override
        public ConcurrentPool.Prune shouldPrune(UsageTrackingInternalConnection usageTrackingConnection) {
            return DefaultConnectionPool.this.shouldPrune(usageTrackingConnection) ? ConcurrentPool.Prune.YES : ConcurrentPool.Prune.NO;
        }
    }

    private class PooledConnection
    implements InternalConnection {
        private final UsageTrackingInternalConnection wrapped;
        private final AtomicBoolean isClosed = new AtomicBoolean();

        PooledConnection(UsageTrackingInternalConnection wrapped) {
            this.wrapped = Assertions.notNull("wrapped", wrapped);
        }

        @Override
        public int getGeneration() {
            return this.wrapped.getGeneration();
        }

        @Override
        public void open() {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.open();
            DefaultConnectionPool.this.connectionPoolListener.connectionReady(new ConnectionReadyEvent(this.getDescription().getConnectionId()));
        }

        @Override
        public void openAsync(final SingleResultCallback<Void> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.openAsync(new SingleResultCallback<Void>(){

                @Override
                public void onResult(Void result, Throwable t) {
                    if (t != null) {
                        DefaultConnectionPool.this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(DefaultConnectionPool.this.serverId, ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR));
                        callback.onResult(null, t);
                    } else {
                        DefaultConnectionPool.this.connectionPoolListener.connectionReady(new ConnectionReadyEvent(PooledConnection.this.getDescription().getConnectionId()));
                        callback.onResult(result, null);
                    }
                }
            });
        }

        @Override
        public void close() {
            if (!this.isClosed.getAndSet(true)) {
                DefaultConnectionPool.this.connectionPoolListener.connectionCheckedIn(new ConnectionCheckedInEvent(DefaultConnectionPool.this.getId(this.wrapped)));
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Checked in connection [%s] to server %s", DefaultConnectionPool.this.getId(this.wrapped), DefaultConnectionPool.this.serverId.getAddress()));
                }
                DefaultConnectionPool.this.pool.release(this.wrapped, this.wrapped.isClosed() || DefaultConnectionPool.this.shouldPrune(this.wrapped));
            }
        }

        @Override
        public boolean opened() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.opened();
        }

        @Override
        public boolean isClosed() {
            return this.isClosed.get() || this.wrapped.isClosed();
        }

        @Override
        public ByteBuf getBuffer(int capacity) {
            return this.wrapped.getBuffer(capacity);
        }

        @Override
        public void sendMessage(List<ByteBuf> byteBuffers, int lastRequestId) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessage(byteBuffers, lastRequestId);
        }

        @Override
        public <T> T sendAndReceive(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.sendAndReceive(message, decoder, sessionContext);
        }

        @Override
        public <T> void send(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.send(message, decoder, sessionContext);
        }

        @Override
        public <T> T receive(Decoder<T> decoder, SessionContext sessionContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receive(decoder, sessionContext);
        }

        @Override
        public boolean supportsAdditionalTimeout() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.supportsAdditionalTimeout();
        }

        @Override
        public <T> T receive(Decoder<T> decoder, SessionContext sessionContext, int additionalTimeout) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receive(decoder, sessionContext, additionalTimeout);
        }

        @Override
        public boolean hasMoreToCome() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.hasMoreToCome();
        }

        @Override
        public <T> void sendAndReceiveAsync(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext, final SingleResultCallback<T> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendAndReceiveAsync(message, decoder, sessionContext, new SingleResultCallback<T>(){

                @Override
                public void onResult(T result, Throwable t) {
                    callback.onResult(result, t);
                }
            });
        }

        @Override
        public ResponseBuffers receiveMessage(int responseTo) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receiveMessage(responseTo);
        }

        @Override
        public void sendMessageAsync(List<ByteBuf> byteBuffers, int lastRequestId, final SingleResultCallback<Void> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessageAsync(byteBuffers, lastRequestId, new SingleResultCallback<Void>(){

                @Override
                public void onResult(Void result, Throwable t) {
                    callback.onResult(null, t);
                }
            });
        }

        @Override
        public void receiveMessageAsync(int responseTo, final SingleResultCallback<ResponseBuffers> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.receiveMessageAsync(responseTo, new SingleResultCallback<ResponseBuffers>(){

                @Override
                public void onResult(ResponseBuffers result, Throwable t) {
                    callback.onResult(result, t);
                }
            });
        }

        @Override
        public ConnectionDescription getDescription() {
            return this.wrapped.getDescription();
        }

        @Override
        public ServerDescription getInitialServerDescription() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.getInitialServerDescription();
        }
    }
}

