/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.xaframework;

import java.io.IOException;
import java.util.Arrays;
import java.util.concurrent.TimeUnit;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.nioneo.store.FileSystemAbstraction;
import org.neo4j.kernel.impl.transaction.xaframework.LogExtractor;
import org.neo4j.kernel.impl.transaction.xaframework.LogPruneStrategy;

public class LogPruneStrategies {
    public static final LogPruneStrategy NO_PRUNING = new LogPruneStrategy(){

        @Override
        public void prune(LogExtractor.LogLoader source) {
        }

        public String toString() {
            return "NO_PRUNING";
        }
    };

    public static LogPruneStrategy nonEmptyFileCount(FileSystemAbstraction fileSystem, int maxLogCountToKeep) {
        return new FileCountPruneStrategy(fileSystem, maxLogCountToKeep);
    }

    public static LogPruneStrategy totalFileSize(FileSystemAbstraction fileSystem, int numberOfBytes) {
        return new FileSizePruneStrategy(fileSystem, numberOfBytes);
    }

    public static LogPruneStrategy transactionCount(FileSystemAbstraction fileSystem, int maxCount) {
        return new TransactionCountPruneStrategy(fileSystem, maxCount);
    }

    public static LogPruneStrategy transactionTimeSpan(FileSystemAbstraction fileSystem, int timeToKeep, TimeUnit timeUnit) {
        return new TransactionTimeSpanPruneStrategy(fileSystem, timeToKeep, timeUnit);
    }

    public static LogPruneStrategy fromConfigValue(FileSystemAbstraction fileSystem, String configValue) {
        String[] tokens = configValue.split(" ");
        if (tokens.length == 0) {
            throw new IllegalArgumentException("Invalid log pruning configuration value '" + configValue + "'");
        }
        String numberWithUnit = tokens[0];
        if (tokens.length == 1) {
            if (numberWithUnit.equals("true")) {
                return NO_PRUNING;
            }
            if (numberWithUnit.equals("false")) {
                return LogPruneStrategies.transactionCount(fileSystem, 1);
            }
            throw new IllegalArgumentException("Invalid log pruning configuration value '" + configValue + "'. The form is 'all' or '<number><unit> <type>' for example '100k txs' " + "for the latest 100 000 transactions");
        }
        String[] types = new String[]{"files", "size", "txs", "hours", "days"};
        String type = tokens[1];
        int number = (int)Config.parseLongWithUnit(numberWithUnit);
        int typeIndex = 0;
        if (type.equals(types[typeIndex++])) {
            return LogPruneStrategies.nonEmptyFileCount(fileSystem, number);
        }
        if (type.equals(types[typeIndex++])) {
            return LogPruneStrategies.totalFileSize(fileSystem, number);
        }
        if (type.equals(types[typeIndex++])) {
            return LogPruneStrategies.transactionCount(fileSystem, number);
        }
        if (type.equals(types[typeIndex++])) {
            return LogPruneStrategies.transactionTimeSpan(fileSystem, number, TimeUnit.HOURS);
        }
        if (type.equals(types[typeIndex++])) {
            return LogPruneStrategies.transactionTimeSpan(fileSystem, number, TimeUnit.DAYS);
        }
        throw new IllegalArgumentException("Invalid log pruning configuration value '" + configValue + "'. Invalid type '" + type + "', valid are " + Arrays.asList(types));
    }

    public static class TransactionTimeSpanPruneStrategy
    extends AbstractPruneStrategy {
        private final int timeToKeep;
        private final TimeUnit unit;

        public TransactionTimeSpanPruneStrategy(FileSystemAbstraction fileSystem, int timeToKeep, TimeUnit unit) {
            super(fileSystem);
            this.timeToKeep = timeToKeep;
            this.unit = unit;
        }

        @Override
        protected Threshold newThreshold() {
            return new Threshold(){
                private long lowerLimit;
                {
                    this.lowerLimit = System.currentTimeMillis() - TransactionTimeSpanPruneStrategy.this.unit.toMillis(TransactionTimeSpanPruneStrategy.this.timeToKeep);
                }

                @Override
                public boolean reached(String file, long version, LogExtractor.LogLoader source) {
                    try {
                        return source.getFirstStartRecordTimestamp(version) < this.lowerLimit;
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
            };
        }
    }

    public static class TransactionCountPruneStrategy
    extends AbstractPruneStrategy {
        private final int maxTransactionCount;

        public TransactionCountPruneStrategy(FileSystemAbstraction fileSystem, int maxTransactionCount) {
            super(fileSystem);
            this.maxTransactionCount = maxTransactionCount;
        }

        @Override
        protected Threshold newThreshold() {
            return new Threshold(){
                private Long highest;

                @Override
                public boolean reached(String file, long version, LogExtractor.LogLoader source) {
                    long tx = source.getFirstCommittedTxId(version);
                    if (this.highest == null) {
                        this.highest = source.getLastCommittedTxId();
                        return false;
                    }
                    return this.highest - tx >= (long)TransactionCountPruneStrategy.this.maxTransactionCount;
                }
            };
        }
    }

    public static class FileSizePruneStrategy
    extends AbstractPruneStrategy {
        private final int maxSize;

        public FileSizePruneStrategy(FileSystemAbstraction fileystem, int maxSizeBytes) {
            super(fileystem);
            this.maxSize = maxSizeBytes;
        }

        @Override
        protected Threshold newThreshold() {
            return new Threshold(){
                private int size;

                @Override
                public boolean reached(String file, long version, LogExtractor.LogLoader source) {
                    this.size = (int)((long)this.size + FileSizePruneStrategy.this.fileSystem.getFileSize(file));
                    return this.size >= FileSizePruneStrategy.this.maxSize;
                }
            };
        }
    }

    private static class FileCountPruneStrategy
    extends AbstractPruneStrategy {
        private final int maxNonEmptyLogCount;

        public FileCountPruneStrategy(FileSystemAbstraction fileSystem, int maxNonEmptyLogCount) {
            super(fileSystem);
            this.maxNonEmptyLogCount = maxNonEmptyLogCount;
        }

        @Override
        protected Threshold newThreshold() {
            return new Threshold(){
                int nonEmptyLogCount = 0;

                @Override
                public boolean reached(String file, long version, LogExtractor.LogLoader source) {
                    return ++this.nonEmptyLogCount >= FileCountPruneStrategy.this.maxNonEmptyLogCount;
                }
            };
        }

        public String toString() {
            return this.getClass().getSimpleName() + "[max:" + this.maxNonEmptyLogCount + "]";
        }
    }

    private static abstract class AbstractPruneStrategy
    implements LogPruneStrategy {
        protected final FileSystemAbstraction fileSystem;

        AbstractPruneStrategy(FileSystemAbstraction fileSystem) {
            this.fileSystem = fileSystem;
        }

        @Override
        public void prune(LogExtractor.LogLoader source) {
            long upper;
            if (source.getHighestLogVersion() == 0L) {
                return;
            }
            Threshold threshold = this.newThreshold();
            boolean exceeded = false;
            for (upper = source.getHighestLogVersion() - 1L; upper >= 0L; --upper) {
                String file = source.getFileName(upper);
                if (!this.fileSystem.fileExists(file)) {
                    return;
                }
                if (this.fileSystem.getFileSize(file) <= 16L || !threshold.reached(file, upper, source)) continue;
                exceeded = true;
                break;
            }
            if (!exceeded) {
                return;
            }
            long lower = upper;
            while (this.fileSystem.fileExists(source.getFileName(lower - 1L))) {
                --lower;
            }
            for (long version = lower; version < upper; ++version) {
                this.fileSystem.deleteFile(source.getFileName(version));
            }
        }

        protected abstract Threshold newThreshold();
    }

    private static interface Threshold {
        public boolean reached(String var1, long var2, LogExtractor.LogLoader var4);
    }
}

