/*
 * Copyright [2007] [University Corporation for Advanced Internet Development, Inc.]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensaml.xml.schema.validator;

import org.opensaml.xml.schema.XSInteger;
import org.opensaml.xml.validation.ValidationException;
import org.opensaml.xml.validation.Validator;

/**
 * Checks {@link org.opensaml.xml.schema.XSInteger} for Schema compliance. 
 * 
 * @param <T> the type to be validated
 */
public class XSIntegerSchemaValidator<T extends XSInteger> implements Validator<T> {
    
    /** Flag specifying whether empty element content should be allowed. */
    private boolean allowEmptyContent;
    
    /**
     * Constructor.
     *
     * @param allowEmptyElementContent flag indicated whether empty content should be allowed
     */
    public XSIntegerSchemaValidator(boolean allowEmptyElementContent) {
        allowEmptyContent = allowEmptyElementContent;
    }
    
    /**
     * Constructor.
     * 
     * Empty content is not allowed.
     *
     */
    public XSIntegerSchemaValidator() {
        allowEmptyContent = false;
    }

    /** {@inheritDoc} */
    public void validate(T xmlObject) throws ValidationException {
        validateIntegerContent(xmlObject);
    }
    
    /**
     * Get the flag which determines whether empty content should be allowed.
     * 
     * @return true if empty content should be allowed, false otherwise
     */
    protected boolean isAllowEmptyContent() {
        return allowEmptyContent;
    }

    /**
     * Validates the content of the XSBase64Binary object.
     * 
     * @param xmlObject the object to evaluate
     * @throws ValidationException thrown if the content of the Base64Binary object is invalid
     */
    protected void validateIntegerContent(T xmlObject) throws ValidationException {
        if (! isAllowEmptyContent()) {
            if (xmlObject.getValue() == null) {
                throw new ValidationException("Integer content may not be empty");
            }
        }
    }

}
