/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.web.server.session;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import org.springframework.util.Assert;
import org.springframework.util.IdGenerator;
import org.springframework.util.JdkIdGenerator;
import org.springframework.web.server.WebSession;
import org.springframework.web.server.session.WebSessionStore;
import reactor.core.publisher.Mono;

public class InMemoryWebSessionStore
implements WebSessionStore {
    private static final Duration EXPIRATION_CHECK_PERIOD = Duration.ofSeconds(60L);
    private static final IdGenerator idGenerator = new JdkIdGenerator();
    private Clock clock = Clock.system(ZoneId.of("GMT"));
    private final ConcurrentMap<String, InMemoryWebSession> sessions = new ConcurrentHashMap<String, InMemoryWebSession>();
    private volatile Instant nextExpirationCheckTime = Instant.now(this.clock).plus(EXPIRATION_CHECK_PERIOD);
    private final ReentrantLock expirationCheckLock = new ReentrantLock();

    public void setClock(Clock clock) {
        Assert.notNull((Object)clock, (String)"Clock is required");
        this.clock = clock;
        this.nextExpirationCheckTime = Instant.now(this.clock);
    }

    public Clock getClock() {
        return this.clock;
    }

    @Override
    public Mono<WebSession> createWebSession() {
        return Mono.fromSupplier(() -> new InMemoryWebSession());
    }

    @Override
    public Mono<WebSession> retrieveSession(String id) {
        InMemoryWebSession session;
        Instant currentTime = Instant.now(this.clock);
        if (!this.sessions.isEmpty() && !currentTime.isBefore(this.nextExpirationCheckTime)) {
            this.checkExpiredSessions(currentTime);
        }
        if ((session = (InMemoryWebSession)this.sessions.get(id)) == null) {
            return Mono.empty();
        }
        if (session.isExpired(currentTime)) {
            this.sessions.remove(id);
            return Mono.empty();
        }
        session.updateLastAccessTime(currentTime);
        return Mono.just((Object)session);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void checkExpiredSessions(Instant currentTime) {
        if (this.expirationCheckLock.tryLock()) {
            try {
                Iterator iterator = this.sessions.values().iterator();
                while (iterator.hasNext()) {
                    InMemoryWebSession session = (InMemoryWebSession)iterator.next();
                    if (!session.isExpired(currentTime)) continue;
                    iterator.remove();
                    session.invalidate();
                }
            }
            finally {
                this.nextExpirationCheckTime = currentTime.plus(EXPIRATION_CHECK_PERIOD);
                this.expirationCheckLock.unlock();
            }
        }
    }

    @Override
    public Mono<Void> removeSession(String id) {
        this.sessions.remove(id);
        return Mono.empty();
    }

    @Override
    public Mono<WebSession> updateLastAccessTime(WebSession webSession) {
        return Mono.fromSupplier(() -> {
            Assert.isInstanceOf(InMemoryWebSession.class, (Object)webSession);
            InMemoryWebSession session = (InMemoryWebSession)webSession;
            session.updateLastAccessTime(Instant.now(this.getClock()));
            return session;
        });
    }

    private static enum State {
        NEW,
        STARTED,
        EXPIRED;

    }

    private class InMemoryWebSession
    implements WebSession {
        private final AtomicReference<String> id = new AtomicReference<String>(String.valueOf(InMemoryWebSessionStore.access$200().generateId()));
        private final Map<String, Object> attributes = new ConcurrentHashMap<String, Object>();
        private final Instant creationTime;
        private volatile Instant lastAccessTime;
        private volatile Duration maxIdleTime = Duration.ofMinutes(30L);
        private final AtomicReference<State> state = new AtomicReference<State>(State.NEW);

        public InMemoryWebSession() {
            this.lastAccessTime = this.creationTime = Instant.now(InMemoryWebSessionStore.this.getClock());
        }

        @Override
        public String getId() {
            return this.id.get();
        }

        @Override
        public Map<String, Object> getAttributes() {
            return this.attributes;
        }

        @Override
        public Instant getCreationTime() {
            return this.creationTime;
        }

        @Override
        public Instant getLastAccessTime() {
            return this.lastAccessTime;
        }

        @Override
        public void setMaxIdleTime(Duration maxIdleTime) {
            this.maxIdleTime = maxIdleTime;
        }

        @Override
        public Duration getMaxIdleTime() {
            return this.maxIdleTime;
        }

        @Override
        public void start() {
            this.state.compareAndSet(State.NEW, State.STARTED);
        }

        @Override
        public boolean isStarted() {
            return this.state.get().equals((Object)State.STARTED) || !this.getAttributes().isEmpty();
        }

        @Override
        public Mono<Void> changeSessionId() {
            String currentId = this.id.get();
            InMemoryWebSessionStore.this.sessions.remove(currentId);
            String newId = String.valueOf(idGenerator.generateId());
            this.id.set(newId);
            InMemoryWebSessionStore.this.sessions.put(this.getId(), this);
            return Mono.empty();
        }

        @Override
        public Mono<Void> invalidate() {
            this.state.set(State.EXPIRED);
            this.getAttributes().clear();
            InMemoryWebSessionStore.this.sessions.remove(this.id.get());
            return Mono.empty();
        }

        @Override
        public Mono<Void> save() {
            if (!this.getAttributes().isEmpty()) {
                this.state.compareAndSet(State.NEW, State.STARTED);
            }
            InMemoryWebSessionStore.this.sessions.put(this.getId(), this);
            return Mono.empty();
        }

        @Override
        public boolean isExpired() {
            return this.isExpired(Instant.now(InMemoryWebSessionStore.this.getClock()));
        }

        private boolean isExpired(Instant currentTime) {
            if (this.state.get().equals((Object)State.EXPIRED)) {
                return true;
            }
            if (this.checkExpired(currentTime)) {
                this.state.set(State.EXPIRED);
                return true;
            }
            return false;
        }

        private boolean checkExpired(Instant currentTime) {
            return this.isStarted() && !this.maxIdleTime.isNegative() && currentTime.minus(this.maxIdleTime).isAfter(this.lastAccessTime);
        }

        private void updateLastAccessTime(Instant currentTime) {
            this.lastAccessTime = currentTime;
        }
    }
}

