package localstore.service.utils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.jdbc.BadSqlGrammarException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.support.rowset.SqlRowSet;
import redis.clients.jedis.exceptions.JedisDataException;
import utils.collection.CollectionUtil;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author torvalds on 2018/8/6 15:00.
 * @version 1.0
 */
public class SequenceUtil implements InitializingBean, DisposableBean {
    private final String ORDERNO_REDIS_KEY = "mall:orderNo";
    private final String CREATE_SEQUECENCE_TABLE_SQL = "create table   mall_sequence_table(sequenceKey varchar(255) ,sequence number)";
    private final String INIT_SEQUECENCE_TABLE_SQL = "insert into mall_sequence_table  (sequenceKey,sequence) values(?,?)";
//    private final String INIT_CIRCULATE_UPDATE_SEQUECENCE_TABLE_SQL = "update  mall_sequence_table  set sequence=0 where sequenceKey=?";
    private final String QUERY_SEQUECENCE_TABLE_SQL = "select sequence from  mall_sequence_table where sequenceKey=? ";
    private final String QUERY_MAX_SEQUECENCE_TABLE_SQL = "select max(sequence) from  mall_sequence_table where sequenceKey=? ";
    private final String QUERY_GROUP_SEQUECENCE_TABLE_SQL = "select sequenceKey  from  mall_sequence_table group by sequenceKey having count(*)>1";
    private final String UPDATE_SEQUECENCE_TABLE_SQL = "update  mall_sequence_table  set sequence=? where (sequence<? or sequence>=?) and sequenceKey=?";
    private final String DELETE_SEQUECENCE_TABLE_SQL = "delete from mall_sequence_table where sequenceKey=?";
    private final String DROP_SEQUECENCE_TABLE_SQL = "drop table mall_sequence_table";
    private Logger logger = LoggerFactory.getLogger(getClass());
    ExecutorService executorService = null;
    RedisTemplate redisTemplate;
    JdbcTemplate jdbcTemplate;
    public Long maxCycle = 1000000L;

    public SequenceUtil(RedisTemplate redisTemplate, JdbcTemplate jdbcTemplate) {
        this.redisTemplate = redisTemplate;
        this.jdbcTemplate = jdbcTemplate;
        List overSequences = null;
        /**
         * 如果表不存在创建表
         */
        try {
            overSequences = jdbcTemplate.queryForList(QUERY_GROUP_SEQUECENCE_TABLE_SQL, String.class);
        } catch (Exception e) {
            if (e instanceof BadSqlGrammarException) {
                try {
                    jdbcTemplate.execute(DROP_SEQUECENCE_TABLE_SQL);
                } catch (Exception ex) {

                }
                jdbcTemplate.execute(CREATE_SEQUECENCE_TABLE_SQL);
            }
        }
        /**
         * 清除多条相同sequenceKey数据
         */
        if (CollectionUtil.isNoneEmpty(overSequences)) {
            overSequences.forEach(sequenceKey -> {
                Long maxSequence = jdbcTemplate.queryForObject(QUERY_MAX_SEQUECENCE_TABLE_SQL, Long.class, sequenceKey);
                jdbcTemplate.update(DELETE_SEQUECENCE_TABLE_SQL, sequenceKey);
                jdbcTemplate.update(INIT_SEQUECENCE_TABLE_SQL, sequenceKey, maxSequence);
            });

        }
    }

    public String getOrderNo(String sequenceKey) throws InterruptedException {
        BoundValueOperations boundValueOperations = redisTemplate.boundValueOps(ORDERNO_REDIS_KEY + ":" + sequenceKey);
        /**
         * 如果redis值为空，初始化redis
         */
        if (boundValueOperations.get() == null) {
            BoundValueOperations sequenceTableLockOperations = redisTemplate.boundValueOps(ORDERNO_REDIS_KEY + ":TABLE:LOCK:" + sequenceKey);
            if (!sequenceTableLockOperations.setIfAbsent(true)) {
                Thread.sleep(300);
                sequenceTableLockOperations.expire(10, TimeUnit.SECONDS);
                return getOrderNo(sequenceKey);
            }
            SqlRowSet existSequenceRowSet = jdbcTemplate.queryForRowSet(QUERY_SEQUECENCE_TABLE_SQL, sequenceKey);
            Long existSequence = 0L;
            if (existSequenceRowSet.next()) {
                existSequence = existSequenceRowSet.getLong(1);
            }
            boundValueOperations.set(existSequence);
        }
        long start = System.currentTimeMillis();
        DateFormat dateFormat = new SimpleDateFormat("YYYYMMddHH");
        String dateTime = dateFormat.format(new Date());
        int t = 0;
        String sequence = null;
        Exception exception = null;
        /**
         * 超过5次生成序列号失败则抛出异常
         */
        while (t < 5) {
            try {
                Long increment = boundValueOperations.increment(1);
                if (increment >= maxCycle) {
                    BoundValueOperations sequenceLockOperations = redisTemplate.boundValueOps(ORDERNO_REDIS_KEY + ":CIRCULATE:LOCK:" + sequenceKey);
                    if (sequenceLockOperations.setIfAbsent(true)) {
                        boundValueOperations.set(0L);
                        sequenceLockOperations.expire(300, TimeUnit.MICROSECONDS);
                        return getOrderNo(sequenceKey);
                    }
                    Thread.sleep(200);
                    return getOrderNo(sequenceKey);
                }
                sequence = dateTime + String.valueOf(increment + maxCycle).substring(1);
                /**
                 * 同步现有sequence到数据库
                 */
                executorService.execute(new Runnable() {
                    @Override
                    public void run() {
                        List lastSequenceList = jdbcTemplate.queryForList(QUERY_SEQUECENCE_TABLE_SQL, sequenceKey);
                        /**
                         * 如果数据库无该sequenceKey记录，初始化
                         */
                        if (CollectionUtil.isEmpty(lastSequenceList)) {
                            BoundValueOperations initDbSequenceLockOperations = redisTemplate.boundValueOps(ORDERNO_REDIS_KEY + ":INIT:LOCK:" + sequenceKey);
                            if (initDbSequenceLockOperations.setIfAbsent(true)) {
                                initDbSequenceLockOperations.expire(1, TimeUnit.MINUTES);
                                jdbcTemplate.update(INIT_SEQUECENCE_TABLE_SQL, sequenceKey, increment);
                            } else {
                                try {
                                    Thread.sleep(1000);
                                } catch (InterruptedException e) {
                                    e.printStackTrace();
                                }
                                updateSequence(increment, sequenceKey);
                            }
                        } else {
                            updateSequence(increment, sequenceKey);
                        }
                    }
                });
                break;
            } catch (Exception e) {
                if (e instanceof JedisDataException) {
                    /**
                     * 数据异常，初始化
                     */
                    boundValueOperations.set(0L);
                }
                exception = e;
                Thread.sleep(100);
                t++;
            }
        }
        if (t >= 5) {
            throw new RuntimeException("生成序列号失败", exception);
        }
        logger.debug("生成序列号耗时：{} ", (System.currentTimeMillis() - start));
        return sequence;
    }

    private void updateSequence(Long increment, String sequenceKey) {
        jdbcTemplate.update(UPDATE_SEQUECENCE_TABLE_SQL, new Object[]{increment, increment,maxCycle-1, sequenceKey});
    }

    @Override
    public void destroy() throws Exception {
        executorService.shutdown();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        /**
         * 需要使用单线程队列方式
         */
//        executorService = Executors.newSingleThreadExecutor();
        executorService = new ThreadPoolExecutor(3, 6, 1, TimeUnit.MINUTES, new LinkedBlockingDeque<>(200000));
    }
}
