package outsideapi.service.apiImpl;

import cart.api.OrderMainApi;
import cart.api.dto.order.OrderMainDto;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import com.weibo.api.motan.config.springsupport.annotation.MotanService;
import member.api.AddressCommonApi;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import outsideapi.api.OutsideProductApi;
import outsideapi.service.factory.OutsideApiHandlerFactory;
import outsideapi.service.factory.impl.DbSpringBeanApiHandlerFactory;
import outsideapi.service.factory.impl.StoreTypeSpringBeanApiHandlerFactory;
import outsideapi.service.handler.OutsideApiHandler;
import outsideapi.service.service.OutsideApiService;
import outsideapi.service.utils.OutsideApiFactoryUtil;
import outsideapi.utils.Constants;
import outsideapi.vo.*;
import spider.api.PriceSpiderApi;
import spider.dto.SpiderLogsDto;
import store.api.StoreApi;
import store.api.dto.modeldto.core.StoreDto;
import utils.GlobalContants;
import utils.Lang;
import utils.log.Log;
import utils.log.Logs;
import utils.rpc.motan.ApiResponseVo;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * Created by Liang Wenxu on 2016/11/24.
 */
@MotanService(basicService = "motanServerBasicConfig")
@Transactional(propagation = Propagation.NOT_SUPPORTED)
public class OutsideProductApiImpl implements OutsideProductApi {

    private static Log log = Logs.getLog(OutsideProductApiImpl.class.getName());

    @Autowired
    OutsideApiService outsideApiService;

    @Resource(name = "redisTemplate")
    private ValueOperations<String, Object> commonCache;

    private static final String STORE_DTO_BY_ID_CACHE_KEY = "OUTSIDE-SERVICE:STORE-DTO:ID:";

    private static final String STORE_DTO_BY_CODE_CACHE_KEY = "OUTSIDE-SERVICE:STORE-DTO:CODE:";

    @Override
    public HandlerRespVo<List<FashionStockStateVo>> getFashionStockState(StoreDto store, FashionStatusRequetVo requetVo) {
        if (store != null && (!Lang.isEmpty(store.getId()) || store.getStoreExt() != null) && requetVo != null) {
            OutsideApiHandler handler = getOutsideApiHandler(store.getId());

            if (handler == null) {
                return handlerNotFoundResp();
            }

            requetVo.setStoreId(store.getId());
            return outsideApiService.queryFashionStockState(handler, requetVo);

        } else {
            HandlerRespVo<List<FashionStockStateVo>> resp = new HandlerRespVo<>();
            resp.setData(null);
            resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
            resp.setErrorCode(1);
            resp.setMessage("Required params [" + (store == null ? "store ," : "") + (requetVo == null ? "requetVo" : "") + "] CAN NOT Be Null");
            return resp;
        }
    }

    @Override
    public HandlerRespVo<Boolean> isOutsideStore(StoreDto store) {
        return isOutsideStore(store.getId());
    }

    @Override
    public HandlerRespVo<Boolean> isOutsideStore(String storeId) {
        HandlerRespVo<Boolean> resp = new HandlerRespVo<>();
        resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
        resp.setErrorCode(0);
        if (!Lang.isEmpty(storeId)) {
            OutsideApiHandler handler = getOutsideApiHandler(storeId);
            if (handler == null) {
                resp = handlerNotFoundResp();
            } else {
                resp.setData(true);
            }
            return resp;
        } else {
            resp.setData(null);
            resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
            resp.setErrorCode(1);
            resp.setMessage("Required params [" + (storeId == null ? "storeId ," : "") + "] CAN NOT Be Null");
            return resp;
        }
    }

    @Override
    public HandlerRespVo<Map<String, Object>> queryShippingFee(String storeId, ShippingFeeRequetVo requet) {
        OutsideApiHandler handler = getOutsideApiHandler(storeId);
        if (handler == null) {
            return handlerNotFoundResp();
        }

        // 增加重复商品的合并
        Map<String, FashionNumsVo> fashionNumsVoMap = new HashMap<>(); // 使用HashMap去重，key 为sku
        FashionNumsVo tempVo;
        for (FashionNumsVo v : requet.getFashionNums()) {
            tempVo = fashionNumsVoMap.get(v.getProductCode());
            if (tempVo != null) {
                tempVo.setCounts(tempVo.getCounts() + v.getCounts());
            } else {
                fashionNumsVoMap.put(v.getProductCode(), v);
            }
        }
        List<FashionNumsVo> newList = new ArrayList<>();
        for (Map.Entry<String, FashionNumsVo> entry : fashionNumsVoMap.entrySet()) {
            newList.add(entry.getValue());
        }
        requet.setFashionNums(newList);
        requet.setStoreId(storeId);

        return handler.queryShippingFee(requet);
    }

    @Override
    public HandlerRespVo<BigDecimal> getSalesPriceRateRangeVal(FashionPricesRequestVo requestVo) {
        OutsideApiHandler handler = getOutsideApiHandler(Lang.isEmpty(requestVo.getStoreCode()) ? requestVo.getStoreId() : requestVo.getStoreCode());
        HandlerRespVo<BigDecimal> resp = new HandlerRespVo<>();
        if (handler == null) {
            resp = handlerNotFoundResp();
        } else {
            resp.setData(handler.getSalesPriceRateRangeVal(requestVo));
            resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
        }
        return resp;
    }

    /**
     * 获取第三方接口处理类
     *
     * @param storeKey 可传storeCode或storeId，优先匹配Code
     * @return
     */
    private OutsideApiHandler getOutsideApiHandler(String storeKey) {
        StoreDto store=null;
//        StoreDto store =(StoreDto) commonCache.get(STORE_DTO_BY_ID_CACHE_KEY + storeKey);
//        if (Lang.isEmpty(store)) {
//            store = (StoreDto) commonCache.get(STORE_DTO_BY_CODE_CACHE_KEY + storeKey);
//        }
        if (Lang.isEmpty(store)) {
            store = storeApi.findByStoreId(storeKey);
            if (!Lang.isEmpty(store)) {
                commonCache.set(STORE_DTO_BY_ID_CACHE_KEY + storeKey, store, 1, TimeUnit.DAYS);
            }
        }
        if (Lang.isEmpty(store)) {
            store = storeApi.findByCode(storeKey);
            if (!Lang.isEmpty(store)) {
                commonCache.set(STORE_DTO_BY_CODE_CACHE_KEY + storeKey, store, 1, TimeUnit.DAYS);
            }
        }

        if (store != null && (!Lang.isEmpty(store.getId()) || store.getStoreExt() != null)) {
            OutsideApiHandler handler = null;
            OutsideApiHandlerFactory factory = null;
            // 第三方接口处理类工厂网关
            if (store.getStoreExt() != null &&
                    GlobalContants.STORE_TYPES.PROTOCOL.equals(store.getStoreExt().getType())) {
                // 协议店铺使用特殊工厂
                factory = outsideApiFactoryUtil.getFactory(DbSpringBeanApiHandlerFactory.class);
            } else if (store.getStoreExt() != null &&
                    GlobalContants.STORE_TYPES.TRDSP.equals(store.getStoreExt().getType())) {
                //TRDSP使用的handler
                handler = outsideApiFactoryUtil.getFactory(StoreTypeSpringBeanApiHandlerFactory.class).getOutsideApiHandlerByStoreType(GlobalContants.STORE_TYPES.TRDSP);
                return handler;
            } else if(store.getStoreExt() != null &&
                    GlobalContants.STORE_TYPES.LOCALSTORE.equals(store.getStoreExt().getType())){
                HandlerKey handlerKey=new HandlerKey();
                handlerKey.setStoreType(GlobalContants.STORE_TYPES.LOCALSTORE);
                handlerKey.setInterfaceName(Constants.HandlerInterfaceNames.OUTSIDE_API_HANDLER.toString());
                handler= outsideApiFactoryUtil.getFactory().getOutsideApiHandler(handlerKey,OutsideApiHandler.class);
                return handler;
            }else {
                // 一般供应商接口处理类工厂
                factory = outsideApiFactoryUtil.getFactory();
            }

            if (Lang.isEmpty(factory)) {
                throw new RuntimeException("outside.common.apihandler.factory not config !");
            }

            // 获取Handler
            HandlerKey handlerKey = new HandlerKey();
            handlerKey.setStoreType(store.getStoreExt().getType());
            if (store.getStoreExt() != null &&
                    GlobalContants.STORE_TYPES.PROTOCOL.equals(store.getStoreExt().getType())) {
                handlerKey.setStoreCode(HandlerKey.PRODUCT_TYPE_ALL);
            } else {
                handlerKey.setStoreCode(store.getStoreExt().getCode());
            }
            handlerKey.setProductType(HandlerKey.PRODUCT_TYPE_ALL);
            handlerKey.setInterfaceName(Constants.HandlerInterfaceNames.OUTSIDE_API_HANDLER.toString());
            handler = factory.getOutsideApiHandler(handlerKey, OutsideApiHandler.class);
            return handler;
        }
        return null;
    }

    @Override
    public HandlerRespVo<List<BigDecimal>> getBatchSalesPriceRateRangeVal(List<FashionPricesRequestVo> requestVoList) {
        HandlerRespVo<List<BigDecimal>> resp = new HandlerRespVo<>();
        // 201804 Drury 版本重构 多线程加速
        List<BigDecimal> rsList = new Vector<>();
        for (FashionPricesRequestVo aRequestVoList : requestVoList) {
            HandlerRespVo<BigDecimal> r = getSalesPriceRateRangeVal(aRequestVoList);
            if (r.getStatus().equals(HandlerRespVo.RESPONSE_STATUS_SUCCESS)) {
                rsList.add(r.getData());
            } else {
                rsList.add(null);
            }
        }

        resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
        resp.setData(rsList);
        return resp;
    }

    /**
     * 查询第三方实时价格接口
     *
     * @param storeKey storeCode或storeId，优先传storeCode
     * @param request
     * @return
     */
    @Override
    public HandlerRespVo<List<FashionPriceVo>> queryFashionPrice(String storeKey, FashionPriceRequestVo request) {
        long methodStartTime = System.currentTimeMillis();
        OutsideApiHandler handler = getOutsideApiHandler(storeKey);
        //TODO: STOREKEY和storeId关系
        request.setStoreId(storeKey);

        HandlerRespVo<List<FashionPriceVo>> resp;
        if (handler == null) {
            resp = handlerNotFoundResp();
        } else {
            resp = new HandlerRespVo<>();
            HandlerRespVo<List<FashionPriceVo>> queryResp = handler.queryFashionPrice(request);
            if (queryResp != null && queryResp.getStatus() == HandlerRespVo.RESPONSE_STATUS_SUCCESS) {
                resp.setData(queryResp.getData());
                resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
            } else {
                resp = queryResp;
            }

        }
        log.info("查询第三方实时价格耗时 {} ms", System.currentTimeMillis() - methodStartTime);
        return resp;
    }

    @Override
    public HandlerRespVo<List<OrderShippingPackageVo>> getShipingTrace(String orderNo) {
        HandlerRespVo<List<OrderShippingPackageVo>> resp = new HandlerRespVo<>();
        OrderMainDto orderMain = orderMainApi.findByOrderNo(orderNo);
        if (orderMain != null) {
            OutsideApiHandler handler = getOutsideApiHandler(orderMain.getStoreId());
            if (handler == null) {
                resp.setData(null);
                resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
                resp.setErrorCode(2);
                resp.setMessage("Handler Not Found!");

            } else {
                HandlerRespVo<List<OrderShippingPackageVo>> handlerRespVo = handler.getShipingTrace(orderMain.getThirdOrderNo());
                if (handlerRespVo.getStatus().equals(HandlerRespVo.RESPONSE_STATUS_SUCCESS)) {
                    resp.setData(handlerRespVo.getData());
                    resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
                } else {
                    resp = handlerRespVo;
                }
            }
        }
        return resp;
    }

    @Override
    public HandlerRespVo<String> orderStatus(String orderNo) {
        HandlerRespVo<String> resp = new HandlerRespVo<>();
        OrderMainDto orderMain = orderMainApi.findByOrderNo(orderNo);
        if (orderMain != null) {
            OutsideApiHandler handler = getOutsideApiHandler(orderMain.getStoreId());
            if (handler == null) {
                resp.setData(null);
                resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
                resp.setErrorCode(2);
                resp.setMessage("Handler Not Found!");

            } else {
                HandlerRespVo<String> handlerRespVo = handler.getOrderStatus(orderMain.getThirdOrderNo());
                if (handlerRespVo.getStatus().equals(HandlerRespVo.RESPONSE_STATUS_SUCCESS)) {
                    resp.setData(handlerRespVo.getData());
                    resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
                } else {
                    resp = handlerRespVo;
                }
            }
        }
        return resp;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Transactional(propagation = Propagation.NOT_SUPPORTED)
    public HandlerRespVo<BigDecimal> runPriceSpider(String storeId, FashionPriceParamVo request, String areaCode, String triggerType, Boolean async) {
        OutsideApiHandler handler = getOutsideApiHandler(storeId);

        HandlerRespVo<BigDecimal> resp = new HandlerRespVo<>();
        if (handler == null) {
            resp = handlerNotFoundResp();
        } else {
            Map<Integer, String> areaCodeMap = addressCommonApi.areaCodeNodeList(areaCode);
            if (areaCodeMap != null) {
                Map<Integer, String> trdAreaCodeMap = handler.transAreaCodeMap(areaCodeMap);
                try {
                    ApiResponseVo<SpiderLogsDto> apiResponseVo = priceSpiderApi.getSalesPrice(storeId, request.getProductCode(), trdAreaCodeMap, triggerType);
                    if (apiResponseVo != null &&
                            GlobalContants.ResponseStatus.SUCCESS.equals(apiResponseVo.getStatus())) {
                        resp.setData(apiResponseVo.getData().getSalesPrice());
                        resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
                    } else {
                        resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
                        resp.setMessage("SpiderApi未返回数据");
                        resp.setData(null);
                        resp.setErrorCode(3);
                    }
                } catch (Exception e) {
//                    e.printStackTrace();
                    log.error(e, "调用价格爬虫API失败...");
                    resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
                    resp.setMessage(e.getLocalizedMessage());
                    resp.setData(null);
                    resp.setErrorCode(2);
                }
            }
        }
        return resp;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Transactional(propagation = Propagation.NOT_SUPPORTED)
    public HandlerRespVo<Map<String, BigDecimal>> runPriceSpider(String storeId, FashionPriceRequestVo request, String triggerType, Boolean async) {
        OutsideApiHandler handler = getOutsideApiHandler(storeId);

        HandlerRespVo<Map<String, BigDecimal>> resp = new HandlerRespVo<>();
        if (handler == null) {
            resp = handlerNotFoundResp();
        } else {
            Map<Integer, String> areaCodeMap = null;
            if (!Lang.isEmpty(request.getTownCode()) || !Lang.isEmpty(request.getCityCode())
                    || !Lang.isEmpty(request.getCountyCode()) || !Lang.isEmpty(request.getProvinceCode())) {
                areaCodeMap = new HashMap<>();
                areaCodeMap.put(AddressCommonApi.AddressLevel.PROVINCE.getValue(), request.getProvinceCode());
                areaCodeMap.put(AddressCommonApi.AddressLevel.CITY.getValue(), request.getCityCode());
                areaCodeMap.put(AddressCommonApi.AddressLevel.AREA.getValue(), request.getCountyCode());
                areaCodeMap.put(AddressCommonApi.AddressLevel.TOWN.getValue(), request.getTownCode());
            }
            Long time = System.currentTimeMillis();
            Map<Integer, String> trdAreaCodeMap = areaCodeMap == null ? null : handler.transAreaCodeMap(areaCodeMap);
            log.info("获取第三方销售价格转换地址：{}ms", System.currentTimeMillis() - time);

            try {
                List<String> skuIds = new ArrayList<>();
                request.getFashionPriceParamVos();
                for (FashionPriceParamVo f : request.getFashionPriceParamVos()) {
                    if (!Lang.isEmpty(f.getProductCode())) {
                        skuIds.add(f.getProductCode());
                    }
                }
                time = System.currentTimeMillis();
                ApiResponseVo<Map<String, SpiderLogsDto>> apiResponseVo = priceSpiderApi.getBatchSalesPrice(storeId, skuIds, trdAreaCodeMap, triggerType);
                log.info("获取第三方销售价格接口：{}ms", System.currentTimeMillis() - time);
                if (apiResponseVo != null &&
                        GlobalContants.ResponseStatus.SUCCESS.equals(apiResponseVo.getStatus())) {
                    Map<String, BigDecimal> resData = new HashMap<>();
                    SpiderLogsDto l = null;
                    for (FashionPriceParamVo f : request.getFashionPriceParamVos()) {
                        l = apiResponseVo.getData().get(f.getProductCode());
                        if (l != null) {
                            resData.put(f.getProductCode(), l.getSalesPrice());
                        }
                    }

                    resp.setData(resData);
                    resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
                }
            } catch (Exception e) {
//                e.printStackTrace();
                log.error(e, "调用价格爬虫API失败...");
                resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
                resp.setMessage(e.getLocalizedMessage());
                resp.setData(null);
                resp.setErrorCode(2);
            }
        }
        return resp;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public HandlerRespVo<Map<Integer, String>> transAreaCodeMap(String storeKey, Map<Integer, String> areaCodeMap) {
        OutsideApiHandler handler = getOutsideApiHandler(storeKey);

        HandlerRespVo<Map<Integer, String>> resp = new HandlerRespVo<>();
        if (handler == null) {
            resp = handlerNotFoundResp();
        } else {
            resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
            resp.setData(handler.transAreaCodeMap(areaCodeMap));
        }
        return resp;
    }

    private HandlerRespVo handlerNotFoundResp() {
        HandlerRespVo<List<FashionPriceVo>> resp = new HandlerRespVo<>();
        resp.setData(null);
        resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
        resp.setErrorCode(2);
        resp.setMessage("Handler Not Found!");
        return resp;
    }


    @Override
    public HandlerRespVo<List<OrderShippingPackageVo>> getManyOrderShipingTrace(List<String> orderNos) {
        HandlerRespVo<List<OrderShippingPackageVo>> resp = new HandlerRespVo<>();
        List<OrderShippingPackageVo> orderShippingPackageVos = new ArrayList<OrderShippingPackageVo>();
        for (String orderNo : orderNos) {
            OrderMainDto orderMain = orderMainApi.findByOrderNo(orderNo);
            if (orderMain != null) {
                OutsideApiHandler handler = getOutsideApiHandler(orderMain.getStoreId());
                if (handler == null) {
                    resp.setData(null);
                    resp.setStatus(HandlerRespVo.RESPONSE_STATUS_ERROR);
                    resp.setErrorCode(2);
                    resp.setMessage("Handler Not Found!");
                    return resp;
                } else {
                    HandlerRespVo<List<OrderShippingPackageVo>> handlerRespVo = handler.getShipingTrace(orderMain.getThirdOrderNo());
                    if (handlerRespVo.getStatus().equals(HandlerRespVo.RESPONSE_STATUS_SUCCESS)) {
                        orderShippingPackageVos.addAll(handlerRespVo.getData());
                        //resp.setData(handlerRespVo.getData());
                        //resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
                    } else {
                        return handlerRespVo;
                    }
                }
            }
        }

        for (OrderShippingPackageVo packageVo : orderShippingPackageVos) {
            packageVo.getTrackInfoList().sort(new Comparator<OrderShippingTrackVo>() {
                @Override
                public int compare(OrderShippingTrackVo o1, OrderShippingTrackVo o2) {
                    return o2.getMsgTime().compareTo(o1.getMsgTime());
                }
            });
        }
        resp.setData(orderShippingPackageVos);
        resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
        return resp;
    }


    @MotanReferer
    PriceSpiderApi priceSpiderApi;

    @MotanReferer
    AddressCommonApi addressCommonApi;

    @Autowired
    OutsideApiFactoryUtil outsideApiFactoryUtil;

    @MotanReferer
    OrderMainApi orderMainApi;

    @MotanReferer
    StoreApi storeApi;

    @Override
    public String toString() {
        System.out.println("method called........");
        return "OutsideProductApiImpl{}";
    }

    @Override
    public HandlerRespVo<List<BatchAreaRequestVo>> batchTransAreaCodeMap(String storeKey, List<BatchAreaRequestVo> requestVos) {

        OutsideApiHandler handler = getOutsideApiHandler(storeKey);

        HandlerRespVo<List<BatchAreaRequestVo>> resp = new HandlerRespVo<>();
        if (handler == null) {
            resp = handlerNotFoundResp();
        } else {
            resp.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
            List<BatchAreaRequestVo> result = new ArrayList<>();
            for (BatchAreaRequestVo batchAreaRequestVo : requestVos) {
                BatchAreaRequestVo batchAreaRequestVoResult = new BatchAreaRequestVo();
                batchAreaRequestVoResult.setIndexKey(batchAreaRequestVo.getIndexKey());
                Map<Integer, String> areaCodeMap = batchAreaRequestVo.getAreaCodeMap();
                batchAreaRequestVoResult.setAreaCodeMap(handler.transAreaCodeMap(areaCodeMap));
                result.add(batchAreaRequestVoResult);
            }
            resp.setData(result);
        }
        return resp;
    }

    /**
     * 8.8 生成预占单
     *
     * @param reserveOrderVo
     * @param storeCode
     * @return 预订单ｉｄ对象
     */
    @Override
    public HandlerRespVo<ReserveVo> createReservation(ReserveOrderVo reserveOrderVo, String storeCode) {
        OutsideApiHandler handler = getOutsideApiHandler(storeCode);

        HandlerRespVo<ReserveVo> resp = new HandlerRespVo<>();
        if (handler == null) {
            resp = handlerNotFoundResp();
        } else {
            resp = handler.createReservation(reserveOrderVo, storeCode);
        }

        return resp;
    }

    /**
     * 8.9　删除预占单
     *
     * @param reserveOrderIdVo
     * @param storeCode
     * @return true/false
     */
    @Override
    public HandlerRespVo<Boolean> delReservation(ReserveOrderIdVo reserveOrderIdVo, String storeCode) {
        OutsideApiHandler handler = getOutsideApiHandler(storeCode);

        HandlerRespVo<Boolean> resp = new HandlerRespVo<>();
        if (handler == null) {
            resp = handlerNotFoundResp();
        } else {
            resp = handler.delReservation(reserveOrderIdVo, storeCode);
        }

        return resp;
    }
}
