package outsideapi.service.apiImpl;

import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import com.weibo.api.motan.config.springsupport.annotation.MotanService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import outsideapi.api.SupplierOrderApi;
import outsideapi.exceptions.OrderCancelException;
import outsideapi.exceptions.SupplierInterfaceInvokeException;
import outsideapi.exceptions.SupplierOrderComfirmFalidException;
import outsideapi.exceptions.SupplierTakeOrderFalidException;
import outsideapi.service.factory.OutsideApiHandlerFactory;
import outsideapi.service.factory.impl.DbSpringBeanApiHandlerFactory;
import outsideapi.service.factory.impl.StoreTypeSpringBeanApiHandlerFactory;
import outsideapi.service.handler.SupplierOrderApiHandler;
import outsideapi.service.utils.OutsideApiFactoryUtil;
import outsideapi.utils.Constants;
import outsideapi.vo.BackOrderResp;
import outsideapi.vo.HandlerKey;
import outsideapi.vo.order.TrdOrder;
import outsideapi.vo.order.TrdOrderKey;
import outsideapi.vo.orderrequest.OrderRequst;
import store.api.StoreApi;
import store.api.dto.modeldto.core.StoreDto;
import utils.GlobalContants;
import utils.Lang;

import java.util.List;

/**
 * @author Liang Wenxu
 * @since 2018/8/13
 */
@Slf4j
@MotanService
public class SupplierOrderApiImpl implements SupplierOrderApi {

    /**
     * 取消订单
     *
     *
     *
     * @param trdOrderKey@return
     * @throws SupplierInterfaceInvokeException 供应商接口链接异常
     */
    @Override
    public Boolean cancelOrder(TrdOrderKey trdOrderKey) throws SupplierInterfaceInvokeException, OrderCancelException {
        return getOutsideApiHandler(trdOrderKey.getStoreCode()).cancelOrder(trdOrderKey);
    }

    /**
     * 向第三方供应商下单
     *
     * @param orderMain 商城订单信息
     * @return 返回生产成功的第三方订单详情
     * @throws SupplierInterfaceInvokeException 供应商接口调用异常
     * @throws SupplierTakeOrderFalidException  供应商下单失败异常
     */
    @Override
    public TrdOrder takeOrder(OrderRequst orderMain) throws SupplierInterfaceInvokeException, SupplierTakeOrderFalidException {
        return getOutsideApiHandler(orderMain.getStoreCode()).takeOrder(orderMain);
    }

    /**
     * 确认下单，调用时记得处理所有的异常噢！
     *
     * @param orderMain 商城订单信息
     * @throws SupplierInterfaceInvokeException   供应商接口调用异常
     * @throws SupplierOrderComfirmFalidException 供应商下单失败异常，其中错误代码3103表示已确认下单无需重复确认
     */
    @Override
    public void confirmOrder(OrderRequst orderMain) throws SupplierInterfaceInvokeException, SupplierOrderComfirmFalidException {
        getOutsideApiHandler(orderMain.getStoreCode()).confirmOrder(orderMain);
    }

    @Override
    public BackOrderResp backOrder(OrderRequst orderMain, List<String> skuIds) throws SupplierInterfaceInvokeException {
        return getOutsideApiHandler(orderMain.getStoreCode()).backOrder(orderMain, skuIds);
    }

    /**
     * 获取订单详情
     *
     *
     * @param trdOrderKey 第三方订单唯一标识，采用storeCode + trdOrderNo作为唯一标识
     * @return 返回第三方订单详情
     * @throws SupplierInterfaceInvokeException 供应商接口链接异常
     */
    @Override
    public TrdOrder getDetail(TrdOrderKey trdOrderKey) throws SupplierInterfaceInvokeException {
        return getOutsideApiHandler(trdOrderKey.getStoreCode()).getDetail(trdOrderKey);
    }

    /**
     * 订单号反查接口
     *
     *
     * @param trdOrderKey@return 第三方供应商订单号
     */
    @Override
    public String queryTrdOrderNo(TrdOrderKey trdOrderKey) throws SupplierInterfaceInvokeException {
        return getOutsideApiHandler(trdOrderKey.getStoreCode()).queryTrdOrderNo(trdOrderKey);
    }

    /**
     * 获取第三方接口处理类
     *
     * @param storeKey 可传storeCode或storeId，优先匹配Code
     * @return
     */
    private SupplierOrderApiHandler getOutsideApiHandler(String storeKey) {
        long startTime = System.currentTimeMillis();
        StoreDto store = storeApi.findByStoreId(storeKey);
        store = store == null ? storeApi.findByCode(storeKey) : store;

        if (store != null && (!Lang.isEmpty(store.getId()) || store.getStoreExt() != null)) {
            SupplierOrderApiHandler handler = null;
            OutsideApiHandlerFactory factory = null;
            // 第三方接口处理类工厂网关
            if (store.getStoreExt() != null &&
                    GlobalContants.STORE_TYPES.PROTOCOL.equals(store.getStoreExt().getType())) {
                // 协议店铺使用特殊工厂
                factory = outsideApiFactoryUtil.getFactory(DbSpringBeanApiHandlerFactory.class);
            } else {
                // 一般供应商接口处理类工厂
                factory = outsideApiFactoryUtil.getFactory();
            }

            if (Lang.isEmpty(factory)) {
                throw new RuntimeException("outside.common.apihandler.factory not config !");
            }

            // 获取Handler
            HandlerKey handlerKey = new HandlerKey();
            handlerKey.setStoreType(store.getStoreExt().getType());
            if (store.getStoreExt() != null &&
                    GlobalContants.STORE_TYPES.PROTOCOL.equals(store.getStoreExt().getType())) {
                handlerKey.setStoreCode(HandlerKey.PRODUCT_TYPE_ALL);
            }else{
                handlerKey.setStoreCode(store.getStoreExt().getCode());
            }
            handlerKey.setProductType(HandlerKey.PRODUCT_TYPE_ALL);
            handlerKey.setInterfaceName(Constants.HandlerInterfaceNames.SUPPLIER_ORDER_API_HANDLER.toString());
            handler = factory.getOutsideApiHandler(handlerKey, SupplierOrderApiHandler.class);
            log.info("获取 handler 耗时 {} 秒", (System.currentTimeMillis() - startTime) / 1000.0);
            return handler;

        }
        log.info("获取 handler 耗时 {} 秒", (System.currentTimeMillis() - startTime) / 1000.0);
        return null;
    }

    @MotanReferer
    StoreApi storeApi;

    @Autowired
    OutsideApiFactoryUtil outsideApiFactoryUtil;
}
