package outsideapi.service.handler.impl;

import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import outsideapi.service.handler.OutsideOrderApiHandler;
import outsideapi.service.service.stub.EmayOrderApiStub;
import outsideapi.service.service.stub.OrderMainApiStub;
import outsideapi.utils.Constants;
import outsideapi.vo.*;
import outsideapi.vo.order.OrderMainDto;
import outsideapi.vo.order.TrdOrderResp;
import recharge.center.api.emay.api.definition.ErrorInfo;
import recharge.center.api.emay.api.request.OrderEntity;
import recharge.center.api.emay.api.request.OrderQueryEntity;
import recharge.center.api.emay.api.response.OrderQueryResp;
import recharge.center.api.emay.api.response.OrderResp;

import java.util.List;

/**
 * 亿美订单相关接口处理实现
 *
 * @author Liang Wenxu
 * @since 2018-06-12
 */
@Slf4j
@Component
@Transactional(readOnly = true, rollbackFor = Exception.class)
public class EmayOutsideOrderApiHandler implements OutsideOrderApiHandler {

    @Override
    public CancelOrderResp cancelOrder(String trdOrderNo, String storeId) {
        CancelOrderResp cancelOrderResp = new CancelOrderResp();
        cancelOrderResp.setCode("0");
        cancelOrderResp.setMessage("success");
        cancelOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        return cancelOrderResp;
    }

    @Override
    public TakeOrderResp takeOrder(OrderMainDto orderMain) {
        // 当前方法返回对象
        TakeOrderResp takeOrderResp = new TakeOrderResp();

        // 封装下单请求参数
        OrderEntity orderEntity = new OrderEntity();
        orderEntity.setClientTaskNo(orderMain.getOrderNo());
        orderEntity.setMobile(orderMain.getOrderItems().get(0).getBusinessCode());
        orderEntity.setProductCode(orderMain.getOrderItems().get(0).getSku());
        log.info("亿美下单请求参数 : {}", JSON.toJSONString(orderEntity));

        long takeOrderTime = System.currentTimeMillis();
        log.info("正在调用亿美供应商接口下单...");
        // 调用接口下单
        OrderResp orderResp;
        try {
            orderResp = orderApi.takeOrder(orderEntity);
        } catch (Exception e) {
            log.error("调用亿美供应商下单接口异常", e);
            takeOrderResp.setReturnCode("1");
            takeOrderResp.setReturnMsg("调用亿美供应商下单接口异常 : " + e.getMessage());
            takeOrderResp.setStatus(Constants.ResponseStatus.ERROR);
            return takeOrderResp;
        }
        log.info("调用亿美供应商接口返回数据 : {}", orderResp);
        log.info("调用亿美供应商接口耗时 {} ms", System.currentTimeMillis() - takeOrderTime);

        // 成功响应代码
        final String successCode = "SUCCESS";

        if (successCode.equals(orderResp.getResponseCode())) {
            // 供应商成功下单
            outsideapi.vo.OrderResp returnOrderResp = new outsideapi.vo.OrderResp();
            returnOrderResp.setTrdOrderNo(orderResp.getTaskNo());
            takeOrderResp.setOrderResp(returnOrderResp);
            takeOrderResp.setReturnCode("0");
            takeOrderResp.setReturnMsg("供应商下单成功");
            takeOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        } else {
            // 供应商
            takeOrderResp.setReturnCode("2");
            takeOrderResp.setReturnMsg("亿美供应商下单失败. " + orderResp.getResponseCode() + " : " + ErrorInfo.ERRORS.get(orderResp.getResponseCode()));
            takeOrderResp.setStatus(Constants.ResponseStatus.ERROR);
        }

        log.info("方法结束返回对象 : {}", JSON.toJSONString(takeOrderResp));
        return takeOrderResp;
    }

    @Override
    public ConfirmOrderResp confirmOrder(String trdOrderNo, String storeId) {
        ConfirmOrderResp confirmOrderResp = new ConfirmOrderResp();
        confirmOrderResp.setCode("0");
        confirmOrderResp.setMessage("SUCCESS");
        confirmOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        return confirmOrderResp;
    }

    @Override
    public BackOrderResp backOrder(OrderMainDto orderMain, List<String> skuIds) {
        BackOrderResp backOrderResp = new BackOrderResp();
        backOrderResp.setCode("0");
        backOrderResp.setMessage("SUCCESS");
        backOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        return backOrderResp;
    }

    @Override
    public String queryTrdOrderNo(String orderNo) {
        return orderNo + "_TRD";
    }

    @Override
    public TrdOrderResp queryOrder(String trdOrderNo) {
        // 当前方法返回对象
        TrdOrderResp queryOrderResp = new TrdOrderResp();

        String orderNo = orderMainApi.findOrderNoByThirdOrderNo(trdOrderNo);

        // 封装下单请求参数
        OrderQueryEntity orderQueryEntity = new OrderQueryEntity(orderNo, trdOrderNo);

        OrderQueryResp orderQueryResp;
        try {
            orderQueryResp = orderApi.queryOrder(orderQueryEntity);
        } catch (Exception e) {
            log.error("调用亿美供应商订单查询接口异常", e);
            queryOrderResp.setCode("ERROR");
            queryOrderResp.setMessage("调用亿美供应商订单查询接口异常 : " + e.getMessage());
            queryOrderResp.setStatus(Constants.ResponseStatus.ERROR);
            return queryOrderResp;
        }

        log.info("返回代码 = {}, 代码注释 = {}", orderQueryResp.getResponseCode(), ErrorInfo.ERRORS.get(orderQueryResp.getResponseCode()));

        queryOrderResp.setCode(orderQueryResp.getResponseCode());
        queryOrderResp.setMessage("SUCCESS");
        queryOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);

        if(ErrorInfo.TIME_OUT.equals(orderQueryResp.getResponseCode())) {
            queryOrderResp.setMessage("TIME_OUT");
            queryOrderResp.setStatus(Constants.ResponseStatus.TIME_OUT);
        }

        return queryOrderResp;
    }

    final
    private EmayOrderApiStub orderApi;

    final
    private OrderMainApiStub orderMainApi;

    @Autowired
    public EmayOutsideOrderApiHandler(EmayOrderApiStub orderApi, OrderMainApiStub orderMainApi) {
        this.orderApi = orderApi;
        this.orderMainApi = orderMainApi;
    }
}
