package outsideapi.service.handler.impl;

import cart.api.dto.order.OrderMainDto;
import cart.api.dto.order.SubOrderDto;
import cart.api.dto.order.SubOrderItemDto;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import goods.model.GoodsPicture;
import goods.model.repository.GoodsPictureRepos;
import jd.api.request.order.JdOrderIdEntity;
import jd.api.request.order.OrderEntity;
import jd.api.request.order.ThirdOrder;
import jd.api.request.product.SkuIdAndNumEntity;
import jd.api.response.order.OccupyStockResp;
import jd.api.response.order.OrderRevResp;
import jd.api.response.order.OrderTrackResp;
import jd.api.response.order.OrderUniteSubmitResp;
import jd.api.vo.order.OrdertrackResultVO;
import jd.api.vo.order.OrdertrackVO;
import jd.api.vo.order.ParentOrderRepVO;
import jd.dto.NationalAddressDto;
import jd.model.repository.NationalAddressRepos;
import lombok.extern.slf4j.Slf4j;
import member.api.AddressCommonApi;
import member.api.MemberAddressApi;
import member.api.dto.shop.MemberAddressDto;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import outsideapi.api.OutsideProductApi;
import outsideapi.exceptions.OrderCancelException;
import outsideapi.exceptions.SupplierInterfaceInvokeException;
import outsideapi.exceptions.SupplierOrderComfirmFalidException;
import outsideapi.exceptions.SupplierTakeOrderFalidException;
import outsideapi.service.handler.SupplierOrderApiHandler;
import outsideapi.service.service.stub.AddressCommonApiStub;
import outsideapi.service.service.stub.JdOrderApiStub;
import outsideapi.service.service.stub.JdProductApiStub;
import outsideapi.service.service.stub.OrderMainApiStub;
import outsideapi.service.utils.JdOrderUtil;
import outsideapi.service.utils.TrdOrderHandlerUtils;
import outsideapi.utils.jd.JdSkuUtil;
import outsideapi.vo.*;
import outsideapi.vo.jdorder.Corder;
import outsideapi.vo.jdorder.JdOrder;
import outsideapi.vo.jdorder.JdOrderConstant;
import outsideapi.vo.jdorder.OrderPriceSnap;
import outsideapi.vo.order.TrdOrder;
import outsideapi.vo.order.TrdOrderKey;
import outsideapi.vo.orderrequest.OrderItemVo;
import outsideapi.vo.orderrequest.OrderRequst;
import sinomall.global.common.vo.outsideapi.OrderCancelOrderResp;
import sysmg.api.SystemConfigApi;
import sysmg.dto.SystemConfigDto;
import sysmg.response.SystemConfigValueDto;
import utils.GlobalContants;
import utils.Lang;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

//import jd.service.jdapi.order.JdOrderApiImpl;
//import jd.service.jdapi.product.JdProductApiImpl;

/**
 * Created by Liang Wenxu on 2016/11/24.
 */
@Slf4j
@Component
@Transactional(readOnly = true)
public class JdSupplierOrderApiHandler implements SupplierOrderApiHandler, ApplicationContextAware {

    @Override
    public Boolean cancelOrder(TrdOrderKey trdOrderKey) throws SupplierInterfaceInvokeException, OrderCancelException {
        Map<String, Object> returnMap = new HashMap<>();
        if (Lang.isEmpty(trdOrderKey.getTrdOrderNo())) {
            throw new OrderCancelException(OrderCancelException.ERROR_CODES.TRD_ORDER_NOTFOUND);
        }
        OrderCancelOrderResp jdResp;
        try {
            jdResp = jdOrderApi.cancelOrder(new JdOrderIdEntity(Long.parseLong(trdOrderKey.getTrdOrderNo())));
        } catch (NumberFormatException nfe) {
            throw new OrderCancelException(OrderCancelException.ERROR_CODES.ILLEGAL_TRD_ORDERNO);
        } catch (Exception e) {
            throw new SupplierInterfaceInvokeException(e.getMessage()
                    , SupplierInterfaceInvokeException.ERROR_CODES.NETWORK_CONNECTION_FAILD
                    , e);
        }

        if(jdResp.getResult()) {
            return true;
        } else {
            /**
             * 需标记为取消失败的京东情况：
             * 3204	不能取消已经生产订单
             * 3208	不能取消已确认订单
             * 3211	取消订单失败，存在已确认的子单
             * 3212	取消订单操作失败
             */
            returnMap.put(GlobalContants.ResponseString.STATUS,GlobalContants.ResponseStatus.ERROR);
            String errorCode = "0";
            if("3204".equals(jdResp.getResultCode())
                    || "3208".equals(jdResp.getResultCode())
                    || "3211".equals(jdResp.getResultCode())) {
                throw new OrderCancelException(OrderCancelException.ERROR_CODES.TRD_ORDER_COMFIRMED);
            } else if ("3212".equals(jdResp.getResultCode())) {
                throw new OrderCancelException(jdResp.getResultCode()+": "+jdResp.getResultMessage(), OrderCancelException.ERROR_CODES.API_ERROR);
            } else {
                throw new OrderCancelException(jdResp.getResultCode()+": "+jdResp.getResultMessage(), OrderCancelException.ERROR_CODES.API_ERROR);
            }
        }
    }

    /**
     * 获取订单物流状态
     *
     * @param trdOrderNo 第三方订单号
     * @return
     */
    @Override
    public List<OrderShippingPackageVo> getShipingTrace(String trdOrderNo) throws SupplierInterfaceInvokeException {
        Long time = System.currentTimeMillis();
        List<OrderShippingPackageVo> resultList = new ArrayList<>();
        HandlerRespVo<List<OrderShippingPackageVo>> handlerRespVo = new HandlerRespVo<>();
        handlerRespVo.setStatus(HandlerRespVo.RESPONSE_STATUS_SUCCESS);
        handlerRespVo.setErrorCode(0);
        if(!Lang.isEmpty(trdOrderNo)) {
            time = System.currentTimeMillis();
            OrderMainDto orderMain = orderMainApi.findOrderByThirdOrderNo(trdOrderNo);
            log.info("###########获取OrderMainDto############：{}ms", System.currentTimeMillis()-time);
            // 获取子订单
            if(orderMain != null) {
                //orderMain = orderMainApi.findById(orderMain.getId());
                for(SubOrderDto subOrder: orderMain.getSubOrders()) {
                    String subOrderNo = subOrder.getSubOrderNo();
                    time = System.currentTimeMillis();
                    List<SubOrderItemDto> subOrderItems = subOrder.getSubOrderItems();//orderMainApi.findSubOrderItemBySubOrderId(subOrder.getId());
                    log.info("###########获取OrderMainDto############：{}ms", System.currentTimeMillis()-time);
                    List<String> goodsIds = new ArrayList();
                    for (SubOrderItemDto subOrderItem : subOrderItems) {
                        if(!Lang.isEmpty(subOrderItem.getOrderItem())) {
                            goodsIds.add(subOrderItem.getOrderItem().getGoodsId());
                        }
                    }
                    time = System.currentTimeMillis();
                    List<GoodsPicture> goodsPictures = goodsPictureRepos.findByShowIndexAndGoodsIdIn(0, goodsIds);//goodsPictureRepos.findByGoods_IdIn(goodsIds);
                    log.info("###########获取goodsPictures############：{}ms", System.currentTimeMillis()-time);
                    List<GoodsInfoVo> goodsInfos = new ArrayList<>();
                    for (GoodsPicture picture : goodsPictures) {
                        GoodsInfoVo goodsInfo = new GoodsInfoVo();
                        goodsInfo.setGoodsId(picture.getGoods().getId());
                        goodsInfo.setGoodsName(picture.getGoods().getName());
                        goodsInfo.setGoodsPictrue(picture.getPicturePath());
                        goodsInfos.add(goodsInfo);
                    }
                    OrderShippingPackageVo orderShippingPackageVo = new OrderShippingPackageVo();
                    orderShippingPackageVo.setGoodsInfos(goodsInfos);
                    orderShippingPackageVo.setTrdSubOrderNo(subOrderNo);
                    orderShippingPackageVo.setPackageId(subOrderNo);
                    if(!Lang.isEmpty(subOrder.getTrdSpState())&& Lang.equals(subOrder.getTrdSpState(),SubOrderDto.TRD_SP_HAS_BEEN_PUT)){
                        orderShippingPackageVo.setReceiptsFlag(true);
                    }else{
                        orderShippingPackageVo.setReceiptsFlag(false);
                    }
                    List<OrderShippingTrackVo> trackInfoList = new ArrayList<>();

                    // 每个子订单号调用一次接口
                    JdOrderIdEntity jdOrderIdEntity = new JdOrderIdEntity();
                    jdOrderIdEntity.setJdOrderId(Long.parseLong(subOrderNo));
                    time = System.currentTimeMillis();
                    OrderTrackResp orderTrackResp = jdOrderApi.queryOrderTrack(jdOrderIdEntity);
                    log.info("###########调用京东物流接口############：{}ms", System.currentTimeMillis()-time);
                    OrdertrackResultVO result = orderTrackResp.getResult();
                    if(result != null) {
                        List<OrdertrackVO> ordertracks = result.getOrderTrack();

                        OrderShippingTrackVo orderShippingTrackVo;
                        for(OrdertrackVO track : ordertracks) {
                            orderShippingTrackVo = new OrderShippingTrackVo();
                            orderShippingTrackVo.setContent(track.getContent());
                            orderShippingTrackVo.setMsgTime(track.getMsgTime());
                            orderShippingTrackVo.setOperator(track.getOperator());
                            trackInfoList.add(orderShippingTrackVo);
                        }
                    }
                    orderShippingPackageVo.setTrackInfoList(trackInfoList);
                    resultList.add(orderShippingPackageVo);
                }
            }

            return resultList;
        } else {
            throw new SupplierInterfaceInvokeException(SupplierInterfaceInvokeException.ERROR_CODES.TRD_ORDER_NO_NOTFOUND);
        }
    }

    /**
     * 获取供应商订单状态
     *
     * @param trdOrderNo 第三方订单号
     * @return 返回的状态码为OutsideProductApi中，以TRD_ORDER_STATUS_开头的常量
     */
    @Override
    public String getOrderStatus(String trdOrderNo) throws SupplierInterfaceInvokeException {
        String status = null;
        if(!Lang.isEmpty(trdOrderNo)) {
            JdOrderIdEntity jdOrderIdEntity = new JdOrderIdEntity();
            jdOrderIdEntity.setJdOrderId(Long.parseLong(trdOrderNo));
            Map<String, Object> queryOrder = jdOrderApi.queryOrder(jdOrderIdEntity);
            if(queryOrder != null) {
                if("1".equals(queryOrder.get("submitState").toString())) {
                    // 已确认订单
                    status = OutsideProductApi.TRD_ORDER_STATUS_COMFIRMED;

                } else if("0".equals(queryOrder.get("orderState").toString())) {
                    // 正常订单
                    status = OutsideProductApi.TRD_ORDER_STATUS_CANCELED;
                } else if("1".equals(queryOrder.get("orderState").toString())) {
                    // 已取消订单
                    status = OutsideProductApi.TRD_ORDER_STATUS_NORMAL;
                }
            } else {
                throw new SupplierInterfaceInvokeException(SupplierInterfaceInvokeException.ERROR_CODES.TRD_ORDER_NO_NOTFOUND);

            }
        } else {
            throw new SupplierInterfaceInvokeException(SupplierInterfaceInvokeException.ERROR_CODES.TRD_ORDER_NO_NOTFOUND);

        }
        return status;
    }

    /**
     * 获取订单物流状态（针对诚通没有分包裹使用老代码）
     *
     * @param trdOrderNo 第三方订单号
     * @return
     */
    @Override
    public List<OrderShippingPackageVo> getCTShipingTrace(String trdOrderNo) throws SupplierInterfaceInvokeException {
        List<OrderShippingPackageVo> resultList = new ArrayList<>();
        if(!Lang.isEmpty(trdOrderNo)) {
            OrderMainDto orderMain = orderMainApi.findOrderByThirdOrderNo(trdOrderNo);
            // 获取子订单
            if(orderMain != null) {
                List<String> trdOrderNos = new ArrayList<>();
//                if(orderMain.getSubOrders().size() > 1) {
                // 拆单
                orderMain = orderMainApi.findById(orderMain.getId());
                for(SubOrderDto subOrder : orderMain.getSubOrders()) {
                    trdOrderNos.add(subOrder.getSubOrderNo());
                }
//                } else {
//                     不拆单
//                    trdOrderNos.add(orderMain.getThirdOrderNo());
//                }
                for(String subOrderNo : trdOrderNos) {
                    OrderShippingPackageVo orderShippingPackageVo = new OrderShippingPackageVo();
                    orderShippingPackageVo.setPackageId(subOrderNo);
                    List<OrderShippingTrackVo> trackInfoList = new ArrayList<>();

                    // 每个子订单号调用一次接口
                    JdOrderIdEntity jdOrderIdEntity = new JdOrderIdEntity();
                    jdOrderIdEntity.setJdOrderId(Long.parseLong(subOrderNo));
                    OrderTrackResp orderTrackResp = jdOrderApi.queryOrderTrack(jdOrderIdEntity);
                    OrdertrackResultVO result = orderTrackResp.getResult();
                    if(result != null) {
                        List<OrdertrackVO> ordertracks = result.getOrderTrack();

                        OrderShippingTrackVo orderShippingTrackVo;
                        for(OrdertrackVO track : ordertracks) {
                            orderShippingTrackVo = new OrderShippingTrackVo();
                            orderShippingTrackVo.setContent(track.getContent());
                            orderShippingTrackVo.setMsgTime(track.getMsgTime());
                            orderShippingTrackVo.setOperator(track.getOperator());
                            trackInfoList.add(orderShippingTrackVo);
                        }
                    }
                    orderShippingPackageVo.setTrackInfoList(trackInfoList);
                    resultList.add(orderShippingPackageVo);
                }
            }
            return resultList;
        } else {
            throw new SupplierInterfaceInvokeException(SupplierInterfaceInvokeException.ERROR_CODES.TRD_ORDER_NO_NOTFOUND);
        }
    }

    /**
     * 订单号反查接口
     *
     *
     * @param trdOrderKey@return 第三方供应商订单号
     */
    @Override
    public String queryTrdOrderNo(TrdOrderKey trdOrderKey) throws SupplierInterfaceInvokeException {
        ThirdOrder thirdOrder = new ThirdOrder();
        thirdOrder.setThirdOrder(trdOrderKey.getTrdOrderNo());
        try {
            OrderRevResp orderRevResp = jdOrderApi.queryOrderByThirdOrder(thirdOrder);
            return orderRevResp.getResult();
        } catch (Exception e) {
            throw new SupplierInterfaceInvokeException(
                    e.getMessage(),
                    SupplierInterfaceInvokeException.ERROR_CODES.NETWORK_CONNECTION_FAILD, e
            );
        }
    }

    /**
     * 向第三方供应商下单
     *
     * @param orderRequst 商城订单信息
     * @return 返回生产成功的第三方订单详情
     * @throws SupplierInterfaceInvokeException 供应商接口调用异常
     * @throws SupplierTakeOrderFalidException  供应商下单失败异常
     */
    @Override
    public TrdOrder takeOrder(OrderRequst orderRequst) throws SupplierInterfaceInvokeException, SupplierTakeOrderFalidException {
        Map<String, Object> returnMap = new HashMap<>();

        // 检查供应商下单预警配置，如被拦截，抛出异常
        checkOrderWanner();

        try {
            //1.封装京东下单实体
            OrderEntity orderEntity = new OrderEntity();
            MemberAddressDto memberAddress = memberAddressApi.findById(orderRequst.getOrderReceiver().getId());


            orderEntity.setThirdOrder(orderRequst.getOrderNo());//第三方订单号 Y-
            orderEntity.setSku(cpySkuList(orderRequst.getOrderItems()));//商品列表 Y-
            orderEntity.setName(memberAddress.getUsername());//下单人姓名 Y-

            Map<Integer, String> comAddrMap = new HashMap<>();
            comAddrMap.put(AddressCommonApi.AddressLevel.PROVINCE.getValue(), memberAddress.getProvinceCode());
            comAddrMap.put(AddressCommonApi.AddressLevel.CITY.getValue(), memberAddress.getCityCode());
            comAddrMap.put(AddressCommonApi.AddressLevel.AREA.getValue(), memberAddress.getAreaCode());
            comAddrMap.put(AddressCommonApi.AddressLevel.TOWN.getValue(), memberAddress.getTownCode());
            Long addrTime = System.currentTimeMillis();
            HandlerRespVo<Map<Integer, String>> jdAddrResp = outsideProductApi.transAreaCodeMap(orderRequst.getStoreCode(), comAddrMap);
            log.info("##转换为京东地址##：{}ms",System.currentTimeMillis() - addrTime);
            if (jdAddrResp.getStatus() != HandlerRespVo.RESPONSE_STATUS_SUCCESS) {
                throw new SupplierTakeOrderFalidException(
                        SupplierTakeOrderFalidException.ERROR_CODES.ADDRESS_TRANS_ERROR
                );
            }

            Map<Integer, String> jdAddrMap = jdAddrResp.getData();

            try {
                orderEntity.setProvince(Integer.parseInt(jdAddrMap.get(AddressCommonApi.AddressLevel.PROVINCE.getValue())));//省 Y-
                orderEntity.setCity(Integer.parseInt(jdAddrMap.get(AddressCommonApi.AddressLevel.CITY.getValue())));//市 Y-
                String countyCode = jdAddrMap.get(AddressCommonApi.AddressLevel.AREA.getValue());
                if(!Lang.isEmpty(countyCode)){
                    orderEntity.setCounty(Integer.parseInt(countyCode));//乡 Y-
                }else{
                    orderEntity.setCounty(NationalAddressDto.DEFULT_ADDRESS_VALUE);//乡 Y-
                }
                String townCode = jdAddrMap.get(AddressCommonApi.AddressLevel.TOWN.getValue());
                if(!Lang.isEmpty(townCode)) {
                    orderEntity.setTown(Integer.parseInt(townCode));//镇-
                }else{
                    orderEntity.setTown(NationalAddressDto.DEFULT_ADDRESS_VALUE);//镇-
                }
                orderEntity.setAddress(memberAddress.getAddressDetail());//详细地址 Y-
                orderEntity.setMobile(memberAddress.getPhone());//手机号码 Y-
            } catch (Exception ex) {
                throw new SupplierTakeOrderFalidException(
                        SupplierTakeOrderFalidException.ERROR_CODES.ADDRESS_TRANS_ERROR, ex
                );
//                return returnMap;
            }
            gennerJdOrderConstant(orderEntity);
            orderEntity.setOrderPriceSnap(generateOrderPriceSnap(orderRequst));//价格快照 Y
            final String orderNo = orderRequst.getOrderNo();
            //2.预占库存
            Long stockTime = System.currentTimeMillis();
            String takeOrderFlag = systemConfigApi.getConfigValue(SystemConfigDto.ORG_CODE_COMMON,
                    SystemConfigDto.JD_ORDER_CONFIG, SystemConfigValueDto.CONFIG_VALUE_TYPE_TAKE_ORDER_FLAG,
                    true, true);
            OrderUniteSubmitResp orderUniteSubmitResp = null;
            Boolean isTakeOrderFlag=true;
            if(!Lang.isEmpty(takeOrderFlag) && takeOrderFlag.equals("0")) {
                orderUniteSubmitResp = new OrderUniteSubmitResp();
                orderUniteSubmitResp.setSuccess(true);
                orderUniteSubmitResp.setResultMessage("SUCCESS");
                ParentOrderRepVO reselt = new ParentOrderRepVO();
                reselt.setFreight(BigDecimal.ZERO);
                reselt.setJdOrderId((long) (Math.random() * 1000L + System.currentTimeMillis()));
                reselt.setOrderPrice(orderRequst.getSumNofreightPrice());
                orderUniteSubmitResp.setResult(reselt);
                isTakeOrderFlag=false;

            } else {
                orderUniteSubmitResp = jdOrderApi.placeOrder(orderEntity);
            }
            log.info("##预占库存##：{}ms", System.currentTimeMillis()-stockTime);

            if (!orderUniteSubmitResp.getSuccess()|| Lang.isEmpty(orderUniteSubmitResp.getResult())) {
                //2.1预占库存失败
                //2.1.1获取所有商品的京东商品编号
                final List<String> skuIds = new ArrayList<>();
                List<SkuIdAndNumEntity> sku = orderEntity.getSku();
                for (SkuIdAndNumEntity skuIdAndNumEntity : sku) {
                    skuIds.add(skuIdAndNumEntity.getId().toString());
                }
                //原因1，价格快照校验失败
                if ("3019".equals(orderUniteSubmitResp.getResultCode())) {
                    throw new SupplierTakeOrderFalidException(
                            SupplierTakeOrderFalidException.ERROR_CODES.CREATE_ORDER_COST_CHANGED
                    );
                } else {
                    //其他原因
                    throw new SupplierTakeOrderFalidException(
                            orderUniteSubmitResp.getResultMessage(),
                            SupplierTakeOrderFalidException.ERROR_CODES.UNKNOWN
                    );
                }
                //TODO: 供应商适配器在处理时不在进行订单取消，需由上层应用捕捉异常后自行处理业务逻辑回滚！
                //2.1.2取消订单
//                orderRepos.updateStatusByOrderNo(OrderMain.order_status_error, orderNo);
//                updateGoodsSaleCount(OrderMain.order_status_error,orderNo);
            } else {
                //2.2预占库存成功
                //2.2.1建立orderMain和jdOrderId的关联
//                Long sucTime = System.currentTimeMillis();
//                orderRepos.updateThirdOrderNoById(orderRequst.getId(), orderUniteSubmitResp.getResult().getJdOrderId().toString(), orderUniteSubmitResp.getResult().getOrderPrice());
                //2.2.2修改订单状态为待支付状态,防止重复确认,重复扣积分
//                orderRepos.updateStatusByOrderNo(OrderMain.order_status_obligation, orderNo);
//                updateGoodsSaleCount(OrderMain.order_status_obligation,orderNo);
                //2.2.3返回下单成功的响应vo
                //TODO: 供应商适配器在处理时不进行订单的相关操作，需由上层应用进行处理
                TrdOrderKey trdOrderKey = new TrdOrderKey( orderUniteSubmitResp.getResult().getJdOrderId().toString(),orderRequst.getStoreCode());

                if(isTakeOrderFlag) {
                    return this.getDetail(trdOrderKey);
                }else{
                    return  null;
                }
//                log.info("##下单成功更新##：{}ms", System.currentTimeMillis() - sucTime);
            }
        } catch (Exception e) {
            log.error("京东下单报错[订单号：{}，第三方订单号：{}]",
                    orderRequst == null ? null : orderRequst.getOrderNo(),
                    orderRequst == null ? null : orderRequst.getThirdOrderNo()
                    , e);
            returnMap.put("returnCode", "99");
            returnMap.put("returnMsg", e.getMessage());

            throw new SupplierTakeOrderFalidException(
                    e.getMessage(),
                    SupplierTakeOrderFalidException.ERROR_CODES.UNKNOWN, e
            );


        }
//        return returnMap;
    }

    /**
     * 确认下单，调用时记得处理所有的异常噢！
     *
     * @param orderMain 商城订单信息
     * @throws SupplierInterfaceInvokeException   供应商接口调用异常
     * @throws SupplierOrderComfirmFalidException 供应商下单失败异常，其中错误代码3103表示已确认下单无需重复确认
     */
    @Override
    public void confirmOrder(OrderRequst orderMain) throws SupplierInterfaceInvokeException, SupplierOrderComfirmFalidException {
        Map returnMap = new HashMap();
        try {
            OccupyStockResp occupyStockResp = jdOrderApi.confirmOccupyStock(new JdOrderIdEntity(Long.parseLong(orderMain.getThirdOrderNo())));
            if(!occupyStockResp.getSuccess()){
                throw new SupplierOrderComfirmFalidException(
                        occupyStockResp.getResultMessage()
                        , SupplierOrderComfirmFalidException.ERROR_CODES.SUPPLIER_COMFIRMED_ERROR);
            }
        }catch (Exception e){
            throw new SupplierInterfaceInvokeException(
                    SupplierInterfaceInvokeException.ERROR_CODES.NETWORK_CONNECTION_FAILD,
                    e
            );
        }
    }

    /**
     * 第三方退货
     *
     * @param orderMain 商城订单信息
     * @param skuIds    退货的商品
     * @return
     * @throws SupplierInterfaceInvokeException
     */
    @Override
    public BackOrderResp backOrder(OrderRequst orderMain, List<String> skuIds) throws SupplierInterfaceInvokeException {
        return null;
    }


    /**
     * 获取订单详情
     *
     * @param trdOrderKey 第三方订单唯一标识，采用storeCode + trdOrderNo作为唯一标识
     * @return 返回第三方订单详情
     * @throws SupplierInterfaceInvokeException 供应商接口链接异常
     */
    @Override
    public TrdOrder getDetail(TrdOrderKey trdOrderKey) throws SupplierInterfaceInvokeException {
        long start = System.currentTimeMillis();
        log.info("start splitOrder...");
        //1.根据订单号查询订单详细信息 jdOrder pOrder cOrder pSku cSku
        Map<String, Object> queryOrder = jdOrderApi.queryOrder(new JdOrderIdEntity(Long.parseLong(trdOrderKey.getTrdOrderNo())));
        JdOrder jdOrder;
        TrdOrder trdOrder = null;
        //2.可能返回子订单(不拆单)也可能返回父订单(拆单)
        if(queryOrder != null) {
            if (queryOrder.get("type").equals(2)) {
                // type=2 子订单
                queryOrder.put("pOrder", null);
                String jsonString = JSON.toJSONString(queryOrder, true);
                Corder corder = JSON.parseObject(jsonString, Corder.class);
                trdOrder = JdOrderUtil.createTrdOrderByCorder(corder);
            } else {
                // type=1 父订单
                String jsonString = JSON.toJSONString(queryOrder, true);
                jdOrder = JSON.parseObject(jsonString, JdOrder.class);
                trdOrder = JdOrderUtil.createTrdOrderByJdOrder(jdOrder);
            }

        }
        log.info("splitOrder end... cost {} ms", System.currentTimeMillis() - start);
        // 处理订单状态
        trdOrder.setFinalStatus(TrdOrderHandlerUtils.judteStatus(trdOrder));
        return trdOrder;
    }

    /**
     * 检查京东配置下单预警，直接进行拦截
     * @return
     */
    private void checkOrderWanner() throws SupplierTakeOrderFalidException {
        String jdOrderWarnStr = systemConfigApi.getConfigValue(SystemConfigDto.ORG_CODE_COMMON,
                SystemConfigDto.CONFIG_CODE_JD_ORDER_CONFIG_WARN, SystemConfigValueDto.CONFIG_VALUE_TYPE_JD_ORDER_CONSTANT_WARN,
                true, true);
        if(!Lang.isEmpty(jdOrderWarnStr)) {
            if(!TrdOrderHandlerUtils.checkOrderWannerConfigJson(jdOrderWarnStr)) {
                throw new SupplierTakeOrderFalidException("下单拦截已配置", SupplierTakeOrderFalidException.ERROR_CODES.TAKE_ORDER_INTERCEPTED);
            }
        }
    }

    //京东下单参数使用配置化
    private void gennerJdOrderConstant(OrderEntity orderEntity){
        String jdOrderConstantStr = systemConfigApi.getConfigValue(SystemConfigDto.ORG_CODE_COMMON,
                SystemConfigDto.CONFIG_CODE_JD_ORDER_CONFIG, SystemConfigValueDto.CONFIG_VALUE_TYPE_JD_ORDER_CONSTANT,
                true, true);
        if(!Lang.isEmpty(jdOrderConstantStr)) {
            try{
                JSONObject obj = JSON.parseObject(jdOrderConstantStr);

                Integer invoiceState = (Integer) obj.get("invoiceState");
                Integer invoiceType = (Integer) obj.get("invoiceType");
                Integer selectedInvoiceTitle = (Integer) obj.get("selectedInvoiceTitle");
                String invoiceName = (String) obj.get("invoiceName");
                Integer invoiceContent = (Integer) obj.get("invoiceContent");
                Integer paymentType = (Integer) obj.get("paymentType");
                Integer isUseBalance = (Integer) obj.get("isUseBalance");
                Integer submitState =(Integer) obj.get("submitState");
                Integer doOrderPriceMode = (Integer) obj.get("doOrderPriceMode");
                String email = (String) obj.get("email");
                orderEntity.setEmail(email);//邮箱 Y-
                orderEntity.setInvoiceState(invoiceState);
                orderEntity.setInvoiceType(invoiceType);
                orderEntity.setSelectedInvoiceTitle(selectedInvoiceTitle);
                orderEntity.setInvoiceName(invoiceName);
                orderEntity.setInvoiceContent(invoiceContent);
                orderEntity.setPaymentType(paymentType);
                orderEntity.setIsUseBalance(isUseBalance);
                orderEntity.setSubmitState(submitState);
                orderEntity.setDoOrderPriceMode(doOrderPriceMode);
            } catch (Exception e) {
                log.info("数据库保存的参数信息是错误的，请修改");
                e.printStackTrace();
                //defaultGennerJdOrderConstant(orderEntity);
            }

        }else {//系统参数未进行配置的时候，使用默认的，避免下单报错
            defaultGennerJdOrderConstant(orderEntity);
        }
    }

    private void defaultGennerJdOrderConstant(OrderEntity orderEntity){
        orderEntity.setEmail(JdOrderConstant.email);//邮箱 Y-
        orderEntity.setInvoiceState(JdOrderConstant.invoiceState);
        orderEntity.setInvoiceType(JdOrderConstant.invoiceType);
        orderEntity.setSelectedInvoiceTitle(JdOrderConstant.selectedInvoiceTitle);
        orderEntity.setInvoiceName(JdOrderConstant.invoiceName);
        orderEntity.setInvoiceContent(JdOrderConstant.invoiceContent);
        orderEntity.setPaymentType(JdOrderConstant.paymentType);
        orderEntity.setIsUseBalance(JdOrderConstant.isUseBalance);
        orderEntity.setSubmitState(JdOrderConstant.submitState);
        orderEntity.setDoOrderPriceMode(JdOrderConstant.doOrderPriceMode);
    }

    /**
     * 从OrderMain拷贝出商品列表
     *
     * @param orderItems
     * @return
     */
    List<SkuIdAndNumEntity> cpySkuList(List<OrderItemVo> orderItems) {
        List<SkuIdAndNumEntity> retSku = new ArrayList<>();
        for (OrderItemVo orderItem : orderItems) {
            //第三方商品编号
            Long skuId = Long.parseLong(orderItem.getSku());
            //商品数量
            Integer num = orderItem.getCount();
            retSku.add(new SkuIdAndNumEntity(skuId, num));
        }
        return retSku;
    }

    /**
     * 生成下单时的价格快照
     *
     * @param orderMain
     * @return
     */
    String generateOrderPriceSnap(OrderRequst orderMain) {
        List<OrderPriceSnap> snaps = new ArrayList<>();
        List<OrderItemVo> orderItems = orderMain.getOrderItems();
        for (OrderItemVo orderItem : orderItems) {
            Long skuId = Long.parseLong(orderItem.getSku());
            snaps.add(new OrderPriceSnap(orderItem.getCostPrice(), skuId));
        }
        return JSON.toJSONString(snaps);
    }
    
    private ApplicationContext applicationContext;

    @Autowired
    JdProductApiStub jdProductApi;

    @Autowired
    JdOrderApiStub jdOrderApi;

    @Autowired
    OrderMainApiStub orderMainApi;

    @Autowired
    NationalAddressRepos nationalAddressRepos;

    @Autowired
    AddressCommonApiStub addressCommonApi;

    @Autowired
    GoodsPictureRepos goodsPictureRepos;

    @Autowired
    OutsideProductApi outsideProductApi;

    @MotanReferer
    MemberAddressApi memberAddressApi;

    @MotanReferer
    SystemConfigApi systemConfigApi;


    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
