package outsideapi.service.handler.impl;

import cart.api.dto.payment.PaymentDto;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import outsideapi.service.handler.OutsideOrderApiHandler;
import outsideapi.service.service.stub.PaymentApiStub;
import outsideapi.service.service.stub.WnOrderApiStub;
import outsideapi.utils.Constants;
import outsideapi.vo.BackOrderResp;
import outsideapi.vo.CancelOrderResp;
import outsideapi.vo.ConfirmOrderResp;
import outsideapi.vo.TakeOrderResp;
import outsideapi.vo.order.OrderMainDto;
import outsideapi.vo.order.TrdOrderResp;
import recharge.center.api.weineng.api.definition.ErrorInfo;
import recharge.center.api.weineng.api.request.OrderEntity;
import recharge.center.api.weineng.api.response.OrderResp;
import utils.Lang;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

/**
 * 微能订单相关接口处理实现
 *
 * @author Liang Wenxu
 * @since 2018-06-12
 */
@Slf4j
@Component
@Transactional(readOnly = true, rollbackFor = Exception.class)
public class WnOutsideOrderApiHandler implements OutsideOrderApiHandler {

    @Override
    public CancelOrderResp cancelOrder(String trdOrderNo, String storeId) {
        CancelOrderResp cancelOrderResp = new CancelOrderResp();
        cancelOrderResp.setCode("0");
        cancelOrderResp.setMessage("success");
        cancelOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        return cancelOrderResp;
    }

    @Override
    public TakeOrderResp takeOrder(OrderMainDto orderMain) {
        // 当前方法返回对象
        TakeOrderResp takeOrderResp = new TakeOrderResp();

        // 封装下单请求参数
        PaymentDto payment = paymentApi.findOne(orderMain.getPaymentId());
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
        OrderEntity orderEntity = new OrderEntity();
        orderEntity.setCustomerNo(orderMain.getOrderItems().get(0).getBusinessCode());
        orderEntity.setCardBatchNo(orderMain.getOrderItems().get(0).getSku().split("_")[1]);
        orderEntity.setOrderId(orderMain.getOrderNo());
        orderEntity.setSubmitTimestamp(sdf.format(new Date()));
        orderEntity.setOrderQuantity(1);
        orderEntity.setOrderTime(sdf.format(new Date()));
        orderEntity.setOrderPoints(payment.getMoney().multiply(new BigDecimal("100")).intValue());
        log.info("微能下单请求参数 : {}", JSON.toJSONString(orderEntity));

        // 调用接口下单
        OrderResp orderResp;
        try {
            long takeOrderTime = System.currentTimeMillis();
            log.info("正在调用微能供应商接口下单...");
            orderResp = orderApi.takeOrder(orderEntity);
            log.info("调用微能供应商接口耗时 {} ms", System.currentTimeMillis() - takeOrderTime);
            log.info("调用微能供应商接口返回数据 : {}", orderResp);
            if(Lang.isEmpty(orderResp)) {
                throw new RuntimeException("空的响应对象");
            }
        } catch (Exception e) {
            log.error("调用微能供应商下单接口异常", e);
            takeOrderResp.setReturnCode("1");
            takeOrderResp.setReturnMsg("调用微能供应商下单接口异常 : " + e.getMessage());
            takeOrderResp.setStatus(Constants.ResponseStatus.ERROR);
            return takeOrderResp;
        }

        // 成功响应代码
        final String successCode = "0000";

        if (successCode.equals(orderResp.getRetResult())) {
            // 供应商成功下单
            outsideapi.vo.OrderResp returnOrderResp = new outsideapi.vo.OrderResp();
            returnOrderResp.setTrdOrderNo(orderResp.getWnOrderId());
            takeOrderResp.setOrderResp(returnOrderResp);
            takeOrderResp.setReturnCode("0");
            takeOrderResp.setReturnMsg("微能供应商下单成功");
            takeOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        } else {
            // 供应商
            takeOrderResp.setReturnCode("2");
            takeOrderResp.setReturnMsg("微能供应商下单失败. " + orderResp.getRetResult() + " : " + ErrorInfo.errorMap.get(orderResp.getRetResult()));
            takeOrderResp.setStatus(Constants.ResponseStatus.ERROR);
        }

        log.info("方法结束返回对象 : {}", JSON.toJSONString(takeOrderResp));
        return takeOrderResp;
    }

    @Override
    public ConfirmOrderResp confirmOrder(String trdOrderNo, String storeId) {
        ConfirmOrderResp confirmOrderResp = new ConfirmOrderResp();
        confirmOrderResp.setCode("0");
        confirmOrderResp.setMessage("SUCCESS");
        confirmOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        return confirmOrderResp;
    }

    @Override
    public BackOrderResp backOrder(OrderMainDto orderMain, List<String> skuIds) {
        BackOrderResp backOrderResp = new BackOrderResp();
        backOrderResp.setCode("0");
        backOrderResp.setMessage("SUCCESS");
        backOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        return backOrderResp;
    }

    @Override
    public String queryTrdOrderNo(String orderNo) {
        return orderNo + "_TRD";
    }

    @Override
    public TrdOrderResp queryOrder(String trdOrderNo) {
        TrdOrderResp queryOrderResp = new TrdOrderResp();
        queryOrderResp.setCode("0");
        queryOrderResp.setMessage("SUCCESS");
        queryOrderResp.setStatus(Constants.ResponseStatus.SUCCESS);
        return queryOrderResp;
    }

    private final WnOrderApiStub orderApi;

    private final PaymentApiStub paymentApi;

    @Autowired
    public WnOutsideOrderApiHandler(WnOrderApiStub orderApi, PaymentApiStub paymentApi) {
        this.orderApi = orderApi;
        this.paymentApi = paymentApi;
    }
}

