package outsideapi.service.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import outsideapi.exceptions.SupplierTakeOrderFalidException;
import outsideapi.vo.order.TrdOrder;
import utils.GlobalContants;

import java.util.ArrayList;
import java.util.List;

/**
 * 供应商订单适配处理器公用工具类
 * 供应商订单适配处理器所用到的一些公共方法可抽离到此工具类中
 * @author Liang Wenxu
 * @since 2018/8/14
 */
@Slf4j
public class TrdOrderHandlerUtils {

    /**
     * 校验供应商下单预警配置Json串，已配置拦截时根据配置情况抛出异常
     * @param configJson 配置字符串（JSON格式）
     * @return 返回true时可正常下单，返回false或抛出异常则为下单拦截
     * @throws SupplierTakeOrderFalidException 已配置拦截或配置项有问题时，抛出相应异常，异常代码详见{@link SupplierTakeOrderFalidException.ERROR_CODES}
     */
    public static Boolean checkOrderWannerConfigJson(String configJson) throws SupplierTakeOrderFalidException {
        try{
            JSONObject obj = JSON.parseObject(configJson);
            Boolean isWarn = (Boolean) obj.get("isWarn");
            String message = (String) obj.get("message");
            if(isWarn){
                throw new SupplierTakeOrderFalidException(message, SupplierTakeOrderFalidException.ERROR_CODES.TAKE_ORDER_INTERCEPTED);
            }
        } catch (Exception e) {
            log.info("数据库保存的参数信息是错误的，请修改");
            throw new SupplierTakeOrderFalidException(e.getMessage(), SupplierTakeOrderFalidException.ERROR_CODES.ILLEGAL_TAKE_ORDER_INTERCEPT_CONFIG, e);
        }

        return true;
    }

    /**
     * 有一个子订单新建，则整体是新建，全部妥投则算作妥投，否则是取消
     * @param trdOrder
     * @return
     */
    public static GlobalContants.ORDER_STATUS judteStatus(TrdOrder trdOrder){
        List<TrdOrder> corders = trdOrder.getCorders();
        List<Integer> states = new ArrayList<Integer>();
        if(corders != null && corders.size() > 0) {
            for (TrdOrder corder : corders) {
                states.add(corder.getState());
            }
        } else {
            states.add(trdOrder.getState());
        }

        Boolean isConfirm = null;
        Boolean isRefuse = false;
        GlobalContants.ORDER_STATUS confirmState;
        for (Integer state : states) {
            switch (state) {
                case TrdOrder.STATE_NEW: //包含未收货子单，已收货标志为false
                    isConfirm = false;
                    break;
                case TrdOrder.STATE_FINISHED: //所有子单是否都为true
                    isConfirm = (isConfirm == null || (isConfirm)) && isConfirm;
                    break;
                case TrdOrder.STATE_REJECT: //包含拒收子单，已收货标志为false
                    isConfirm = false;
                    isRefuse = true;
                    break;
            }
        }

        if(trdOrder.getSubmitstate() != TrdOrder.SUBMIT_STATE_SUBMITED) {
            // 未确认下单
            return GlobalContants.ORDER_STATUS.NEWORDER;
        }

        if(trdOrder.getOrderstate() == TrdOrder.ORDER_STATE_CANCELED) {
            // 供应商订单取消
            return GlobalContants.ORDER_STATUS.OUTOFTIME_CANCEL;
        }

        if(isRefuse) {
            // 存在任意已拒收子订单，订单状态为拒收
            return GlobalContants.ORDER_STATUS.REFUSE;
        } else {
            if(isConfirm) {
                // 所有子订单都是已收货，主订单状态已收货
                return GlobalContants.ORDER_STATUS.EVALUATE;
            } else {
                // 存在任意未收货子订单，订单状态为待收货
                return GlobalContants.ORDER_STATUS.OUNFILLED;
            }
        }
    }
}
