package spider.service.handler.impl;

import com.alibaba.fastjson.JSON;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import ma.glasnost.orika.MapperFacade;
import org.apache.http.HttpEntity;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import spider.dto.SpiderLogsDto;
import spider.model.SpiderLogs;
import spider.service.handler.PriceSpiderHandler;
import spider.service.service.SpiderLogsService;
import store.api.StoreApi;
import store.api.dto.modeldto.core.StoreDto;
import sysmg.api.SystemConfigApi;
import sysmg.dto.SystemConfigDto;
import utils.GlobalContants;
import utils.Lang;
import utils.log.Log;
import utils.log.Logs;
import utils.rpc.motan.ApiResponseVo;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 京东价格爬虫API
 * @author Liang Wenxu
 * @since 2017-07-04
 * Created by Liang Wenxu on 2017-07-04.
 */
@Component
public class JdPriceSpiderHandlerImpl implements PriceSpiderHandler {

    Log logs = Logs.getLog(JdPriceSpiderHandlerImpl.class.getName());

    /** Attributes ============== */
    private static StoreDto store;


    @Value("${spider.httpclient.request.connectTimeout:3000}")
    private Integer connectTimeout;

    @Value("${spider.httpclient.request.connectionRequestTimeout:3000}")
    private Integer connectionRequestTimeout;

    @Value("${spider.httpclient.request.socketTimeout:5000}")
    private Integer socketTimeout;

    @Autowired
    MapperFacade mapperFacade;

    /** Methods ============== */

    /**
     * {@inheritDoc}
     */
    @Override
    public ApiResponseVo<SpiderLogsDto> getSalesPrice(String sku) throws IOException {
        return this.getSalesPrice(sku, null, SpiderLogs.TRIGGER_TYPE_MANUAL);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ApiResponseVo<Map<String, SpiderLogsDto>> getBatchSalesPrice(List<String> skus) throws IOException {
        return this.getBatchSalesPrice(skus, null, SpiderLogs.TRIGGER_TYPE_MANUAL);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ApiResponseVo<SpiderLogsDto> getSalesPrice(String sku, Map<Integer, String> areaCodeMap, String triggerType) throws IOException {
        List<String> skus = new ArrayList<>();
        skus.add(sku);
        ApiResponseVo<Map<String, SpiderLogsDto>> batchRes =  getBatchSalesPrice(skus, areaCodeMap, triggerType);
        if(batchRes.getStatus().equals(GlobalContants.ResponseStatus.SUCCESS)) {
            return ApiResponseVo.createSucResp(batchRes.getData().get(sku), "suc", "0");
        } else {
            return ApiResponseVo.createErrorResp(batchRes.getMessage(), batchRes.getCode(), null);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ApiResponseVo<Map<String, SpiderLogsDto>> getBatchSalesPrice(List<String> skus, Map<Integer, String> areaCodeMap, String triggerType) throws IOException {
        ApiResponseVo responseVo = null;
        String priceUrl = systemConfigApi.getConfigValue(
                SystemConfigDto.ORG_CODE_COMMON, SystemConfigDto.CONFIG_CODE_JD_SPIDER_CONFIGS,
                "JD_PRICE_URL", true, true);
        if(!Lang.isEmpty(priceUrl)) {
            StringBuilder skuIdSb = new StringBuilder();
            for(String sku : skus) {
                skuIdSb.append(",").append("J_").append(sku);
            }
            priceUrl = priceUrl.replace("{sku}", skuIdSb.substring(1));
            StringBuilder addressStr = new StringBuilder();
            if(areaCodeMap != null) {
                for(int lv = 1; lv <= 4; lv++) {
                    if(areaCodeMap.get(lv) != null) {
                        addressStr.append(areaCodeMap.get(lv));
                    } else {
                        addressStr.append("0");
                    }

                    if(lv != 4) {
                        addressStr.append("_");
                    }
                }
            }

            RequestConfig requestConfig = RequestConfig.custom()
                    .setConnectTimeout(connectTimeout).setConnectionRequestTimeout(connectionRequestTimeout)
                    .setSocketTimeout(socketTimeout).build();

            priceUrl = priceUrl.replace("{area}", addressStr.toString());
            CloseableHttpClient httpclient = HttpClients.createDefault();
            HttpGet httpGet = new HttpGet(priceUrl);
            httpGet.setConfig(requestConfig);
            CloseableHttpResponse response = null;

            Map<String, SpiderLogsDto> returnData = new HashMap<>();

            Long time = System.currentTimeMillis();
            try {
                response = httpclient.execute(httpGet);
                System.out.println(response.getStatusLine());
                if(response.getStatusLine().getStatusCode() == 200) {
                    HttpEntity entity = response.getEntity();
                    String contentJson = EntityUtils.toString(entity);
                    System.out.println("JD Spider HTTP Response: " + contentJson);
                    boolean sucFlag = false;
                    try {
                        List<Map> jsonArray = JSON.parseArray(contentJson, Map.class);
                        for(Map m : jsonArray) {
                            SpiderLogs spLog = new SpiderLogs();
                            spLog.setTriggerType(Lang.isEmpty(triggerType) ? SpiderLogs.TRIGGER_TYPE_MANUAL : triggerType);
                            spLog.setProductCode(((String) m.get("id")).toUpperCase().replace("J_", ""));
                            spLog.setSalesPrice(Lang.isEmpty(m.get("p")) ? BigDecimal.ZERO : new BigDecimal(m.get("p").toString()));
                            spLog.setStoreId(getStore().getId());
                            spLog.setContent(contentJson);
                            spiderLogsService.saveLog(spLog);
                            returnData.put(spLog.getProductCode(), mapperFacade.map(spLog,SpiderLogsDto.class));
                        }
                        sucFlag = true;
                        responseVo = ApiResponseVo.createSucResp(returnData, "suc", "0");
                    } catch (Exception e) {
                        e.printStackTrace();
                        sucFlag = false;
                    }

                    if(!sucFlag) {
                        responseVo = ApiResponseVo.createErrorResp("抓取京东数据失败，该sku无法抓取到京东数据", "2", "" + contentJson);
                    }
                }
            } catch (ClientProtocolException e) {
                e.printStackTrace();
                responseVo = ApiResponseVo.createErrorResp("抓取京东数据失败："+e.getLocalizedMessage(), "3", Lang.toString(e));
            } catch (IOException e) {
                e.printStackTrace();
                responseVo = ApiResponseVo.createErrorResp("抓取京东数据失败："+e.getLocalizedMessage(), "3", Lang.toString(e));
            } finally {
                if(response != null) {
                    response.close();
                }
            }
            logs.info("销售价格接口（京东）时长：{}ms", System.currentTimeMillis()-time);
        } else {
            responseVo = ApiResponseVo.createErrorResp("JD_PRICE_URL未配置", "1", null);
        }
        return responseVo;
    }

    // 获取店铺（写死当前的店铺code即可）
    private StoreDto getStore() {
        if(store == null) {
            store = storeApi.findByCode("jd");
        }
        return store;
    }

    /** Dependent Components ============== */
    @MotanReferer
    SystemConfigApi systemConfigApi;

    @Autowired
    SpiderLogsService spiderLogsService;

    @MotanReferer
    StoreApi storeApi;
    /** Setter and getters ============== */
}
