/*
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License.  You may obtain a
 * copy of the License at the following location:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jasig.cas.client.authentication;

import org.jasig.cas.client.Protocol;
import org.jasig.cas.client.configuration.ConfigurationKeys;
import org.jasig.cas.client.util.AbstractCasFilter;
import org.jasig.cas.client.util.CommonUtils;
import org.jasig.cas.client.util.ReflectUtils;
import org.jasig.cas.client.validation.Assertion;

import javax.imageio.ImageIO;
import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.HashMap;
import java.util.Map;

/**
 * Filter implementation to intercept all requests and attempt to authenticate
 * the user by redirecting them to CAS (unless the user has a ticket).
 * <p>
 * This filter allows you to specify the following parameters (at either the context-level or the filter-level):
 * <ul>
 * <li><code>casServerLoginUrl</code> - the url to log into CAS, i.e. https://cas.rutgers.edu/login</li>
 * <li><code>renew</code> - true/false on whether to use renew or not.</li>
 * <li><code>gateway</code> - true/false on whether to use gateway or not.</li>
 * </ul>
 * <p/>
 * <p>Please see AbstractCasFilter for additional properties.</p>
 *
 * @author Scott Battaglia
 * @author Misagh Moayyed
 * @since 3.0
 */
public class AuthenticationFilter extends AbstractCasFilter {
    /**
     * The URL to the CAS Server login.
     */
    private String casServerLoginUrl;

    /**
     * Whether to send the renew request or not.
     */
    private boolean renew = false;

    private String clientSourceTarget;

    private String casServerUrlPrefix;

    private String ajaxLoginCallBackUrlPrefix;

    /**
     * Whether to send the gateway request or not.
     */
    private boolean gateway = false;

    private GatewayResolver gatewayStorage = new DefaultGatewayResolverImpl();

    private AuthenticationRedirectStrategy authenticationRedirectStrategy = new DefaultAuthenticationRedirectStrategy();

    private UrlPatternMatcherStrategy ignoreUrlPatternMatcherStrategyClass = null;

    private static final Map<String, Class<? extends UrlPatternMatcherStrategy>> PATTERN_MATCHER_TYPES =
            new HashMap<String, Class<? extends UrlPatternMatcherStrategy>>();

    static {
        PATTERN_MATCHER_TYPES.put("CONTAINS", ContainsPatternUrlPatternMatcherStrategy.class);
        PATTERN_MATCHER_TYPES.put("REGEX", RegexUrlPatternMatcherStrategy.class);
        PATTERN_MATCHER_TYPES.put("EXACT", ExactUrlPatternMatcherStrategy.class);
    }

    public AuthenticationFilter() {
        this(Protocol.CAS2);
    }

    protected AuthenticationFilter(final Protocol protocol) {
        super(protocol);
    }

    public void setCasServerUrlPrefix(String casServerUrlPrefix) {
        this.casServerUrlPrefix = casServerUrlPrefix;
    }

    public void setAjaxLoginCallBackUrlPrefix(String ajaxLoginCallBackUrlPrefix) {
        this.ajaxLoginCallBackUrlPrefix = ajaxLoginCallBackUrlPrefix;
    }

    protected void initInternal(final FilterConfig filterConfig) throws ServletException {
        if (!isIgnoreInitConfiguration()) {
            super.initInternal(filterConfig);
            setCasServerLoginUrl(getString(ConfigurationKeys.CAS_SERVER_LOGIN_URL));
            //设置业务来源
            setClientSourceTarget(filterConfig.getInitParameter("clientSourceTarget"));
            //set prefix
            setAjaxLoginCallBackUrlPrefix(filterConfig.getInitParameter("ajaxLoginCallBackUrlPrefix"));
            setCasServerUrlPrefix(filterConfig.getInitParameter("casServerUrlPrefix"));
            if (this.casServerUrlPrefix == null) {
                this.casServerUrlPrefix = casServerLoginUrl.replace("/login", "");
            }
            setRenew(getBoolean(ConfigurationKeys.RENEW));
            setGateway(getBoolean(ConfigurationKeys.GATEWAY));

            final String ignorePattern = getString(ConfigurationKeys.IGNORE_PATTERN);
            final String ignoreUrlPatternType = getString(ConfigurationKeys.IGNORE_URL_PATTERN_TYPE);

            if (ignorePattern != null) {
                final Class<? extends UrlPatternMatcherStrategy> ignoreUrlMatcherClass = PATTERN_MATCHER_TYPES.get(ignoreUrlPatternType);
                if (ignoreUrlMatcherClass != null) {
                    this.ignoreUrlPatternMatcherStrategyClass = ReflectUtils.newInstance(ignoreUrlMatcherClass.getName());
                } else {
                    try {
                        logger.trace("Assuming {} is a qualified class name...", ignoreUrlPatternType);
                        this.ignoreUrlPatternMatcherStrategyClass = ReflectUtils.newInstance(ignoreUrlPatternType);
                    } catch (final IllegalArgumentException e) {
                        logger.error("Could not instantiate class [{}]", ignoreUrlPatternType, e);
                    }
                }
                if (this.ignoreUrlPatternMatcherStrategyClass != null) {
                    this.ignoreUrlPatternMatcherStrategyClass.setPattern(ignorePattern);
                }
            }

            final Class<? extends GatewayResolver> gatewayStorageClass = getClass(ConfigurationKeys.GATEWAY_STORAGE_CLASS);

            if (gatewayStorageClass != null) {
                setGatewayStorage(ReflectUtils.newInstance(gatewayStorageClass));
            }

            final Class<? extends AuthenticationRedirectStrategy> authenticationRedirectStrategyClass = getClass(ConfigurationKeys.AUTHENTICATION_REDIRECT_STRATEGY_CLASS);

            if (authenticationRedirectStrategyClass != null) {
                this.authenticationRedirectStrategy = ReflectUtils.newInstance(authenticationRedirectStrategyClass);
            }
        }
    }

    public void init() {
        super.init();
        CommonUtils.assertNotNull(this.casServerLoginUrl, "casServerLoginUrl cannot be null.");
    }

    private final String getLoginImgStr(String url) {
        String script = "(function(){\n" +
                "\tvar body = document.getElementsByTagName('body')[0];\n" +
                "\tvar img = document.createElement(\"img\");\n" +
                "\timg.src = '" + url + "';\n" +
                "\timg.style.display='none'\n" +
                "\tbody.appendChild(img); \n" +
                "})();\n";
        return script;
    }

    public final void doFilter(final ServletRequest servletRequest, final ServletResponse servletResponse,
                               final FilterChain filterChain) throws IOException, ServletException {
        //直接通过filter 不处理
        if (servletRequest.getAttribute("__pass_sso") != null) {
            filterChain.doFilter(servletRequest, servletResponse);
            return;
        }
        final HttpServletRequest request = (HttpServletRequest) servletRequest;
        final HttpServletResponse response = (HttpServletResponse) servletResponse;

        if (isRequestUrlExcluded(request)) {
            logger.debug("Request is ignored.");
            filterChain.doFilter(request, response);
            return;
        }

        //登录状态判断
        Boolean cookLoginFlag = false;
        if (request.getCookies() != null) {
            for (Cookie cookie : request.getCookies()) {
                if ("__l".equals(cookie.getName())) {
                    if ("true".equals(cookie.getValue())) {
                        cookLoginFlag = true;
                        break;
                    }
                }
            }

        }

        Boolean sessionLoginFlag = false;
        Boolean cookieUuidFlag = false;
        String cookieUid = null;
        String uuid =null;
        if (request.getAttribute(CONST_CAS_ASSERTION) != null || (request.getSession() != null && request.getSession().getAttribute(CONST_CAS_ASSERTION) != null)) {
            sessionLoginFlag = true;
            uuid = (String) request.getSession().getAttribute(PRINCIPAL_UU_ID);

            if (request.getCookies() != null && uuid != null && !"".equals(uuid)) {
                for (Cookie cookie : request.getCookies()) {
                    if ("__u".equals(cookie.getName())) {
                        cookieUid = cookie.getValue();
                        break;
                    }
                }

            }
        }
        if (cookieUid == null || "".equals(cookieUid) || cookieUid.equals(uuid)) {
            cookieUuidFlag = true;
        }
        //logger.info("============:session-id{}",request.getSession().getId());
        //logger.info("============:session{}",request.getSession());
//        logger.info("===========cookieUuidFlag:{}",cookieUuidFlag);
        //logger.info("===========sessionLoginFlag:{}",sessionLoginFlag);
        //logger.info("===========cookLoginFlag:{}",cookLoginFlag);
        //logger.info(PRINCIPAL_UU_ID+":{}", request.getSession().getAttribute(PRINCIPAL_UU_ID));

//        final HttpSession session = request.getSession(false);
//        final Assertion assertion = session != null ? (Assertion) session.getAttribute(CONST_CAS_ASSERTION) : null;

        //当写cookie 域不在当前服务内可能会出现死循环
        if (sessionLoginFlag && cookLoginFlag && cookieUuidFlag) {

            //jsonp 返回用户信息，数据脱敏
            if (request.getRequestURI().endsWith("_getuserinfo.jsonp")){
                response.setContentType("application/javascript");
                String functionName = request.getParameter("callback");
                if(functionName==null ||"".endsWith(functionName)){
                    functionName = "callback";
                }
                functionName = functionName.replace("<","");
                StringBuilder sb = new StringBuilder();
                Assertion assertion = (Assertion) request.getSession().getAttribute(CONST_CAS_ASSERTION);
                Map map = assertion.getPrincipal().getAttributes();
                sb.append("{");
                String phone = (String) map.get("phone");
                String email = (String) map.get("email");
                String userName = (String) map.get("userName");

                if(phone!=null && phone.length()>3){
                    sb.append("\"phone\":\""+phone.substring(0,3)+"****\",");
                }
                if(userName!=null&& !"".equals(userName.trim())){
                    if(userName.length()>3){
                        sb.append("\"userName\":\""+userName.substring(0,3)+"****\",");
                    }else{
                        sb.append("\"userName\":\""+userName+"****\",");
                    }
                }
                if(email!=null && email.indexOf("@")>0){
                    String emailend = email.substring(email.indexOf("@"),email.length());
                    email = email.substring(0,email.indexOf("@"));
                    if(email.length()>3){
                        sb.append("\"email\":\""+email.substring(0,3)+"****"+emailend+"\",");
                    }else{
                        sb.append("\"email\":\""+email+"****"+emailend+"\",");
                    }
                }
                sb.append("\"version\":"+System.currentTimeMillis()+"");
                sb.append("}");

                response.setContentType("application/javascript; charset=UTF-8");
                response.getWriter().write(functionName + "("+sb.toString()+")"); //返回jsonp数据
                return;
            }

            //自动登录小图，直接返回
            if (request.getRequestURI().indexOf("_autologin_") >= 0) {
                response.setContentType("image/jpeg");
                BufferedImage bi = new BufferedImage(1, 1, BufferedImage.TYPE_INT_RGB);
                ImageIO.write(bi, "jpg", response.getOutputStream());
                return;
            }
            //状态检查, sessionflag 与cookieflag 都为True ,100% 已经登录，js什么都不做
            if (request.getRequestURI().indexOf("_checkStatus.js") > 0) {
//                String render = request.getHeader("Referer");
//                if(render == null) {
//                    render = "";
//                }
//
//                render =ajaxLoginCallBackUrlPrefix+"/_ajaxLoginCallback.js?_backUrl="+URLEncoder.encode(render, "UTF-8");
//
//                response.setContentType("application/javascript");
//                response.getWriter().write(getLoginImgStr(this.casServerUrlPrefix+"/checkStatus?service="+URLEncoder.encode(render, "UTF-8")));
                response.setContentType("application/javascript");
                response.getWriter().write("");
                return;
            }
            // 解决前后端分离登录问题
            if (request.getRequestURI().indexOf("_ajaxLoginCallback.js") > 0) {
                String logout = request.getParameter("logout");
                if (logout != null && "true".equals(logout.trim())) {
                    if (request.getSession(false) != null) request.getSession(false).invalidate();
                }
                String backuri = request.getParameter("_backUrl");
                if (backuri == null) {
                    response.sendRedirect(request.getContextPath());
                } else {
                    response.sendRedirect(backuri);
                }
                return;
            }

            filterChain.doFilter(request, response);
            return;
        }

        //因为该页面为受保护页面，直接转到登录页面
        if (request.getRequestURI().endsWith("_checkStatus.js")) {
            String render = request.getParameter("r");
            if (render == null || render.trim() == "") {
                render = request.getHeader("Referer");
                if (render == null) {
                    render = "";
                }
            }

            render = ajaxLoginCallBackUrlPrefix + "/_ajaxLoginCallback.js?_source_target=" + makeSourceTarget(request, response) + "&_backUrl=" + URLEncoder.encode(makeLoginCallBackUrl(render, makeSourceTarget(request, response), request, response), "UTF-8");

            String prefix = this.casServerLoginUrl.indexOf("?") > 0 ? "&" : "?";
            String urlToRedirectTo = this.casServerLoginUrl + prefix + "service=" + URLEncoder.encode(render, "UTF-8");
            String soureTarget = makeSourceTarget(request, response);
            if (soureTarget != null) {
                urlToRedirectTo = urlToRedirectTo + "&_source_target=" + soureTarget;
            }
            response.setContentType("application/javascript");
            response.getWriter().write("window.location.href='" + urlToRedirectTo + "';");
            return;
        }

        final String serviceUrl = constructServiceUrl(request, response);
        final String ticket = retrieveTicketFromRequest(request);
        final boolean wasGatewayed = this.gateway && this.gatewayStorage.hasGatewayedAlready(request, serviceUrl);

        if (CommonUtils.isNotBlank(ticket) || wasGatewayed) {
            filterChain.doFilter(request, response);
            return;
        }

        String modifiedServiceUrl;

        logger.debug("no ticket and no assertion found");
        if (this.gateway) {
            logger.debug("setting gateway attribute in session");
            modifiedServiceUrl = this.gatewayStorage.storeGatewayInformation(request, serviceUrl);
        } else {
            modifiedServiceUrl = serviceUrl;
        }

        modifiedServiceUrl = makeLoginCallBackUrl(modifiedServiceUrl, makeSourceTarget(request, response), request, response);

        logger.debug("Constructed service url: {}", modifiedServiceUrl);

        String urlToRedirectTo = CommonUtils.constructRedirectUrl(this.casServerLoginUrl,
                getProtocol().getServiceParameterName(), modifiedServiceUrl, this.renew, this.gateway);

        String soureTarget = makeSourceTarget(request, response);
        if (soureTarget != null) {
            String prefix = urlToRedirectTo.indexOf("?") > 0 ? "&" : "?";
            urlToRedirectTo = urlToRedirectTo + prefix + "_source_target=" + soureTarget;
        }
        logger.debug("redirecting to \"{}\"", urlToRedirectTo);
        this.authenticationRedirectStrategy.redirect(request, response, urlToRedirectTo, cookieUuidFlag);
    }

    public final void setRenew(final boolean renew) {
        this.renew = renew;
    }

    public final void setGateway(final boolean gateway) {
        this.gateway = gateway;
    }

    public final void setCasServerLoginUrl(final String casServerLoginUrl) {
        this.casServerLoginUrl = casServerLoginUrl;
    }

    public final void setClientSourceTarget(final String clientSourceTarget) {
        this.clientSourceTarget = clientSourceTarget;
    }
    /**
     * 自定义登录成功后回调方法
     *
     * @param defaultUrl
     * @param request
     * @param response
     * @return
     */
    protected String makeLoginCallBackUrl(String defaultUrl, String source, HttpServletRequest request, HttpServletResponse response) {
//        if(source!=null){
//            if(defaultUrl!=null){
//                String prefix = defaultUrl.indexOf("?")>0 ? "&": "?";
//                return defaultUrl +prefix + "_source_target="+source;
//            }else{
//                return defaultUrl;
//            }
//        }else{
//            return defaultUrl;
//        }
        return defaultUrl;
    }

    /**
     * 自定义业务来源，会把_source_target 传到统一认证中
     *
     * @param request
     * @param response
     * @return
     */
    protected String makeSourceTarget(HttpServletRequest request, HttpServletResponse response) {
        return this.clientSourceTarget;
    }

    public final void setGatewayStorage(final GatewayResolver gatewayStorage) {
        this.gatewayStorage = gatewayStorage;
    }

    private boolean isRequestUrlExcluded(final HttpServletRequest request) {
        if (this.ignoreUrlPatternMatcherStrategyClass == null) {
            return false;
        }

        final StringBuffer urlBuffer = request.getRequestURL();
        if (request.getQueryString() != null) {
            urlBuffer.append("?").append(request.getQueryString());
        }
        final String requestUri = urlBuffer.toString();
        return this.ignoreUrlPatternMatcherStrategyClass.matches(requestUri);
    }
}
