package pool.service.listeners;

import goods.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jms.annotation.JmsListener;
import org.springframework.stereotype.Component;
import pool.dto.ProviderGoodsDto;
import pool.dto.ProviderGoodsPrice;
import pool.dto.ProviderGoodsStatus;
import pool.model.ProviderGoodsMqLog;
import pool.model.repository.ProviderGoodsMqLogRepos;
import pool.service.service.ProviderGoodsService;
import pool.service.service.StoreService;
import pool.vo.ProductReturnVo;
import pool.vo.ProductUpdateResult;
import sinomall.global.common.response.BaseResponse;
import store.api.dto.modeldto.core.StoreDto;
import utils.Lang;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

/**
 * taofeng
 */
@Component
@Slf4j
public class MqHandleListener {

    @Autowired
    StoreService storeService;

    @Autowired
    ProviderGoodsService providerGoodsService;


    @Autowired
    ProviderGoodsMqLogRepos providerGoodsMqLogRepos;

    /**
     * 更新商品价格
     */
    @JmsListener(destination = "${provider.updatePrice.queue}", containerFactory = "queueListenerFactory")
    public void updatePrice(PriceUpdateMessageVo priceUpdateMessageVo) {

        // TODO: 2018-9-5 messageId传过来 
        StoreDto storeDto = storeService.getStoreByCode(priceUpdateMessageVo.getStoreCode());
        List<ProviderGoodsPrice> providerGoodsPrices = priceUpdateMessageVo.getProviderGoodsPrices();

        log.info("providerPool处理供应商商品价格消息：{},===={}",priceUpdateMessageVo.getStoreCode(),providerGoodsPrices.toString());
        BaseResponse rsp = providerGoodsService.updateGoodsPrice(providerGoodsPrices,storeDto.getId());

        if(!Lang.isEmpty(rsp.getResult())) {
            ProductUpdateResult pur = (ProductUpdateResult) rsp.getResult();
            if(!Lang.isEmpty(pur) && !Lang.isEmpty(pur.getFailList())) {
                for (int i = 0; i <pur.getFailList().size(); i++) {
                    ProductReturnVo prv = (ProductReturnVo) pur.getFailList().get(i);
                    log.info("价格变更处理失败的sku数据：{}===={}",priceUpdateMessageVo.getStoreCode(),prv.getSku());
                    addProviderGoodsMqLog(prv.getSku(),ProviderGoodsMqLog.UPDATEPRICE,priceUpdateMessageVo.getStoreCode());
                }
            }
        }

    }


    /**
     * 删除商品
     */
    @JmsListener(destination = "${provider.productDelete.queue}", containerFactory = "queueListenerFactory")
    public void productDelete(ProductDeleteMessageVo productDeleteMessageVo) {
        StoreDto storeDto = storeService.getStoreByCode(productDeleteMessageVo.getStoreCode());
        List<String> skus = productDeleteMessageVo.getSkuIds();
        log.info("providerPool处理供应商删除消息：{},===={}",productDeleteMessageVo.getStoreCode(),skus.toString());
        BaseResponse rsp = providerGoodsService.deleteProviderGoodsBatch(skus,storeDto.getId());
        if(!rsp.isSuccess()) {
            for (int i = 0; i < skus.size(); i++) {
                log.info("商品删除处理失败的sku数据：{}===={}",productDeleteMessageVo.getStoreCode(),skus.get(i));
                addProviderGoodsMqLog(skus.get(i),ProviderGoodsMqLog.PRODUCTDELETE,productDeleteMessageVo.getStoreCode());
            }
        }
    }

    /**
     * 新增
     */
    @JmsListener(destination = "${provider.productAdd.queue}", containerFactory = "queueListenerFactory")
    public void productAdd(ProductAddMessageVo productAddMessageVo) {
        StoreDto storeDto = storeService.getStoreByCode(productAddMessageVo.getStoreCode());
        ProviderGoodsDto providerGoodsDto = productAddMessageVo.getProviderGoodsDto();
        providerGoodsDto.setStoreId(storeDto.getId());
        BaseResponse rsp = providerGoodsService.saveProviderGoods(providerGoodsDto);
        if(!rsp.isSuccess()) {
            addProviderGoodsMqLog(providerGoodsDto.getSku(),ProviderGoodsMqLog.PRODUCTADD,productAddMessageVo.getStoreCode());
        }

    }

    /**
     * 更新商品介绍和参数
     */
    @JmsListener(destination = "${provider.productDetail.queue}", containerFactory = "queueListenerFactory")
    public void productDetail(ProductDetailUpdateMessageVo productDetailUpdateMessageVo) {

        StoreDto storeDto = storeService.getStoreByCode(productDetailUpdateMessageVo.getStoreCode());
        ProviderGoodsDto providerGoodsDto = productDetailUpdateMessageVo.getProviderGoodsDto();

        providerGoodsDto.setStoreId(storeDto.getId());

        log.info("providerPool处理供应商商品信息变更消息：messageId:{}===={}===={}",productDetailUpdateMessageVo.getMessageId(),
                productDetailUpdateMessageVo.getStoreCode(),providerGoodsDto.toString());
        BaseResponse rsp = providerGoodsService.updateProviderGoods(providerGoodsDto);
        if(!rsp.isSuccess()) {
            addProviderGoodsMqLog(providerGoodsDto.getSku(),ProviderGoodsMqLog.PRODUCTDETAIL,productDetailUpdateMessageVo.getStoreCode());
        }
    }

    /**
     * 更新商品上下架信息
     */
    @JmsListener(destination = "${provider.updateSkuStatus.queue}", containerFactory = "queueListenerFactory")
    public void updateSkuStatus(SkuStatusUpdateMessageVo skuStatusUpdateMessageVo) {

        StoreDto storeDto = storeService.getStoreByCode(skuStatusUpdateMessageVo.getStoreCode());
        List<GoodsStateVo> goodsStateVos = skuStatusUpdateMessageVo.getGoodsStateVoList();
        List<ProviderGoodsStatus> pgss = new ArrayList<>();
        for (int i = 0; i <goodsStateVos.size() ; i++) {
            ProviderGoodsStatus pgs = new ProviderGoodsStatus();
            pgs.setSku(goodsStateVos.get(i).getSku());
            pgs.setStatus(goodsStateVos.get(i).getState());
            pgss.add(pgs);
        }
        log.info("providerPool处理供应商上下架消息：storeCode:{}===={}",
                skuStatusUpdateMessageVo.getStoreCode(),pgss.toString());
        BaseResponse rsp = providerGoodsService.updateGoodsStatus(pgss, storeDto.getId());

        if(!Lang.isEmpty(rsp.getResult())) {
            ProductUpdateResult pur = (ProductUpdateResult) rsp.getResult();
            if(!Lang.isEmpty(pur) && !Lang.isEmpty(pur.getFailList())) {
                for (int i = 0; i <pur.getFailList().size(); i++) {
                    ProductReturnVo prv = (ProductReturnVo) pur.getFailList().get(i);
                    log.info("上下架处理失败的sku数据：{}===={}",skuStatusUpdateMessageVo.getStoreCode(),prv.getSku());
                    addProviderGoodsMqLog(prv.getSku(),ProviderGoodsMqLog.UPDATESKUSTATUS,skuStatusUpdateMessageVo.getStoreCode());
                }
            }
        }

    }




    /**
     * 记录日志
     */
    private void addProviderGoodsMqLog(String sku, String type, String providerName) {
        try {
            ProviderGoodsMqLog providerGoodsMqLog = new ProviderGoodsMqLog();
            providerGoodsMqLog.setSku(sku);
            providerGoodsMqLog.setType(type);
            providerGoodsMqLog.setProvidername(providerName);
//            providerGoodsMqLog.setMessageId(messageId);
//            providerGoodsMqLog.setRemark(remark);

            // TODO: 2018-8-27
            CompletableFuture.runAsync(() -> {
                providerGoodsMqLogRepos.save(providerGoodsMqLog);
            });
        } catch (Exception e) {
            log.error("SKU {} --- type {} --- 日志保存异常", sku, type, e);
        }
    }

}
