package pool.service.listeners;

import goods.vo.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jms.core.JmsTemplate;
import org.springframework.stereotype.Component;
import pool.dto.ProviderGoodsDto;
import pool.dto.ProviderGoodsPrice;

import java.util.List;

/**
 * Created by Jian on 2018-8-28.
 */
@Component
@Slf4j
public class MqHandleProducer {

    @Autowired
    JmsTemplate jmsTemplate;

    @Value("${taiping.updatePrice.queue}")
    private String taipingUpdatePriceQueue;

    @Value("${taiping.productDelete.queue}")
    private String taipingProductDeleteQueue;

    @Value("${taiping.productAdd.queue}")
    private String taipingProductAddQueue;

    @Value("${taiping.productDetail.queue}")
    private String taipingProductDetailQueue;

    @Value("${taiping.updateSkuStatusUp.queue}")
    private String taipingUpdateSkuStatusUpQueue;

    @Value("${taiping.updateSkuStatusDown.queue}")
    private String taipingUpdateSkuStatusDownQueue;


    /**
     * 向金服商品池发送消息（价格变更）
     * @param providerGoodsPrices
     * @param storeCode
     */
    public void sendPriceUpdateMessage(List<ProviderGoodsPrice> providerGoodsPrices,String storeCode){

        PriceUpdateMessageVo pumv = new PriceUpdateMessageVo();
        pumv.setProviderGoodsPrices(providerGoodsPrices);
        pumv.setStoreCode(storeCode);
        log.info("价格变更消息处理，向金服商品池推送,供应商：{}=====sku:{}",storeCode,providerGoodsPrices.toString());
        jmsTemplate.convertAndSend(taipingUpdatePriceQueue, pumv);

    }


    /**
     * 向金服商品池发送消息（商品详情，修改）
     * @param providerGoodsDto
     * @param storeCode
     */
    public void sendProductDetailMessage(ProviderGoodsDto providerGoodsDto,String storeCode){

        ProductDetailUpdateMessageVo pdumv = new ProductDetailUpdateMessageVo();
        pdumv.setProviderGoodsDto(providerGoodsDto);
        pdumv.setStoreCode(storeCode);
        log.info("商品详情变更消息处理，向金服商品池推送,供应商：{}=====sku:{}====detail：{}",storeCode,providerGoodsDto.getSku(),providerGoodsDto.toString());
        jmsTemplate.convertAndSend(taipingProductDetailQueue, pdumv);

    }


    /**
     * 向金服商品池发送消息（删除）
     * @param skus
     * @param storeCode
     */
    public void sendProductDeleteMessage(List<String> skus,String storeCode){

        ProductDeleteMessageVo pdmv = new ProductDeleteMessageVo();
        pdmv.setSkuIds(skus);
        pdmv.setStoreCode(storeCode);
        log.info("商品删除消息处理，向金服商品池推送,供应商：{}=====skus：{}",storeCode,skus.toString());
        jmsTemplate.convertAndSend(taipingProductDeleteQueue, pdmv);

    }


    /**
     * 向金服商品池发送消息（上架）
     * @param skus
     * @param storeId
     * @param organizationCode
     */
    public void sendSkuUpMessage(List<String> skus, String storeId,String organizationCode){

        SkuBatchUpdateStatusVo skuBatchUpdateStatusVo = new SkuBatchUpdateStatusVo();
        skuBatchUpdateStatusVo.setStoreId(storeId);
        skuBatchUpdateStatusVo.setOrganizationCode(organizationCode);
        skuBatchUpdateStatusVo.setSkus(skus);

        jmsTemplate.convertAndSend(taipingUpdateSkuStatusUpQueue, skuBatchUpdateStatusVo);

    }


    /**
     * 向金服商品池发送消息（下架）
     * @param skus
     * @param storeId
     */
    public void sendSkuDownMessage(List<String> skus, String storeId){

        SkuBatchUpdateStatusVo skuBatchUpdateStatusVo = new SkuBatchUpdateStatusVo();
        skuBatchUpdateStatusVo.setStoreId(storeId);
        skuBatchUpdateStatusVo.setSkus(skus);

        jmsTemplate.convertAndSend(taipingUpdateSkuStatusDownQueue, skuBatchUpdateStatusVo);

    }

    /**
     * 向金服商品池发送消息（新增）
     * @param providerGoodsDto
     * @param storeCode
     */
    public void sendProductAddMessage(ProviderGoodsDto providerGoodsDto, String storeCode){

        ProductAddMessageVo pamv = new ProductAddMessageVo();
        pamv.setProviderGoodsDto(providerGoodsDto);
        pamv.setStoreCode(storeCode);

        jmsTemplate.convertAndSend(taipingProductAddQueue, pamv);

    }


}
