package pool.service.service.ServiceImpl;

import cms.api.OrganizationApi;
import com.weibo.api.motan.config.springsupport.annotation.MotanReferer;
import goods.api.ProductBrandApi;
import goods.api.ProductCategoryApi;
import goods.api.ProductCategoryconfigApi;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.MapperFacade;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pool.commonUtil.MessageDictionary;
import pool.dto.ProductModelDto;
import pool.dto.ProductModelQueryDto;
import pool.dto.brand.brandmapper.BrandMapperDto;
import pool.model.ProviderProduct;
import pool.model.ProviderProductFashion;
import pool.model.repository.ProviderProductBrandRepos;
import pool.model.repository.ProviderProductCategoryRepos;
import pool.model.repository.ProviderProductFashionRepos;
import pool.model.repository.ProviderProductRepos;
import pool.service.service.BrandService;
import pool.service.service.ProductModelService;
import pool.vo.BrandVo;
import pool.vo.ProductModelPageVo;
import pool.vo.ProductModelVo;
import sinomall.global.common.response.BaseResponse;
import utils.Lang;
import utils.sql.PageVo;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;

@Slf4j
@Service
public class ProductModelServiceImpl implements ProductModelService {

    @Autowired
    private ProviderProductRepos providerProductRepos;
    @Autowired
    private MapperFacade mapperFacade;
    @Autowired
    private BrandService brandService;

    /**
     * 查询商品模型，分页
     *
     * @param productModelQueryDto
     * @return
     */
    @Override
    public BaseResponse<PageVo<ProductModelVo>> queryModels(ProductModelQueryDto productModelQueryDto) {
        BaseResponse baseResponse = new BaseResponse<>();
        try {
            if (!Lang.isEmpty(productModelQueryDto) && Lang.isEmpty(productModelQueryDto.getPageNo())) {
                productModelQueryDto.setPageNo(MessageDictionary.DEFAULT_PAGENO);
            }
            if (!Lang.isEmpty(productModelQueryDto) && Lang.isEmpty(productModelQueryDto.getPageSize())) {
                productModelQueryDto.setPageSize(MessageDictionary.DEFAULT_PAGESIZE);
            }
            if (Lang.isEmpty(productModelQueryDto)) {
                productModelQueryDto = new ProductModelQueryDto();
                productModelQueryDto.setPageNo(MessageDictionary.DEFAULT_PAGENO);
                productModelQueryDto.setPageSize(MessageDictionary.DEFAULT_PAGESIZE);
            }
            int pageSize = productModelQueryDto.getPageSize();
            int realPageNo = productModelQueryDto.getPageNo() - 1 > 0 ? productModelQueryDto.getPageNo() - 1 : 0;
            Pageable pageable = new PageRequest(realPageNo, pageSize);
            ProviderProduct finalProviderProduct = mapperFacade.map(productModelQueryDto, ProviderProduct.class);
            Specification<ProviderProduct> specification = new Specification<ProviderProduct>() {
                @Override
                public Predicate toPredicate(Root<ProviderProduct> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                    List<Predicate> predicates = new ArrayList<>();
                    if (!Lang.isEmpty(finalProviderProduct.getBrandId())) {
                        predicates.add(criteriaBuilder.equal(root.get("brandId"), finalProviderProduct.getBrandId()));
                    }
                    if (!Lang.isEmpty(finalProviderProduct.getCategoryId())) {
                        predicates.add(criteriaBuilder.equal(root.get("categoryId"), finalProviderProduct.getCategoryId()));
                    }
                    if (!Lang.isEmpty(finalProviderProduct.getModelSku())) {
                        predicates.add(criteriaBuilder.equal(root.get("modelSku"), finalProviderProduct.getModelSku()));
                    }
                    if (!Lang.isEmpty(finalProviderProduct.getStoreId())) {
                        predicates.add(criteriaBuilder.equal(root.get("storeId"), finalProviderProduct.getStoreId()));
                    }
                    predicates.add(criteriaBuilder.equal(root.get("isDelete"), false));
                    return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
                }
            };
            Page<ProviderProduct> page = providerProductRepos.findAll(specification, pageable);
            List<ProviderProduct> providerProductList = page.getContent();
//            List<ProductModelVo> productModelVoList = mapperFacade.mapAsList(providerProductList,ProductModelVo.class);
            List<ProductModelVo> productModelVos = new ArrayList<>();
            for (ProviderProduct item : providerProductList) {
                ProductModelVo productModelVo = new ProductModelVo();
                productModelVo.setModelId(item.getModelId());
                productModelVo.setName(item.getName());
                productModelVo.setModelSku(item.getModelSku());
                productModelVos.add(productModelVo);
            }
            //返回页码数 +1
            ProductModelPageVo productModelPageVo = new ProductModelPageVo(productModelVos, page.getNumber() + 1, page.getSize(), page.getTotalElements());

            baseResponse.setResult(productModelPageVo);
            baseResponse.setResultCode(MessageDictionary.RETURN_SUCCESS_CODE);
            baseResponse.setResultMessage(MessageDictionary.RETURN_SUCCESS_MESSAGE);
            baseResponse.setSuccess(true);
        } catch (Exception e) {
            baseResponse = new BaseResponse(false, MessageDictionary.RETURN_INTERNAL_ERROR_MESSAGE, "", MessageDictionary.RETURN_INTERNAL_EXCEPTION_CODE);
            e.printStackTrace();
        }
        return baseResponse;
    }

    @Override
    @Transactional
    public BaseResponse saveModel(ProductModelDto productModelDto) {
        BaseResponse baseResponse = new BaseResponse<>();
        try {

            if (Lang.isEmpty(productModelDto)) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            if (!Lang.isEmpty(productModelDto) && Lang.isEmpty(productModelDto.getName())) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_NAME_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            if (!Lang.isEmpty(productModelDto) && Lang.isEmpty(productModelDto.getModelSku())) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_MODELSKU_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            if (!Lang.isEmpty(productModelDto) && Lang.isEmpty(productModelDto.getCategoryId())) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_CATEGORYID_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            if (!Lang.isEmpty(productModelDto) && Lang.isEmpty(productModelDto.getBrandId())) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_BRANDID_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }

            ProviderProduct providerProduct = mapperFacade.map(productModelDto, ProviderProduct.class);
            //新增前根据modelSku和storeId判断商品模型是否已经存在
            ProviderProduct providerProduct2 = providerProductRepos.findOneByModelSkuAndStoreId(providerProduct.getModelSku(), providerProduct.getStoreId());
            if (!Lang.isEmpty(providerProduct2)) {
                return new BaseResponse(false, MessageDictionary.RETURN_ALREADY_EXIST_PRODUCT_MODEL_MESSAGE, "", MessageDictionary.RETURN_ALREADY_EXIST_OBJECT_CODE);
            }
            String providerProductBrandId = providerProduct.getBrandId(); //供应商品牌id
            BrandMapperDto brandMapperDto = brandService.queryBrand(providerProductBrandId);
            if (!Lang.isEmpty(brandMapperDto)) {
                providerProduct.setProductBrandId(brandMapperDto.getProductBrandId());
            }else{
                return new BaseResponse(false,MessageDictionary.RETURN_NO_EXIST_BRAND_MESSAGE,"",MessageDictionary.RETURN_NO_EXIST_OBJECT_CODE);
            }
            providerProduct = providerProductRepos.saveAndFlush(providerProduct);

            //要求返回：ProductModelVo
            ProductModelVo productModelVo = new ProductModelVo();
            productModelVo.setModelSku(providerProduct.getModelSku());
            productModelVo.setName(providerProduct.getName());
            productModelVo.setModelId(providerProduct.getModelId());

            baseResponse.setResult(productModelVo);
            baseResponse.setResultCode(MessageDictionary.RETURN_SUCCESS_CODE);
            baseResponse.setSuccess(true);
            baseResponse.setResultMessage(MessageDictionary.RETURN_SUCCESS_MESSAGE);
        } catch (Exception e) {
            e.printStackTrace();
            baseResponse = new BaseResponse(false, MessageDictionary.RETURN_INTERNAL_ERROR_MESSAGE, "", MessageDictionary.RETURN_INTERNAL_EXCEPTION_CODE);
        }
        return baseResponse;
    }

    @Override
    @Transactional
    public BaseResponse updateModel(ProductModelDto productModelDto) {
        BaseResponse baseResponse = null;
        try {
            if (Lang.isEmpty(productModelDto)) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            if (!Lang.isEmpty(productModelDto) && Lang.isEmpty(productModelDto.getName())) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_NAME_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            if (!Lang.isEmpty(productModelDto) && Lang.isEmpty(productModelDto.getModelSku())) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_MODELSKU_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            if (!Lang.isEmpty(productModelDto) && Lang.isEmpty(productModelDto.getCategoryId())) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_CATEGORYID_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            if (!Lang.isEmpty(productModelDto) && Lang.isEmpty(productModelDto.getBrandId())) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_BRANDID_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            baseResponse = new BaseResponse<>();
            //@TODO 查询商品模型,根据name,modelSku,brandId,categoryId,modelId,最后根据modelSku和storeId,查询原数据
            ProviderProduct providerProduct = providerProductRepos.findOneByModelSkuAndStoreId(productModelDto.getModelSku(), productModelDto.getStoreId());

            //判断要修改的模型存不存在
            if (Lang.isEmpty(providerProduct)) {
                return new BaseResponse(false, MessageDictionary.RETURN_NO_EXIST_PRODUCT_MODEL_MESSAGE, "", MessageDictionary.RETURN_NO_EXIST_OBJECT_CODE);
            }
            String providerProductBrandId = providerProduct.getBrandId(); //供应商品牌id
            BrandMapperDto brandMapperDto = brandService.queryBrand(providerProductBrandId);
            if (!Lang.isEmpty(brandMapperDto)) {
                providerProduct.setProductBrandId(brandMapperDto.getProductBrandId());
            }
            //修改
            providerProduct.setName(productModelDto.getName());
            providerProduct.setModelSku(productModelDto.getModelSku());
            providerProduct.setCategoryId(productModelDto.getCategoryId());
            providerProduct.setBrandId(productModelDto.getBrandId());
            providerProduct.setProductBrandId(productModelDto.getBrandId());
            providerProduct = providerProductRepos.save(providerProduct);
            //回显
            ProductModelVo productModelVo = mapperFacade.map(providerProduct, ProductModelVo.class);   //回显数据

            baseResponse.setResult(productModelVo);
            baseResponse.setResultCode(MessageDictionary.RETURN_SUCCESS_CODE);
            baseResponse.setSuccess(true);
            baseResponse.setResultMessage(MessageDictionary.RETURN_SUCCESS_MESSAGE);
        } catch (Exception e) {
            e.printStackTrace();
            baseResponse = new BaseResponse(false, MessageDictionary.RETURN_INTERNAL_ERROR_MESSAGE, "", MessageDictionary.RETURN_INTERNAL_EXCEPTION_CODE);
        }
        return baseResponse;
    }

    @Override
    public BaseResponse<String> deleteModel(String modelSku, String storeId) {
        BaseResponse<String> baseResponse = new BaseResponse<>();
        try {
            if (Lang.isEmpty(modelSku)) {
                return new BaseResponse(false, MessageDictionary.RETURN_NULL_PARAM_ERROR_MESSAGE, "", MessageDictionary.RETURN_RUNTIME_CONDITION_MISSING_CODE);
            }
            ProviderProduct providerProduct = providerProductRepos.findOneByModelSkuAndStoreId(modelSku, storeId);
            if (Lang.isEmpty(providerProduct)) {
                baseResponse.setResultCode(MessageDictionary.RETURN_NO_EXIST_OBJECT_CODE);
                baseResponse.setResultMessage(MessageDictionary.RETURN_NO_EXIST_PRODUCT_MODEL_MESSAGE);
                baseResponse.setSuccess(false);
                return baseResponse;
            }
            if (!Lang.isEmpty(providerProduct) && providerProduct.getIsDelete()) {
                baseResponse.setResultCode(MessageDictionary.RETURN_ALREADY_LOGIC_DELETE_CODE);
                baseResponse.setResultMessage(MessageDictionary.RETURN_ALREADY_LOGIC_DELETE_MODEL_MESSAGE);
                baseResponse.setSuccess(false);
                return baseResponse;
            }
            providerProduct.setIsDelete(true);    //逻辑删除
            providerProductRepos.save(providerProduct);

            baseResponse.setSuccess(true);
            baseResponse.setResultMessage(MessageDictionary.RETURN_SUCCESS_MESSAGE);
            baseResponse.setResultCode(MessageDictionary.RETURN_SUCCESS_CODE);
            baseResponse.setResult("");
        } catch (Exception e) {
            e.printStackTrace();
            baseResponse = new BaseResponse(false, MessageDictionary.RETURN_INTERNAL_ERROR_MESSAGE, "", MessageDictionary.RETURN_INTERNAL_EXCEPTION_CODE);
        }
        return baseResponse;
    }

    @Transactional
    public List<ProductModelDto> findByBrandIdAndIsDelete(String providerProductBrandId, Boolean isDelete) {
        List<ProviderProduct> providerProductList = providerProductRepos.findAll(new Specification<ProviderProduct>() {
            @Override
            public Predicate toPredicate(Root<ProviderProduct> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder criteriaBuilder) {
                List<Predicate> predicateList = new ArrayList<>();
                predicateList.add(criteriaBuilder.equal(root.get("brandId"), providerProductBrandId));
                predicateList.add(criteriaBuilder.equal(root.get("isDelete"), isDelete));
                return criteriaBuilder.and(predicateList.toArray(new Predicate[predicateList.size()]));
            }
        });
        List<ProductModelDto> productModelDtoList = mapperFacade.mapAsList(providerProductList, ProductModelDto.class);
        return productModelDtoList;
    }

    @Override
    public ProductModelDto findByModelSkuAndStoreId(String modelSku, String storeId) {
        ProviderProduct providerProduct = providerProductRepos.findOneByModelSkuAndStoreId(modelSku, storeId);
        ProductModelDto productModelDto = mapperFacade.map(providerProduct, ProductModelDto.class);
        return productModelDto;
    }

    @Autowired
    ProviderProductCategoryRepos providerProductCategoryRepos;
    @Autowired
    ProviderProductBrandRepos providerProductBrandRepos;
    @MotanReferer
    ProductCategoryApi productCategoryApi;
    @MotanReferer
    ProductBrandApi productBrandApi;
    @Autowired
    ProviderProductFashionRepos providerProductFashionRepos;
    @MotanReferer
    ProductCategoryconfigApi ProductCategoryconfigApi;
    @MotanReferer
    OrganizationApi organizationApi;

    /**
     * 判断商品分类和品牌是否映射
     */
    @Override
    public boolean isSynCateAndBrand(String sku, String storeId, String orgCode) {

        String orgId = organizationApi.findOrganizationIdByOrgCode(orgCode);

        List<ProviderProductFashion> providerProductFashionList = providerProductFashionRepos.findBySkuAndStoreIdAndIsDelete(sku, storeId, false);
        if (providerProductFashionList != null && providerProductFashionList.size()>0) {
            ProviderProduct providerProduct = providerProductFashionList.get(0).getProviderProduct();
            if (!Lang.isEmpty(providerProduct)) {
                if (Lang.isEmpty(providerProduct.getCategoryId()) || Lang.isEmpty(providerProduct.getProductBrandId())) {
                    log.info("========sku:{}分类品牌未映射storeId:{}",sku,storeId);
                    return false;
                }
                boolean isUp = ProductCategoryconfigApi.findIsUpByCategoryIdAndOrgIdAndIsDelete(providerProduct.getCategoryId(), orgId, false);
                log.info("========sku:{}所属分类：{}在商城：{}是否上架isUp:{}",sku,providerProduct.getCategoryId(),orgCode,isUp);
                return isUp;
            }
        }
        return false;
    }

    @Override
    @Transactional
    public void updateProviderProductMappingInfo(String providerBrandId, boolean isDelete, BrandVo brandVo) {
        try {
            List<ProviderProduct> providerProducts=  providerProductRepos.findByBrandIdAndIsDelete(providerBrandId,isDelete);
            for (ProviderProduct providerProduct:providerProducts) {
                providerProduct.setProductBrandId(brandVo.getId());
            }
            providerProductRepos.save(providerProducts);
        }catch (Exception e) {
            log.info("provider product update mapping info failed!");
        }
    }
}
