package score.service.service.impl;

import com.alibaba.fastjson.JSON;
import com.weibo.api.motan.config.springsupport.annotation.MotanService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.JpaSort;
import org.springframework.util.StringUtils;
import score.api.ScoreApi;
import score.api.definition.ScoreBusinessType;
import score.api.dto.entitydto.*;
import score.api.dto.requestdto.*;
import score.api.dto.responsedto.PageResponseDTO;
import score.api.dto.responsedto.ScoreStatisticsResponseDTO;
import score.api.dto.responsedto.TransactionResponseDTO;
import score.api.exception.GiveAwayFailException;
import score.model.ScoreAccount;
import score.model.ScoreExpenseDetail;
import score.model.ScoreGiveAwayLog;
import score.model.ScoreIncomeDetail;
import score.model.repository.ScoreAccountRepos;
import score.model.repository.ScoreExpenseDetailRepos;
import score.model.repository.ScoreGiveAwayLogRepos;
import score.model.repository.ScoreIncomeDetailRepos;
import score.model.specification.ScoreDetailSpecification;
import score.service.service.ScoreService;
import score.service.service.UniqueSequenceNoService;
import sinomall.config.api.base.BaseResponse;
import utils.GlobalContants;
import utils.Lang;
import utils.data.BeanMapper;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@MotanService
public class ScoreServiceImpl implements ScoreService {
    @Autowired
    ScoreAccountRepos accountRepos;
    @Autowired
    ScoreIncomeDetailRepos scoreIncomeDetailRepos;
    @Autowired
    ScoreExpenseDetailRepos scoreExpenseDetailRepos;
    Logger logger = LoggerFactory.getLogger(getClass());
    @Autowired
    ScoreApi scoreApi;
    @Autowired
    ScoreGiveAwayLogRepos scoreGiveAwayLogRepos;
    @Autowired
    UniqueSequenceNoService uniqueSequenceNoService;

    @Transactional
    @Override
    public ScoreAccountDTO queryScoreAccount(ScoreAccountRequestDTO scoreAccountRequestDTO) {
        if (StringUtils.isEmpty(scoreAccountRequestDTO.getOrganizationCode())) {
            throw new RuntimeException("organizationCode不能为空");
        }
        ScoreAccount scoreAccount = null;
        if (!StringUtils.isEmpty(scoreAccountRequestDTO.getUserId())) {
            scoreAccount = accountRepos.findByOrganizationCodeAndUserId(scoreAccountRequestDTO.getOrganizationCode(), scoreAccountRequestDTO.getUserId());
            if (scoreAccount != null && (!StringUtils.isEmpty(scoreAccountRequestDTO.getPhone()) || !StringUtils.isEmpty(scoreAccountRequestDTO.getEmail()))) {
                //通过userId查询账户，如果手机号邮箱变更更新积分账户手机号邮箱
                if (!StringUtils.isEmpty(scoreAccountRequestDTO.getPhone()) && !scoreAccountRequestDTO.getPhone().equals(scoreAccount.getPhone())) {
                    scoreAccount.setPhone(scoreAccountRequestDTO.getPhone());
                }
                if (!StringUtils.isEmpty(scoreAccountRequestDTO.getEmail()) && !scoreAccountRequestDTO.getEmail().equals(scoreAccount.getEmail())) {
                    scoreAccount.setEmail(scoreAccountRequestDTO.getEmail());
                }
                if (!StringUtils.isEmpty(scoreAccountRequestDTO.getNickName()) && !scoreAccountRequestDTO.getNickName().equals(scoreAccount.getNickName())) {
                    scoreAccount.setNickName(scoreAccountRequestDTO.getNickName());
                }
                accountRepos.saveAndFlush(scoreAccount);
            }
        } else if (!StringUtils.isEmpty(scoreAccountRequestDTO.getPhone())) {
            scoreAccount = accountRepos.findByOrganizationCodeAndPhone(scoreAccountRequestDTO.getOrganizationCode(), scoreAccountRequestDTO.getPhone());
        } else if (!StringUtils.isEmpty(scoreAccountRequestDTO.getEmail())) {
            scoreAccount = accountRepos.findByOrganizationCodeAndEmail(scoreAccountRequestDTO.getOrganizationCode(), scoreAccountRequestDTO.getEmail());
        }
        if (scoreAccount == null) {
            //查询不到账户，创建一个新账户
            scoreAccount = BeanMapper.map(scoreAccountRequestDTO, ScoreAccount.class);
            scoreAccount.setIncomeScoreVal(new BigDecimal(0));
            scoreAccount.setExpendedScoreVal(new BigDecimal(0));
            scoreAccount.setFreezeScoreVal(new BigDecimal(0));
            scoreAccount.setInvalidScoreVal(new BigDecimal(0));
            scoreAccount.setAboutOfInvalidScoreVal(new BigDecimal(0));
            scoreAccount.setLastUpdated(new Date());
            accountRepos.saveAndFlush(scoreAccount);
        }
        return BeanMapper.map(scoreAccount, ScoreAccountDTO.class);
    }

    @Transactional
    @Override
    public TransactionResponseDTO addScore(ScoreIncomeDetailRequestDTO scoreIncomeDetailRequestDTO) {
        TransactionResponseDTO transactionResponseDTO = new TransactionResponseDTO();
        transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.SUCCESS);
        if (!validateRequestParam(scoreIncomeDetailRequestDTO, transactionResponseDTO)) {
            transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.ERROR);
            return transactionResponseDTO;
        }
        String returnMsg;
        ScoreAccount scoreAccount = accountRepos.findOne(scoreIncomeDetailRequestDTO.getScoreAccountId());
        if (scoreAccount == null) {
            returnMsg = "查询不到相关账户";
            transactionResponseDTO.setResponseMsg(returnMsg);
            transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.ERROR);
            logger.error(returnMsg + ",scoreIncomeDetailRequestDTO={}", JSON.toJSONString(scoreIncomeDetailRequestDTO), new RuntimeException(returnMsg));
            return transactionResponseDTO;
        }
        Integer countTransactionNo = scoreIncomeDetailRepos.validateIncomeTransactionNo(scoreAccount.getOrganizationCode(), scoreIncomeDetailRequestDTO.getTransactionNo());
        if (countTransactionNo > 0) {
            returnMsg = "交易流水号重复";
            transactionResponseDTO.setResponseMsg(returnMsg);
            transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.ERROR);
            logger.error(returnMsg + ",scoreIncomeDetailRequestDTO={}", JSON.toJSONString(scoreIncomeDetailRequestDTO), new RuntimeException(returnMsg));
            return transactionResponseDTO;
        }
        ScoreIncomeDetail incomeDetail = BeanMapper.map(scoreIncomeDetailRequestDTO, ScoreIncomeDetail.class);
        incomeDetail.setScoreAccount(scoreAccount);
        incomeDetail.setScoreRcptCode(uniqueSequenceNoService.generateUniqueSequenceNo());
        scoreIncomeDetailRepos.saveAndFlush(incomeDetail);
        scoreAccount.setIncomeScoreVal(scoreAccount.getIncomeScoreVal().add(incomeDetail.getScoreVal()));
        accountRepos.saveAndFlush(scoreAccount);
        transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.SUCCESS);
        transactionResponseDTO.setResponseMsg("调用成功");
        transactionResponseDTO.setScoreRcptCode(incomeDetail.getScoreRcptCode());
        return transactionResponseDTO;
    }

    public boolean validateRequestParam(ScoreDetailRequestDTO scoreDetailRequestDTO, TransactionResponseDTO transactionResponseDTO) {
        String returnMsg = "";
        if (StringUtils.isEmpty(scoreDetailRequestDTO.getScoreAccountId())) {
            returnMsg = "积分账户不能为空";
            transactionResponseDTO.setResponseMsg(returnMsg);
            logger.error(returnMsg + ",scoreIncomeDetailRequestDTO={}", JSON.toJSONString(scoreDetailRequestDTO), new RuntimeException(returnMsg));
            return false;
        }
        // TODO: 2017-11-14 锁并发
        if (StringUtils.isEmpty(scoreDetailRequestDTO.getTransactionNo())) {
            returnMsg = "请求流水号不能为空";
            transactionResponseDTO.setResponseMsg(returnMsg);
            logger.error(returnMsg + ",scoreIncomeDetailRequestDTO={}", JSON.toJSONString(scoreDetailRequestDTO), new RuntimeException(returnMsg));
            return false;
        }
        if(scoreDetailRequestDTO.getScoreVal()==null||scoreDetailRequestDTO.getScoreVal().compareTo(new BigDecimal(0))<0){
            returnMsg = "传入积分值不正确";
            transactionResponseDTO.setResponseMsg(returnMsg);
            logger.error(returnMsg + ",scoreIncomeDetailRequestDTO={}", JSON.toJSONString(scoreDetailRequestDTO), new RuntimeException(returnMsg));
            return false;
        }


        return true;
    }

    @Transactional
    @Override
    public TransactionResponseDTO expenseScore(ScoreExpenseDetailRequestDTO scoreExpenseDetailRequestDTO) {
        TransactionResponseDTO transactionResponseDTO = new TransactionResponseDTO();
        transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.SUCCESS);
        if (!validateRequestParam(scoreExpenseDetailRequestDTO, transactionResponseDTO)) {
            transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.ERROR);
            return transactionResponseDTO;
        }
        String returnMsg;
        ScoreAccount scoreAccount = accountRepos.findOne(scoreExpenseDetailRequestDTO.getScoreAccountId());
        if (scoreAccount == null) {
            returnMsg = "查询不到相关账户";
            transactionResponseDTO.setResponseMsg(returnMsg);
            transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.ERROR);
            logger.error(returnMsg + ",scoreIncomeDetailRequestDTO={}", JSON.toJSONString(scoreExpenseDetailRequestDTO), new RuntimeException(returnMsg));
            return transactionResponseDTO;
        }
        Integer countTransactionNo = scoreExpenseDetailRepos.validateExpenseTransactionNo(scoreAccount.getOrganizationCode(), scoreExpenseDetailRequestDTO.getTransactionNo());
        if (countTransactionNo > 0) {
            returnMsg = "交易流水号重复";
            transactionResponseDTO.setResponseMsg(returnMsg);
            transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.ERROR);
            logger.error(returnMsg + ",scoreIncomeDetailRequestDTO={}", JSON.toJSONString(scoreExpenseDetailRequestDTO), new RuntimeException(returnMsg));
            return transactionResponseDTO;
        }
        if (scoreAccount.getIncomeScoreVal().setScale(2, BigDecimal.ROUND_HALF_DOWN).compareTo(scoreAccount.getExpendedScoreVal().add(scoreExpenseDetailRequestDTO.getScoreVal()).setScale(2, BigDecimal.ROUND_HALF_DOWN)) < 0) {
            returnMsg = "账户积分不足";
            transactionResponseDTO.setResponseMsg(returnMsg);
            transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.ERROR);
            logger.error(returnMsg + ",scoreIncomeDetailRequestDTO={}", JSON.toJSONString(scoreExpenseDetailRequestDTO), new RuntimeException(returnMsg));
            return transactionResponseDTO;
        }
        ScoreExpenseDetail expenseDetail = BeanMapper.map(scoreExpenseDetailRequestDTO, ScoreExpenseDetail.class);
        expenseDetail.setScoreAccount(scoreAccount);
        expenseDetail.setScoreRcptCode(uniqueSequenceNoService.generateUniqueSequenceNo());
        scoreExpenseDetailRepos.saveAndFlush(expenseDetail);
        scoreAccount.setExpendedScoreVal(scoreAccount.getExpendedScoreVal().add(expenseDetail.getScoreVal()));
        accountRepos.saveAndFlush(scoreAccount);
        transactionResponseDTO.setStatus(GlobalContants.ResponseStatus.SUCCESS);
        transactionResponseDTO.setResponseMsg("调用成功");
        transactionResponseDTO.setScoreRcptCode(expenseDetail.getScoreRcptCode());
        return transactionResponseDTO;
    }

    @Override
    public PageResponseDTO<ScoreIncomeDetailDTO> queryScoreIncomeDetais(ScoreDetailQueryDTO scoreDetailQueryDTO) {
        ScoreDetailSpecification<ScoreIncomeDetail> scoreDetailSpecification = new ScoreDetailSpecification<>(scoreDetailQueryDTO);
        //默认交易时间倒叙
        if (scoreDetailQueryDTO.getPage() == null) {
            List<ScoreIncomeDetail> scoreIncomeDetailsAll = scoreIncomeDetailRepos.findAll(scoreDetailSpecification, new Sort(Sort.Direction.DESC, "transaTime"));
            return new PageResponseDTO<>(1, (long) scoreIncomeDetailsAll.size(), BeanMapper.mapList(scoreIncomeDetailsAll, ScoreIncomeDetail.class, ScoreIncomeDetailDTO.class));
        }
        Page<ScoreIncomeDetail> scoreIncomeDetails = scoreIncomeDetailRepos.findAll(scoreDetailSpecification, new PageRequest(scoreDetailQueryDTO.getPage(), scoreDetailQueryDTO.getPageSize(), new Sort(Sort.Direction.DESC, "transaTime")));
        return new PageResponseDTO<>(scoreIncomeDetails.getTotalPages(), scoreIncomeDetails.getTotalElements(), BeanMapper.mapList(scoreIncomeDetails.getContent(), ScoreIncomeDetail.class, ScoreIncomeDetailDTO.class));
    }

    @Override
    public PageResponseDTO<ScoreExpenseDetailDTO> queryScoreExpenseDetails(ScoreDetailQueryDTO scoreDetailQueryDTO) {
        ScoreDetailSpecification<ScoreExpenseDetail> scoreDetailSpecification = new ScoreDetailSpecification<>(scoreDetailQueryDTO);
        //默认交易时间倒叙
        if (scoreDetailQueryDTO.getPage() == null) {
            List<ScoreExpenseDetail> scoreExpenseDetailAll = scoreExpenseDetailRepos.findAll(scoreDetailSpecification, new Sort(Sort.Direction.DESC, "transaTime"));
            return new PageResponseDTO<>(1, (long)scoreExpenseDetailAll.size(), BeanMapper.mapList(scoreExpenseDetailAll, ScoreExpenseDetail.class, ScoreExpenseDetailDTO.class));
        }
        Page<ScoreExpenseDetail> scoreExpenseDetails = scoreExpenseDetailRepos.findAll(scoreDetailSpecification, new PageRequest(scoreDetailQueryDTO.getPage(), scoreDetailQueryDTO.getPageSize(), new Sort(Sort.Direction.DESC, "transaTime")));
        return new PageResponseDTO<>(scoreExpenseDetails.getTotalPages(), scoreExpenseDetails.getTotalElements(), BeanMapper.mapList(scoreExpenseDetails.getContent(), ScoreExpenseDetail.class, ScoreExpenseDetailDTO.class));
    }

    /**
     * 个人支出收入明细api
     *
     * @param scoreDetailQueryDTO
     * @return
     */
    @Override
    public PageResponseDTO<ScoreIncomeExpenseDetailDTO> queryScoreIncomeExpenseDetails(ScoreDetailQueryDTO scoreDetailQueryDTO) {
        List<ScoreIncomeExpenseDetailDTO> scoreIncomeExpenseDetailList = new ArrayList<>();
        Integer totalPage = 1;
        BigDecimal rowCount = new BigDecimal(0);
        if (!Lang.isEmpty(scoreDetailQueryDTO) && !Lang.isEmpty(scoreDetailQueryDTO.getScoreAccountIds())) {
            String scoreAccountId = scoreDetailQueryDTO.getScoreAccountIds().get(0);
            rowCount = scoreExpenseDetailRepos.getIncomeExpenseDetailCount(scoreAccountId);
            int start = scoreDetailQueryDTO.getPage() * scoreDetailQueryDTO.getPageSize() + 1;
            int end = start + scoreDetailQueryDTO.getPageSize();
            totalPage = rowCount.intValue() % scoreDetailQueryDTO.getPageSize() == 0 ? rowCount.intValue() / scoreDetailQueryDTO.getPageSize() : rowCount.intValue() / scoreDetailQueryDTO.getPageSize() + 1;
            List<Object[]> rows = scoreExpenseDetailRepos.queryIncomeExpenseDetail(scoreAccountId, start, end);
            for (Object[] obj : rows) {
                Date transaTime = (Date) obj[2];
                String busiSource = (String) obj[3];
                BigDecimal scoreVal = (BigDecimal) obj[4];
                String memo = (String) obj[5];

                ScoreIncomeExpenseDetailDTO scoreIncomeExpenseDetail = new ScoreIncomeExpenseDetailDTO();
                scoreIncomeExpenseDetail.setTransaTime(transaTime);
                scoreIncomeExpenseDetail.setBusiSource(busiSource);
                scoreIncomeExpenseDetail.setScoreVal(scoreVal);
                scoreIncomeExpenseDetail.setMemo(memo);

                scoreIncomeExpenseDetailList.add(scoreIncomeExpenseDetail);
            }
        }
        return new PageResponseDTO<>(totalPage, rowCount.longValue(), scoreIncomeExpenseDetailList);
    }

    @Transactional
    @Override
    public ScoreGiveAwayLogDTO scoreGiveAway(ScoreGiveAwayRequestDTO scoreGiveAwayRequestDTO) {
        //扣减积分
        ScoreGiveAwayLogDTO scoreGiveAwayLogDTO = new ScoreGiveAwayLogDTO();
        ScoreExpenseDetailRequestDTO scoreExpenseDetailRequestDTO = new ScoreExpenseDetailRequestDTO();
        scoreExpenseDetailRequestDTO.setScoreAccountId(scoreGiveAwayRequestDTO.getGiveAwayScoreAccountId());
        scoreExpenseDetailRequestDTO.setMemo(scoreGiveAwayRequestDTO.getMemo());
        scoreExpenseDetailRequestDTO.setBusiCode(ScoreBusinessType.BUSI_EXPENSE_GIVE_AWAY.getBusinessCode());
        scoreExpenseDetailRequestDTO.setBusiSource(ScoreBusinessType.BUSI_EXPENSE_GIVE_AWAY.getBusinessSource());
        scoreExpenseDetailRequestDTO.setTransactionNo(uniqueSequenceNoService.generateUniqueSequenceNo());
        scoreExpenseDetailRequestDTO.setScoreVal(scoreGiveAwayRequestDTO.getScoreVal());
        scoreExpenseDetailRequestDTO.setTransaTime(scoreGiveAwayRequestDTO.getGiveAwayTime());
        TransactionResponseDTO expenseTransactionResponseDTO = scoreApi.expenseScore(scoreExpenseDetailRequestDTO);

        if (!GlobalContants.ResponseStatus.SUCCESS.equals(expenseTransactionResponseDTO.getStatus())) {
            //积分扣减失败
            scoreGiveAwayLogDTO.setStatus(expenseTransactionResponseDTO.getStatus());
            scoreGiveAwayLogDTO.setResponseMsg(expenseTransactionResponseDTO.getResponseMsg());
            throw new GiveAwayFailException(JSON.toJSONString(scoreGiveAwayLogDTO));

//            return scoreGiveAwayLogDTO;
        }
        //赠送积分
        ScoreIncomeDetailRequestDTO scoreIncomeDetailRequestDTO = new ScoreIncomeDetailRequestDTO();
        scoreIncomeDetailRequestDTO.setScoreAccountId(scoreGiveAwayRequestDTO.getReceiveScoreAccountId());
        scoreIncomeDetailRequestDTO.setBusiCode(ScoreBusinessType.BUSI_INCOME_RECEIVE.getBusinessCode());
        scoreIncomeDetailRequestDTO.setBusiSource(ScoreBusinessType.BUSI_INCOME_RECEIVE.getBusinessSource());
        scoreIncomeDetailRequestDTO.setMemo(scoreGiveAwayRequestDTO.getMemo());
        scoreIncomeDetailRequestDTO.setScoreVal(scoreGiveAwayRequestDTO.getScoreVal());
        scoreIncomeDetailRequestDTO.setTransactionNo(uniqueSequenceNoService.generateUniqueSequenceNo());
        scoreIncomeDetailRequestDTO.setTransaTime(scoreGiveAwayRequestDTO.getGiveAwayTime());
        TransactionResponseDTO incomeTransactionResponseDTO = scoreApi.addScore(scoreIncomeDetailRequestDTO);
        if (!GlobalContants.ResponseStatus.SUCCESS.equals(incomeTransactionResponseDTO.getStatus())) {
            //赠送失败
            scoreGiveAwayLogDTO.setStatus(incomeTransactionResponseDTO.getStatus());
            scoreGiveAwayLogDTO.setResponseMsg(incomeTransactionResponseDTO.getResponseMsg());
            throw new GiveAwayFailException(JSON.toJSONString(scoreGiveAwayLogDTO));
//            return scoreGiveAwayLogDTO;
        }
        //记录赠送
        ScoreGiveAwayLog scoreGiveAwayLog = BeanMapper.map(scoreGiveAwayRequestDTO, ScoreGiveAwayLog.class);
        scoreGiveAwayLog.setGiveAwayScoreAccount(accountRepos.findOne(scoreGiveAwayRequestDTO.getGiveAwayScoreAccountId()));
        scoreGiveAwayLog.setReceiveScoreAccount(accountRepos.findOne(scoreGiveAwayRequestDTO.getReceiveScoreAccountId()));
        scoreGiveAwayLog.setScoreExpenseRcptCode(expenseTransactionResponseDTO.getScoreRcptCode());
        scoreGiveAwayLog.setScoreIncomeRcptCode(incomeTransactionResponseDTO.getScoreRcptCode());
        scoreGiveAwayLogRepos.saveAndFlush(scoreGiveAwayLog);
        scoreGiveAwayLogDTO = BeanMapper.map(scoreGiveAwayLog, ScoreGiveAwayLogDTO.class);
        scoreGiveAwayLogDTO.setStatus(GlobalContants.ResponseStatus.SUCCESS);
        scoreGiveAwayLogDTO.setResponseMsg("调用成功");
        return scoreGiveAwayLogDTO;
    }

    @Override
    public ScoreGiveAwayLogDTO getScoreGiveAwayInfo(GetScoreGiveAwayInfoRequestDTO getScoreGiveAwayInfoRequestDTO) {
        String scoreIncomeRcptCode = getScoreGiveAwayInfoRequestDTO.getScoreIncomeRcptCode();
        String scoreExpenseRcptCode = getScoreGiveAwayInfoRequestDTO.getScoreExpenseRcptCode();
        ScoreGiveAwayLog scoreGiveAwaylog;
        if (Lang.isEmpty(scoreIncomeRcptCode)) {
            scoreGiveAwaylog = scoreGiveAwayLogRepos.getScoreGiveAwayByExpenseCode(scoreExpenseRcptCode);
        } else if (Lang.isEmpty(scoreExpenseRcptCode)) {
            scoreGiveAwaylog = scoreGiveAwayLogRepos.getScoreGiveAwayByIncomeCode(scoreIncomeRcptCode);
        } else {
            scoreGiveAwaylog = scoreGiveAwayLogRepos.getScoreGiveAwayInfo(scoreIncomeRcptCode, scoreExpenseRcptCode);
        }
        if (scoreGiveAwaylog == null) {
            return null;
        } else {
            return BeanMapper.map(scoreGiveAwaylog, ScoreGiveAwayLogDTO.class);
        }
    }

    @Override
    public ScoreStatisticsResponseDTO scoreExpenseStatics(ScoreStatisticsRequestDTO scoreStatisticsRequestDTO) {
        BigDecimal scoreVal;
        if (StringUtils.isEmpty(scoreStatisticsRequestDTO.getScoreAccountId())) {
            scoreVal = scoreExpenseDetailRepos.queryOrgExpenseStaticsVal(scoreStatisticsRequestDTO.getOrganizationCode(), scoreStatisticsRequestDTO.getBusiCode());
            return new ScoreStatisticsResponseDTO(scoreVal);
        }
        scoreVal = scoreExpenseDetailRepos.queryExpenseStaticsVal(scoreStatisticsRequestDTO.getOrganizationCode(), scoreStatisticsRequestDTO.getScoreAccountId(), scoreStatisticsRequestDTO.getBusiCode());
        if (scoreVal == null) {
            scoreVal = new BigDecimal("0");
        }
        return new ScoreStatisticsResponseDTO(scoreVal);
    }

    @Override
    public ScoreStatisticsResponseDTO scoreIncomeStatics(ScoreStatisticsRequestDTO scoreStatisticsRequestDTO) {
        BigDecimal scoreVal;
        if (StringUtils.isEmpty(scoreStatisticsRequestDTO.getScoreAccountId())) {
            scoreVal = scoreIncomeDetailRepos.queryOrgIncomeStaticsVal(scoreStatisticsRequestDTO.getOrganizationCode(), scoreStatisticsRequestDTO.getBusiCode());
            return new ScoreStatisticsResponseDTO(scoreVal);
        }
        scoreVal = scoreIncomeDetailRepos.queryIncomeStaticsVal(scoreStatisticsRequestDTO.getOrganizationCode(), scoreStatisticsRequestDTO.getScoreAccountId(), scoreStatisticsRequestDTO.getBusiCode());
        if (scoreVal == null) {
            scoreVal = new BigDecimal("0");
        }
        return new ScoreStatisticsResponseDTO(scoreVal);
    }

    @Override
    public BaseResponse deleteAccount(String scoreAccountId) {
        BaseResponse baseResponse = new BaseResponse(GlobalContants.ResponseStatus.ERROR, "失败");
        if (StringUtils.isEmpty(scoreAccountId)) {
            baseResponse.setResponseMsg("账户id不太能为空");
            return baseResponse;
        }
        ScoreAccount scoreAccount = accountRepos.findOne(scoreAccountId);
        if (scoreAccount == null) {
            baseResponse.setResponseMsg("无对应账户");
            return baseResponse;
        }
        scoreAccount.setIsDelete(true);
        scoreAccount.setUserId(scoreAccountId + "__delete");
        accountRepos.saveAndFlush(scoreAccount);
        baseResponse.setStatus(GlobalContants.ResponseStatus.SUCCESS);
        baseResponse.setResponseMsg("注销成功");
        return baseResponse;
    }

}
