package org.swift.cache.impl;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.util.TreeMap;
import java.util.concurrent.locks.ReentrantLock;

import org.swift.cache.ISoftRefCache;
import org.swift.cache.KeySoftReference;


public class SoftRefCache<K extends Comparable<K>, V> implements
		ISoftRefCache<K, V> {

	private ReferenceQueue<V> queue = new ReferenceQueue<V>();
	private TreeMap<K, KeySoftReference<K, V>> cache = new TreeMap<K, KeySoftReference<K, V>>();
	private final ReentrantLock lock = new ReentrantLock();
	private int hit = 0;
	private int miss = 0;

	@SuppressWarnings("unchecked")
	private void removeUnenable() {
		for (int i = 0;; i++) {
			SoftReference<V> ref = (SoftReference<V>) queue.poll();
			if (ref == null) {
				if (i > 0) {
					System.out.println("remove " + i + " reference");
				}
				return;
			}
			if (!(ref instanceof KeySoftReference)) {
				continue;
			}
			KeySoftReference<K, V> pair = (KeySoftReference<K, V>) ref;
			cache.remove(pair.getKey());
		}
	}

	public void clear() {
		try {
			lock.lock();
			removeUnenable();
			cache.clear();
			hit = 0;
			miss = 0;
		} finally {
			lock.unlock();
		}
	}

	public boolean contains(K k) {
		try {
			lock.lock();
			removeUnenable();
			return cache.containsKey(k);
		} finally {
			lock.unlock();
		}
	};

	public V get(K k) {
		try {
			lock.lock();
			removeUnenable();
			SoftReference<V> ref = cache.get(k);
			if (ref == null) {
				miss++;
				return null;
			}
			hit++;
			return ref.get();
		} finally {
			lock.unlock();
		}
	}

	public int getSize() {
		try {
			lock.lock();
			removeUnenable();
			return cache.size();
		} finally {
			lock.unlock();
		}
	}

	public V put(K k, V v) {
		try {
			lock.lock();
			removeUnenable();
			KeySoftReference<K, V> ref = new KeySoftReference<K, V>(k, v, queue);
			KeySoftReference<K, V> old = cache.put(k, ref);
			if (old == null) {
				return null;
			}
			return old.get();
		} finally {
			lock.unlock();
		}
	}

	public V remove(K k) {
		try {
			lock.lock();
			removeUnenable();
			KeySoftReference<K, V> old = cache.remove(k);
			if (old == null) {
				return null;
			}
			return old.get();
		} finally {
			lock.unlock();
		}
	}

	public int getHit() {
		try {
			lock.lock();
			removeUnenable();
			return hit;
		} finally {
			lock.unlock();
		}
	}

	public int getMiss() {
		try {
			lock.lock();
			removeUnenable();
			return miss;
		} finally {
			lock.unlock();
		}
	}

}
