package org.swift.util;

import org.swift.bean.BeanEnv;
import org.swift.bean.BeanFactory;
import org.swift.func.ILock;


/**
 * 功能描述：带锁的缓存，如缓存被加锁，就不会被消除（线程安全）
 * 
 * 缓存按访问频率排序，而且缓存有个最大量，超出会被消除。
 * 
 */
public interface ILockCache<K extends Comparable<K>, V extends ILock> {
	public class Factory {
		private static BeanFactory<ILockCache<?, ILock>> factory = new BeanFactory<ILockCache<?, ILock>>(
				ILockCache.class, BeanEnv.getUtilBeanConfig());

		/**
		 * 功能描述：创建一个最大大小已定的缓存实例
		 * 
		 * @param size
		 * @return
		 */
		@SuppressWarnings("unchecked")
		public static <K extends Comparable<K>, V extends ILock> ILockCache<K, V> New(int maxSize) {
			Class<?>[] pType = { int.class };
			Object[] pValue = { maxSize };
			return (ILockCache<K, V>) factory.New(pType, pValue);
		}
	}

	/**
	 * 功能描述：真实大小
	 * 
	 * @return
	 */
	int getSize();

	/**
	 * 功能描述：最大缓存大小
	 * 
	 * @return
	 */
	int getMaxSize();

	void setMaxSize(int size);

	/**
	 * 功能描述：启用最后元素状态
	 * 
	 * 当缓存已经超过最大并启用此功能时，系统只判断最后一个元素是否被锁，
	 * 
	 * 如是锁定状态就不再移除多余元素
	 * 
	 */
	void enableLastStatus();

	/**
	 * 功能描述：缓存中是否存在
	 * 
	 * @param k
	 * @return
	 */
	boolean containsKey(K k);

	/**
	 * 功能描述：添加缓存数据，如缓存已经满出，就消除最后几个没被锁的数据，只到缓存大小刚好
	 * 
	 * @param k
	 * @param v
	 */
	V put(K k, V v);

	/**
	 * 功能描述：取缓存数据
	 * 
	 * @param k
	 * @return
	 */
	V get(K k);

	/**
	 * 功能描述：取缓存数据,并锁定
	 * 
	 * @param k
	 * @return
	 */
	V getLocked(K k);

	/**
	 * 功能描述：删除缓存数据
	 * 
	 * @param k
	 */
	V remove(K k);

	/**
	 * 功能描述：删除所有没锁定的数据
	 * 
	 */
	void removeAllUnlocked();

	/**
	 * 功能描述：强制删除所有数据
	 * 
	 */
	void clear();

	/**
	 * 功能描述：命中次数
	 * 
	 * @return
	 */
	int getHit();

	/**
	 * 功能描述：没命中次数
	 * 
	 * @return
	 */
	int getMiss();

	/**
	 * 功能描述：命中率
	 * 
	 * @return
	 */
	int getHitPercent();
}
