package org.swift.util.encrypt;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

public class TEAx {

	private long TEA_DELTA = 0x9E3779B9; // Magic number for key schedule
	static final int BLOCK_SIZE = 8;

	long k0;
	long k1;
	long k2;
	long k3;

	byte[] subKey;

	public TEAx() {
		k0 = k1 = k2 = k3 = 0;
	}

	public TEAx(byte key[]) {
		setKey(key);
	}

		
	/**
	 * 设置加密秘钥
	 * 
	 * @param key
	 *            长度没有限制，如果为null，数据将不经加密直接被拷贝
	 */
	public void setKey(byte key[]) {

		k0 = k1 = k2 = k3 = 0;

		if (key.length == 0) {
			subKey = null;
		} else {

			MessageDigest md5;
			try {
				md5 = MessageDigest.getInstance("MD5");
				byte[] md5value = md5.digest(key);

				long[] k = new long[4];

				for (int i = 0; i < 4; i++) {

					for (int j = 0; j < 4; j++) {
						// java 总是big endian的
						long temp = (md5value[i * 4 + j] + 256) & 0xff;
						temp <<= (8 * j);
						k[i] |= temp;
					}
				}
				k0 = k[0];
				k1 = k[1];
				k2 = k[2];
				k3 = k[3];
				// 第二次md5作为subKey
				subKey = md5.digest(md5value);

			} catch (NoSuchAlgorithmException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

	}

	/**
	 * 加密
	 * 
	 * @param clear
	 * @return
	 */
	public byte[] encode(byte clear[]) {

		byte[] coded = new byte[clear.length];
		if (subKey == null) {

			for (int i = 0; i < clear.length; i++)
				coded[i] = clear[i];

		} else {

			int bytesLeft = clear.length;

			while (bytesLeft >= BLOCK_SIZE) {

				encodeBlock(clear, coded, clear.length - bytesLeft);

				bytesLeft -= BLOCK_SIZE;
			}

			if (bytesLeft > 0)
				subCrypt(clear, coded, clear.length - bytesLeft);
		}

		return coded;

	}

	/**
	 * 解密
	 * 
	 * @param coded
	 * @return
	 */
	public byte[] decode(byte coded[]) {

		byte[] clear = new byte[coded.length];

		if (subKey == null) {

			for (int i = 0; i < coded.length; i++)
				clear[i] = coded[i];

		} else {

			int bytesLeft = coded.length;

			while (bytesLeft >= BLOCK_SIZE) {

				decodeBlock(coded, clear, coded.length - bytesLeft);

				bytesLeft -= BLOCK_SIZE;
			}

			if (bytesLeft > 0)
				subCrypt(coded, clear, coded.length - bytesLeft);
		}

		return clear;
	}
	/**
	 * 加密一个块
	 * 
	 * @param clear
	 * @param coded
	 * @param offset
	 */
	private void encodeBlock(byte clear[], byte coded[], int offset) {

		long y = 0;
		for (int i = 0; i < 4; i++) {
			long temp = (clear[offset + i] + 256) & 0xff;
			temp <<= (8 * (3 - i));
			y |= temp;
		}

		long z = 0;
		for (int i = 0; i < 4; i++) {
			long temp = (clear[offset + 4 + i] + 256) & 0xff;
			temp <<= (8 * (3 - i));
			z |= temp;
		}

		long sum = 0;
		for (int count = 32; count > 0; count--) {

			sum += TEA_DELTA; // Magic number for key schedule
			sum &= 0xffffffffl;
			y += (z << 4) + k0 ^ z + sum ^ (z >> 5) + k1;
			y &= 0xffffffffl;
			z += (y << 4) + k2 ^ y + sum ^ (y >> 5) + k3; /* end cycle */
			z &= 0xffffffffl;
		}

		for (int i = 0; i < 4; i++)
			coded[offset + i] = (byte) ((y >> (8 * (3 - i))) & 0xff);

		for (int i = 0; i < 4; i++)
			coded[offset + 4 + i] = (byte) ((z >> (8 * (3 - i))) & 0xff);

	}

	/**
	 * 解密一个块
	 * 
	 * @param coded
	 * @param clear
	 * @param offset
	 */
	private void decodeBlock(byte coded[], byte clear[], int offset) {

		long y = 0;
		for (int i = 0; i < 4; i++) {

			long temp = (coded[offset + i] + 256) & 0xff;
			temp <<= (8 * (3 - i));
			y |= temp;
		}

		long z = 0;
		for (int i = 0; i < 4; i++) {

			long temp = (coded[offset + 4 + i] + 256) & 0xff;
			temp <<= (8 * (3 - i));
			z |= temp;
		}

		long sum = TEA_DELTA << 5;
		sum &= 0xffffffffl;
		for (int count = 32; count > 0; count--) {
			z += 0x100000000l;
			z -= (((y << 4) + k2 ^ y + sum ^ (y >> 5) + k3) & 0xffffffffl);
			z &= 0xffffffffl;
			y += 0x100000000l;
			y -= (((z << 4) + k0 ^ z + sum ^ (z >> 5) + k1) & 0xffffffffl);
			y &= 0xffffffffl;
			sum += 0x100000000l;
			sum -= TEA_DELTA; // Magic number for key schedule
			sum &= 0xffffffffl;

		}

		for (int i = 0; i < 4; i++)
			clear[offset + i] = (byte) ((y >> (8 * (3 - i))) & 0xff);

		for (int i = 0; i < 4; i++)
			clear[offset + 4 + i] = (byte) ((z >> (8 * (3 - i))) & 0xff);
	}

	/**
	 * 加密/解密未对齐的部分
	 * 
	 * @param in
	 * @param out
	 * @param offset
	 */
	private void subCrypt(byte in[], byte out[], int offset) {

		// do sub encrypt
		for (int i = 0; i < in.length - offset; i++) {

			if (subKey.length > 0) {

				int temp = (subKey[(i + in.length) % subKey.length] + 256) & 0xff;
				temp %= subKey.length;
				temp = (subKey[temp] + 256) & 0xff;

				out[i + offset] = (byte) (in[i + offset] ^ temp);
			} else
				out[i + offset] = in[i + offset];
		}
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// TODO Auto-generated method stub
		TEAx teax = new TEAx("6789012345".getBytes());
		
		String s = "72064411;72d4d026ee493ebb3268a34bfb7c238b";
		byte[] coded = teax.encode(s.getBytes());
		
		String c = Security.byteArrayToHexString(coded);
		
		System.out.println(c);

		byte[] clear = teax.decode(Security.hexStringToByteArray(c));

		s = new String(clear);

		System.out.println(s);
	}

}
