package org.swift.util.file;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import org.swift.util.IResource;


/**
 * 文件处理<br>
 * 鲍庆丰<br>
 * baoqingfeng@gmail.com<br>
 * 示例:<br>
 * String path="c:\\test\\"; //获取文件路径<br>
 * String file="test.txt"; //文件名<br>
 * FileHandle fh = new FileHandle();<br>
 * fh.newFolder(path); //新建目录<br>
 * fh.newFile(path+file, "test测试"); //新建文件<br>
 * fh.delAllFile(path); //删除文件<br>
 * fh.delFolder(path); //删除目录<br>
 */
public class FileHandler {

	/**
	 * 获得文件最后修改时间
	 * @param 文件或者目录名
	 * @return 最后修改时间
	 */
	public static long getLastModified(String path){
		File file = new File(path);
		return file.lastModified();
	}
	
	/**
	 * 修改属性文件
	 *
	 * @param propFile
	 *            要修改的属性文件
	 * @param modifyProp
	 *            要修改的属性
	 * @throws IOException
	 */
	public static void modifyProp(String propFile, Map modifyProp) throws IOException {
		Properties proper = new Properties();

		FileInputStream finPut = new FileInputStream(propFile);
		proper.load(finPut);

		Iterator keys = (Iterator) modifyProp.keySet().iterator();

		while (keys.hasNext()) {
			String key = (String) keys.next();
			String value = (String) modifyProp.get(key);
			System.out.println("key==" + key);
			System.out.println("value==" + value);
			proper.put(key, value);
		}

		FileOutputStream foutPut = new FileOutputStream(propFile);
		proper.store(foutPut, "");
	}
	
	/**
	 * 读取文件到内存
	 * 
	 * @param filename
	 *            String 如 c:/fqf
	 */
	public static String loadFileToString(String filename) throws IOException {
		File file = new File(filename);
		if (!file.exists() || file.isDirectory()) {
			return null;
		}
		IResource resource = IResource.Factory.New(filename);
		String ret = null;
		try {
			String line = null;
			StringBuffer sb = new StringBuffer((int) file.length());
			resource.open(IResource.OpenMode.ReadOnly);
			while ((line = resource.read()) != null) {
				sb.append(line).append("\n");
			}
			ret = sb.toString();
		} finally {
			resource.close();
		}
		return ret;
	}

	/**
	 * 新建目录
	 * 
	 * @param folderPath
	 *            String 如 c:/fqf
	 */
	public static boolean newFolder(String folderPath) {
		File folder = new File(folderPath);
		if (folder.exists()) {
			if (folder.isDirectory()) {
				return true;
			}
			if (folder.isFile()) {
				return false;
			}
		}
		return folder.mkdirs();
		// try {
		// File file = new File(folderPath);
		// if (file.exists()) {
		// return true;
		// }
		// File parent = file.getParentFile();
		// if (parent == null) {
		// return file.mkdir();
		// }
		// if (!parent.exists()) {
		// if (!newFolder(file.getParent())) {
		// return false;
		// }
		// }
		// return file.mkdir();
		// } catch (Exception e) {
		// System.out.println("新建目录操作出错");
		// e.printStackTrace();
		// return false;
		// }
	}

	/**
	 * 新建文件
	 * 
	 * @param fileName
	 *            String 文件路径及名称 如c:/fqf.txt
	 * @param fileContent
	 *            String 文件内容
	 */
	public static boolean newFile(String fileName) {
		return newFile(fileName, null);
	}

	/**
	 * 新建文件
	 * 
	 * @param fileName
	 *            String 文件路径及名称 如c:/fqf.txt
	 * @param fileContent
	 *            String 文件内容
	 */
	public static boolean newFile(String fileName, String fileContent) {
		File file = new File(fileName);
		if (file.exists()) {
			if (file.isFile()) {
				if (fileContent == null) {
					return true;
				}
			}
			return false;
		}
		if (!file.exists() && !newFolder(file.getParent())) {
			return false;
		}
		PrintWriter writer = null;
		try {
			if (!file.exists() && fileContent == null) {
				if (!file.createNewFile()) {
					return false;
				}
				return true;
			}
			writer = new PrintWriter(file, "utf-8");
			writer.println(fileContent);
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		} finally {
			if (writer != null) {
				writer.close();
			}
		}
	}
	
	public static boolean newFileStream(String fileName, InputStream fileContent) {
		File file = new File(fileName);
		if (file.exists()) {
			if (file.isFile()) {
				if (fileContent == null) {
					return true;
				}
			}
			return false;
		}
		if (!file.exists() && !newFolder(file.getParent())) {
			return false;
		}
		OutputStream writer = null;
		try {
			if (!file.exists() && fileContent == null) {
				if (!file.createNewFile()) {
					return false;
				}
				return true;
			}

			writer = new FileOutputStream(file);
		   int bytesRead = 0;
		   byte[] buffer = new byte[8192];
		   while ((bytesRead = fileContent.read(buffer, 0, 8192)) != -1) {
			   writer.write(buffer, 0, bytesRead);
		   }
		   fileContent.close();

			
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		} finally {
			if (writer != null) {
				try {
					writer.close();
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * 删除文件
	 * 
	 * @param file
	 *            String 文件路径及名称 如c:/fqf.txt
	 */
	public static boolean delFile(String file) {
		return new File(file).delete();
	}

	/**
	 * 删除文件夹
	 * 
	 * @param folderPath
	 *            String 文件夹路径 如c:/fqf
	 */
	public static boolean delFolder(String folderPath) {
		if (!delAllFile(folderPath)) { // 删除完里面所有内容
			return false;
		}
		return new File(folderPath).delete(); // 删除空文件夹
	}

	/**
	 * 删除文件夹里面的所有文件
	 * 
	 * @param path
	 *            String 文件夹路径 如 c:/fqf
	 */
	public static boolean delAllFile(String path) {
		File file = new File(path);
		if (!file.exists() || !file.isDirectory()) {
			return false;
		}
		String[] tempList = file.list();
		if (!path.endsWith(File.separator)) {
			path += File.separator;
		}
		for (int i = 0; i < tempList.length; i++) {
			File f = new File(path + tempList[i]);
			if (f.isFile()) {
				if (!f.delete()) {
					return false;
				}
			}
			if (f.isDirectory()) {
				if (!delAllFile(path + File.separator + tempList[i])) { // 先删除文件夹里面的文件
					return false;
				}
				if (!delFolder(path + File.separator + tempList[i])) { // 再删除空文件夹
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * 复制单个文件
	 * 
	 * @param oldPath
	 *            String 原文件路径 如：c:/fqf.txt
	 * @param newPath
	 *            String 复制后路径 如：f:/fqf.txt
	 * @return boolean
	 * @throws IOException
	 */
	public static boolean copyFile(String oldPath, String newPath) throws IOException {
		File oldFile = new File(oldPath);
		if (!oldFile.exists() || oldFile.isDirectory()) {
			return false;
		}
		if (!newFile(newPath)) {
			return false;
		}
		BufferedInputStream input = null;
		BufferedOutputStream output = null;
		try {
			input = new BufferedInputStream(new FileInputStream(oldPath)); // 读入原文件
			output = new BufferedOutputStream(new FileOutputStream(newPath));
			byte[] buffer = new byte[1024];
			int len = 0;
			while ((len = input.read(buffer)) > 0) {
				output.write(buffer, 0, len);
			}
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		} finally {
			try {
				if (input != null) {
					input.close();
				}

				if (output != null) {
					output.close();
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return true;
	}

	/**
	 * 复制整个文件夹内容
	 * 
	 * @param oldPath
	 *            String 原文件路径 如：c:/fqf
	 * @param newPath
	 *            String 复制后路径 如：f:/fqf/ff
	 * @return boolean
	 * @throws Exception
	 */
	public static boolean copyFolder(String oldPath, String newPath) throws Exception {
		File oldFolder = new File(oldPath);
		if (!oldFolder.exists() || oldFolder.isFile()) {
			return false;
		}
		File newFolder = new File(newPath);
		if (newFolder.exists() && newFolder.isFile()) {
			return false;
		}
		if (!newFolder.exists() && !newFolder.mkdirs()) {
			return false;
		}
		if (!oldPath.endsWith(File.separator)) {
			oldPath += File.separator;
		}
		if (!newPath.endsWith(File.separator)) {
			newPath += File.separator;
		}
		try {
			String[] file = oldFolder.list();
			File temp = null;
			for (int i = 0; i < file.length; i++) {
				temp = new File(oldPath + file[i]);
				if (temp.isFile()) {
					if (!copyFile(temp.getPath(), newPath + temp.getName())) {
						return false;
					}
					continue;
				}
				if (temp.isDirectory()) {
					if (!copyFolder(temp.getPath(), newPath + temp.getName())) {
						return false;
					}
					continue;
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	/**
	 * 移动文件到指定目录
	 * 
	 * @param oldPath
	 *            String 如：c:/fqf.txt
	 * @param newPath
	 *            String 如：d:/fqf.txt
	 * @throws IOException
	 */
	public static boolean moveFile(String oldPath, String newPath) throws IOException {
		if (!copyFile(oldPath, newPath)) {
			return false;
		}
		return delFile(oldPath);
	}

	/**
	 * 移动文件到指定目录
	 * 
	 * @param oldPath
	 *            String 如：c:/fqf
	 * @param newPath
	 *            String 如：d:/fqf
	 * @throws Exception
	 */
	public static boolean moveFolder(String oldPath, String newPath) throws Exception {
		if (!copyFolder(oldPath, newPath)) {
			return false;
		}
		return delFolder(oldPath);
	}

}
