package org.swift.util.http;

import java.io.UnsupportedEncodingException;
import java.util.Enumeration;
import java.util.HashMap;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 操作request方法
 */
public class RequestUtil {
	RequestUtil() {
	}

	/**
	 * 转换编码,将"8859_1"转换成"gb2312"
	 * 
	 * @param value
	 *            待转换的字符串
	 * @return
	 * @throws UnsupportedEncodingException
	 */
	private static String stringEncode(String value) throws UnsupportedEncodingException {
		if (value != null) {
			return new String(value.getBytes("8859_1"), "gb2312");
		} else {
			return null;
		}
	}

	/**
	 * 得到默认值,当字符串value为null时返回默认值d(String)
	 * 
	 * @param value
	 * @param d
	 *            默认值
	 * @return
	 */
	private static String getDefaultValue(String value, String d) {
		if (value == null)
			return d;
		return value;
	}

	/**
	 * 得到默认值,当字符串value为null时返回默认值d(int)
	 * 
	 * @param value
	 * @param d
	 *            默认值
	 * @return
	 */
	private static int getDefaultValue(String value, int d) {
		if (value == null)
			return d;
		else
			return Integer.parseInt(value);
	}

	/**
	 * 得到默认值,当字符串value为null时返回默认值d(long)
	 * 
	 * @param value
	 * @param d
	 * @return
	 */
	private static long getDefaultValue(String value, long d) {
		if (value == null)
			return d;
		else
			return Long.parseLong(value);
	}

	/**
	 * 得到默认值,当字符串value为null时返回默认值d(boolean)
	 * 
	 * @param value
	 * @param d
	 * @return
	 */
	private static boolean getDefaultValue(String value, boolean d) {
		if (value == null)
			return d;
		if ((value.compareToIgnoreCase("true") == 0) || (value.compareToIgnoreCase("1") == 0))
			return true;
		return false;
	}

	/**
	 * 得到默认值,当字符串value为null时返回默认值d(double)
	 * 
	 * @param value
	 * @param d
	 * @return
	 */
	private static double getDefaultValue(String value, double d) {
		if (value == null)
			return d;
		return Double.parseDouble(value);
	}

	// /Parameter
	/**
	 * 通过request.getParameter()获得数据
	 * 
	 * @param request
	 * @param name
	 * @return 经过编码转换成GB2123的String类型的字符串
	 * @throws UnsupportedEncodingException
	 */
	public static String getParaString(HttpServletRequest request, String name) throws UnsupportedEncodingException {
		String value = request.getParameter(name);
		return stringEncode(value);
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getParameter()获得的数据(String),默认值为d(String)
	 * @throws UnsupportedEncodingException
	 */
	public static String getParaString(HttpServletRequest request, String name, String d) throws UnsupportedEncodingException {
		String value = getParaString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * @param request
	 * @param name
	 * @return request.getParameter()获得的数据(int)
	 * @throws NumberFormatException
	 * @throws UnsupportedEncodingException
	 */
	public static int getParaInt(HttpServletRequest request, String name) throws NumberFormatException, UnsupportedEncodingException {
		return Integer.parseInt(getParaString(request, name));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getParameter()获得的数据(int),默认值d(int)
	 * @throws NumberFormatException
	 * @throws UnsupportedEncodingException
	 */
	public static int getParaInt(HttpServletRequest request, String name, int d) throws UnsupportedEncodingException {
		String value = getParaString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getParameter()获得的数据(long)
	 * @throws NumberFormatException
	 * @throws UnsupportedEncodingException
	 */
	public static long getParaLong(HttpServletRequest request, String name, long d) throws UnsupportedEncodingException {
		String value = getParaString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * @param request
	 * @param name
	 * @return request.getParameter()获得的数据(boolean),默认值d(boolean)
	 * @throws NumberFormatException
	 * @throws UnsupportedEncodingException
	 */
	public static boolean getParaBoolean(HttpServletRequest request, String name, boolean d) throws UnsupportedEncodingException {
		String value = getParaString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getParameter()获得的数据(double),默认值d(double)
	 * @throws NumberFormatException
	 * @throws UnsupportedEncodingException
	 */
	public static double getParaDouble(HttpServletRequest request, String name, double d) throws Exception {
		String value = getParaString(request, name);
		return getDefaultValue(value, d);
	}

	// attribute

	/**
	 * @param request
	 * @param name
	 * @return request.getAttribute()获得的数据(String),经过编码转换
	 * @throws UnsupportedEncodingException
	 */
	public static String getAttrString(HttpServletRequest request, String name) throws UnsupportedEncodingException {
		Object value = request.getAttribute(name);
		if (value == null)
			return null;
		return stringEncode((String) value);
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getAttribute()获得的数据(String),默认值d(String)
	 * @throws UnsupportedEncodingException
	 */
	public static String getAttrString(HttpServletRequest request, String name, String d) throws UnsupportedEncodingException {
		String value = getAttrString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置Attribute, 即request.setAttribute(name, value)
	 * 
	 * @param request
	 * @param name
	 * @param value
	 */
	public static void setAttrString(HttpServletRequest request, String name, String value) {
		request.setAttribute(name, value);
	}

	/**
	 * @param request
	 * @param name
	 * @return request.getAttribute()获得的数据(int)
	 * @throws NumberFormatException
	 * @throws UnsupportedEncodingException
	 */
	public static int getAttrInt(HttpServletRequest request, String name) throws NumberFormatException, UnsupportedEncodingException {
		return Integer.parseInt(getParaString(request, name));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getAttribute()获得的数据(int),默认d(int)
	 * @throws UnsupportedEncodingException
	 */
	public static int getAttrInt(HttpServletRequest request, String name, int d) throws UnsupportedEncodingException {
		String value = getAttrString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置Attribute, 数据类型为int
	 * 
	 * @param request
	 * @param name
	 * @param value
	 */
	public static void setAttrInt(HttpServletRequest request, String name, int value) {
		setAttrString(request, name, String.valueOf(value));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getAttribute()获得的数据(int),默认d(int)
	 * @throws UnsupportedEncodingException
	 */
	public static long getAttrLong(HttpServletRequest request, String name, long d) throws UnsupportedEncodingException {
		String value = getAttrString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置Attribute, 数据类型为long
	 * 
	 * @param request
	 * @param name
	 * @param value
	 */
	public static void setAttrLong(HttpServletRequest request, String name, long value) {
		setAttrString(request, name, String.valueOf(value));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getAttribute()获得的数据(boolean),默认d(boolean)
	 * @throws UnsupportedEncodingException
	 */
	public static boolean getAttrBoolean(HttpServletRequest request, String name, boolean d) throws UnsupportedEncodingException {
		String value = getAttrString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置Attribute, 数据类型为boolean
	 * 
	 * @param request
	 * @param name
	 * @param value
	 */
	public static void setAttrBoolean(HttpServletRequest request, String name, boolean value) {
		setAttrString(request, name, String.valueOf(value));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return request.getAttribute()获得的数据(double),默认d(double)
	 * @throws Exception
	 */
	public static double getAttrDouble(HttpServletRequest request, String name, double d) throws Exception {
		String value = getParaString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置Attribute, 数据类型为double
	 * 
	 * @param request
	 * @param name
	 * @param value
	 */
	public static void setAttrDouble(HttpServletRequest request, String name, double value) {
		setAttrString(request, name, String.valueOf(value));
	}

	// cookie
	/**
	 * @param request
	 * @param name
	 * @return 得到名为name的cookie(String)
	 */
	public static String getCookieString(HttpServletRequest request, String name) {
		Cookie[] cookies = request.getCookies();
		for (int i = 0; i < cookies.length; i++) {
			Cookie cookie = cookies[i];
			if (name.equals(cookie.getName()))
				return (cookie.getValue());
		}
		return null;
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return 得到名为name的cookie(String),默认d(String)
	 * @throws UnsupportedEncodingException
	 */
	public static String getCookieString(HttpServletRequest request, String name, String d) throws UnsupportedEncodingException {
		String value = getCookieString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置cookie(String)
	 * 
	 * @param response
	 * @param name
	 * @param value
	 */
	public static void setCookieString(HttpServletResponse response, String name, String value) {
		Cookie cookie = new Cookie(name, value);
		response.addCookie(cookie);
	}

	/**
	 * @param request
	 * @param name
	 * @return 得到名为name的cookie(int)
	 * @throws NumberFormatException
	 * @throws UnsupportedEncodingException
	 */
	public static int getCookieInt(HttpServletRequest request, String name) throws NumberFormatException, UnsupportedEncodingException {
		return Integer.parseInt(getParaString(request, name));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return 得到名为name的cookie(int),默认d(int)
	 * @throws UnsupportedEncodingException
	 */
	public static int getCookieInt(HttpServletRequest request, String name, int d) throws UnsupportedEncodingException {
		String value = getAttrString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置cookie值(int)
	 * 
	 * @param response
	 * @param name
	 * @param value
	 */
	public static void setCookieInt(HttpServletResponse response, String name, int value) {
		setCookieString(response, name, String.valueOf(value));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return 得到名为name的cookie(long),默认d(long)
	 * @throws UnsupportedEncodingException
	 */
	public static long getCookieLong(HttpServletRequest request, String name, long d) throws UnsupportedEncodingException {
		String value = getAttrString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置cookie值(long)
	 * 
	 * @param response
	 * @param name
	 * @param value
	 */
	public static void setCookieLong(HttpServletResponse response, String name, long value) {
		setCookieString(response, name, String.valueOf(value));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return 得到名为name的cookie(boolean),默认d(boolean)
	 * @throws UnsupportedEncodingException
	 */
	public static boolean getCookieBoolean(HttpServletRequest request, String name, boolean d) throws UnsupportedEncodingException {
		String value = getAttrString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置cookie值(boolean)
	 * 
	 * @param response
	 * @param name
	 * @param value
	 */
	public static void setCookieBoolean(HttpServletResponse response, String name, boolean value) {
		setCookieString(response, name, String.valueOf(value));
	}

	/**
	 * @param request
	 * @param name
	 * @param d
	 * @return 得到名为name的cookie(double),默认d(double)
	 * @throws Exception
	 */
	public static double getCookieDouble(HttpServletRequest request, String name, double d) throws Exception {
		String value = getParaString(request, name);
		return getDefaultValue(value, d);
	}

	/**
	 * 设置cookie值(double)
	 * 
	 * @param response
	 * @param name
	 * @param value
	 */
	public static void setCookieDouble(HttpServletResponse response, String name, double value) {
		setCookieString(response, name, String.valueOf(value));
	}

	// /////

	/**
	 * 设置错误信息,存放在session里,名为PARAMUTIL_ERROR_INFO
	 * 
	 * @param request
	 * @param s
	 */
	public static void setErrorInfo(HttpServletRequest request, String s) {
		request.getSession().setAttribute("PARAMUTIL_ERROR_INFO", s);
	}

	/**
	 * 取得错误信息,从session里取
	 * 
	 * @param request
	 * @return
	 */
	public static String getErrorInfo(HttpServletRequest request) {
		Object obj = request.getSession().getAttribute("PARAMUTIL_ERROR_INFO");

		if (obj != null) {
			request.getSession().removeAttribute("PARAMUTIL_ERROR_INFO");

			return (String) obj;
		}

		return null;
	}

	/**
	 * 判断当有是否存在错误信息
	 * 
	 * @param request
	 * @return
	 */
	public static boolean hasErrorInfo(HttpServletRequest request) {
		return request.getSession().getAttribute("PARAMUTIL_ERROR_INFO") != null;
	}

	/**
	 * 得到地址栏完整的参数字符串
	 * 
	 * @param request
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static String computeQueryString(HttpServletRequest request) {
		String queryString = "";
		Enumeration<String> paramNames = request.getParameterNames();
		boolean first = true;

		while (paramNames.hasMoreElements()) {
			String name = paramNames.nextElement();

			if (!first)
				queryString += "&";
			queryString += name + "=" + request.getParameter(name);
			first = false;
		}
		return queryString;
	}

	public static HashMap<String, String> getRequestParameter2HashMap(String req) {
		HashMap<String, String> ham = new HashMap<String, String>();
		if (req == null || req.length() == 0)
			return ham;
		String reqs[] = req.split("&");
		for (String r : reqs) {
			String p[] = r.split("=");
			int l = p.length;
			if (l == 2) {
				ham.put(p[0], p[1]);
			} else if (l == 1) {
				ham.put(p[0], "");
			}
		}

		return ham;

	}

	// 获得客户端IP
	public static String getClientIpAddress(HttpServletRequest request) {
		String ip = request.getHeader("X-Forwarded-For");
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("Proxy-Client-IP");
		}
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("WL-Proxy-Client-IP");
		}
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("HTTP_CLIENT_IP");
		}
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("HTTP_X_FORWARDED_FOR");
		}
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getRemoteAddr();
		}
		return ip;
	}
}
