package org.swift.util.impl;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.swift.util.IConfig;
import org.swift.util.IResource;


/**
 * 
 * 功能描述：系统配置信息
 * 
 */
public class Config implements IConfig {
	private Map<String, Map<String, String>> config;
	private String dfltSec;
	private IResource resource;

	public Config(IResource resource) {
		this.resource = resource;
		this.dfltSec = "";
		config = new LinkedHashMap<String, Map<String, String>>();
		reload();
	}

	private synchronized void reload() {
		if (resource == null || !resource.exists()) {
			return;
		}
		try {
			resource.open(IResource.OpenMode.ReadOnly);
		} catch (IOException e1) {
			e1.printStackTrace();
			return;
		}
		config = new LinkedHashMap<String, Map<String, String>>();
		try {
			String sec = dfltSec;
			String line = resource.read();
			while (line != null) {
				line = line.trim();
				if (line.length() == 0 || line.startsWith(";") || line.startsWith("#")) {
					line = resource.read();
					continue;
				}
				if (line.startsWith("[") && line.endsWith("]")) {
					line = line.substring(1);
					line = line.substring(0, line.length() - 1);
					sec = line.trim();
					Map<String, String> secValues = config.get(sec);
					if (secValues == null) {
						secValues = new LinkedHashMap<String, String>();
						config.put(sec, secValues);
					}
					line = resource.read();
					continue;
				}
				Map<String, String> secValues = config.get(sec);
				if (secValues == null) {
					secValues = new LinkedHashMap<String, String>();
					config.put(sec, secValues);
				}
				String kv[] = line.split("=", 2);
				if (kv.length == 2) {
					secValues.put(kv[0], kv[1]);
				} else {
					secValues.put(kv[0], "");
				}
				line = resource.read();
			}
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				resource.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	@Deprecated
	public void checkModification() {
		checkModifyAndReload();
	}

	public boolean checkModifyAndReload() {
		if (resource == null) {
			return false;
		}
		if (resource.checkModification()) {
			reload();
			return true;
		}
		return false;
	}

	public synchronized void deleteKey(String key) {
		deleteKey(dfltSec, key);
	}

	public synchronized void deleteKey(String section, String key) {
		Map<String, String> kv = config.get(section);
		if (kv == null) {
			return;
		}
		kv.remove(key);
	}

	public synchronized void deleteSection() {
		deleteSection(dfltSec);
	}

	public synchronized void deleteSection(String section) {
		config.remove(section);
	}

	public synchronized Map<String, String> getAllValues() {
		return getAllValues(dfltSec);
	}

	public synchronized Map<String, String> getAllValues(String section) {
		return new LinkedHashMap<String, String>(config.get(section));
	}

	public synchronized boolean getBoolean(String key) {
		return getBoolean(key, false);
	}

	public synchronized boolean getBoolean(String key, boolean dflt) {
		return getBoolean(dfltSec, key, dflt);
	}

	public synchronized boolean getBoolean(String section, String key, boolean dflt) {
		String value = getString(section, key, null);
		if (value == null) {
			return dflt;
		}
		return Boolean.valueOf(value).booleanValue();
	}

	public synchronized String getDefaultSection() {
		return dfltSec;
	}

	public synchronized double getDouble(String key) {
		return getDouble(key, 0);
	}

	public synchronized double getDouble(String key, double dflt) {
		return getDouble(dfltSec, key, dflt);
	}

	public synchronized double getDouble(String section, String key, double dflt) {
		String value = getString(section, key, null);
		if (value == null) {
			return dflt;
		}
		return new Double(value).doubleValue();
	}

	public synchronized float getFloat(String key) {
		return getFloat(key, 0);
	}

	public synchronized float getFloat(String key, float dflt) {
		return getFloat(dfltSec, key, dflt);
	}

	public synchronized float getFloat(String section, String key, float dflt) {
		String value = getString(section, key, null);
		if (value == null) {
			return dflt;
		}
		return new Float(value).floatValue();
	}

	public synchronized int getInt(String key) {
		return getInt(key, 0);
	}

	public synchronized int getInt(String key, int dflt) {
		return getInt(dfltSec, key, dflt);
	}

	public synchronized int getInt(String section, String key, int dflt) {
		String value = getString(section, key, null);
		if (value == null) {
			return dflt;
		}
		return new Integer(value).intValue();
	}

	public synchronized Set<String> getKeys() {
		return getKeys(dfltSec);
	}

	public synchronized Set<String> getKeys(String section) {
		Map<String, String> kv = config.get(section);
		if (kv == null) {
			return null;
		}
		return new HashSet<String>(kv.keySet());
	}

	public synchronized long getLong(String key) {
		return getLong(key, 0);
	}

	public synchronized long getLong(String key, long dflt) {
		return getLong(dfltSec, key, dflt);
	}

	public synchronized long getLong(String section, String key, long dflt) {
		String value = getString(section, key, null);
		if (value == null) {
			return dflt;
		}
		return new Long(value).longValue();
	}

	public synchronized Set<String> getSections() {
		return new HashSet<String>(config.keySet());
	}

	public synchronized String getString(String key) {
		return getString(key, null);
	}

	public synchronized String getString(String key, String dflt) {
		return getString(dfltSec, key, dflt);
	}

	public synchronized String getString(String section, String key, String dflt) {
		Map<String, String> kv = config.get(section);
		if (kv == null) {
			return dflt;
		}
		String value = kv.get(key);
		if (value == null) {
			return dflt;
		}
		return value;
	}

	public synchronized boolean hasKey(String key) {
		return hasKey(dfltSec, key);
	}

	public synchronized boolean hasKey(String section, String key) {
		Map<String, String> kv = config.get(section);
		if (kv == null) {
			return false;
		}
		return kv.containsKey(key);
	}

	public synchronized boolean save() {
		if (resource == null) {
			return false;
		}
		try {
			resource.open(IResource.OpenMode.WriteOnly);
		} catch (IOException e1) {
			e1.printStackTrace();
			return false;
		}
		try {
			if (config.size() == 0) {
				resource.write(null);
			} else {
				for (String sec : config.keySet()) {
					if (sec != null && sec.length() > 0) {
						resource.write("");
						resource.write("[" + sec + "]");
					}
					Map<String, String> kv = config.get(sec);
					if (kv == null) {
						resource.write("");
						continue;
					}
					for (Map.Entry<String, String> entry : kv.entrySet()) {
						String key = entry.getKey();
						String value = entry.getValue();
						if (value == null) {
							value = "";
						}
						resource.write(key + "=" + value);
					}
				}
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
			return false;
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		} finally {
			try {
				resource.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return true;
	}

	public synchronized void setBoolean(String key, boolean value) {
		setBoolean(dfltSec, key, value);
	}

	public synchronized void setBoolean(String section, String key, boolean value) {
		setString(section, key, String.valueOf(value));
	}

	public synchronized void setDefaultSection(String section) {
		dfltSec = section;
	}

	public synchronized void setDouble(String key, double value) {
		setDouble(dfltSec, key, value);
	}

	public synchronized void setDouble(String section, String key, double value) {
		setString(section, key, new Double(value).toString());
	}

	public synchronized void setFloat(String key, float value) {
		setFloat(dfltSec, key, value);
	}

	public synchronized void setFloat(String section, String key, float value) {
		setString(section, key, new Float(value).toString());
	}

	public synchronized void setInt(String key, int value) {
		setInt(dfltSec, key, value);
	}

	public synchronized void setInt(String section, String key, int value) {
		setString(section, key, String.valueOf(value));
	}

	public synchronized void setLong(String key, long value) {
		setLong(dfltSec, key, value);
	}

	public synchronized void setLong(String section, String key, long value) {
		setString(section, key, String.valueOf(value));
	}

	public synchronized void setString(String key, String value) {
		setString(dfltSec, key, value);
	}

	public synchronized void setString(String section, String key, String value) {
		Map<String, String> kv = config.get(section);
		if (kv == null) {
			kv = new LinkedHashMap<String, String>();
			config.put(section, kv);
		}
		kv.put(key, value);
	}

	public synchronized IResource getResource() {
		return resource;
	}
}
