package org.swift.util.impl;

import java.util.TreeMap;
import java.util.concurrent.locks.ReentrantLock;

import org.swift.func.ILock;
import org.swift.tools.LinkList;
import org.swift.util.ILockCache;


public class LRULockCache<K extends Comparable<K>, V extends ILock> implements ILockCache<K, V> {

	private class Pair<F, S> {
		public F first;
		public S secend;
	}

	private final ReentrantLock lock = new ReentrantLock();
	private LinkList<K> list;
	private TreeMap<K, Pair<V, LinkList<K>.Iterator>> map;
	private int maxSize;
	private int hit;
	private int miss;
	private boolean lastStatus;

	public LRULockCache(int maxSize) {
		this.maxSize = maxSize;
		list = new LinkList<K>();
		map = new TreeMap<K, Pair<V, LinkList<K>.Iterator>>();
		hit = 0;
		miss = 0;
		lastStatus = false;
	}

	public void clear() {
		try {
			lock.lock();
			list.clear();
			map.clear();
			hit = 0;
			miss = 0;
		} finally {
			lock.unlock();
		}
	}

	public boolean containsKey(K k) {
		if (k == null) {
			return false;
		}
		try {
			lock.lock();
			if (map.containsKey(k)) {
				hit++;
				return true;
			} else {
				miss++;
				return false;
			}
		} finally {
			lock.unlock();
		}
	}

	public V get(K k) {
		if (k == null) {
			return null;
		}
		try {
			lock.lock();
			if (!map.containsKey(k)) {
				miss++;
				return null;
			}
			Pair<V, LinkList<K>.Iterator> pair = map.get(k);
			V v = pair.first;
			pair.secend.remove();
			list.addFirst(k);
			pair.secend = list.begin();
			hit++;
			return v;
		} finally {
			lock.unlock();
		}
	}

	public int getHit() {
		try {
			lock.lock();
			return hit;
		} finally {
			lock.unlock();
		}
	}

	public int getHitPercent() {
		try {
			lock.lock();
			long total = hit + miss;
			if (total == 0) {
				return 0;
			}
			return (int) (hit * 100L / total);
		} finally {
			lock.unlock();
		}
	}

	public V getLocked(K k) {
		try {
			lock.lock();
			V v = get(k);
			if (v != null) {
				v.lock();
			}
			return v;
		} finally {
			lock.unlock();
		}
	}

	public int getMaxSize() {
		try {
			lock.lock();
			return maxSize;
		} finally {
			lock.unlock();
		}
	}

	public int getMiss() {
		try {
			lock.lock();
			return miss;
		} finally {
			lock.unlock();
		}
	}

	public int getSize() {
		try {
			lock.lock();
			return map.size();
		} finally {
			lock.unlock();
		}
	}

	public V put(K k, V v) {
		if (k == null) {
			return null;
		}
		try {
			lock.lock();
			Pair<V, LinkList<K>.Iterator> pair = new Pair<V, LinkList<K>.Iterator>();
			pair.first = v;
			list.addFirst(k);
			pair.secend = list.begin();
			pair = map.put(k, pair);
			V ov = null;
			if (pair != null) {
				ov = pair.first;
				pair.secend.remove();
			}
			int more = map.size() - maxSize;
			if (more <= 0) {
				return ov;
			}
			LinkList<K>.Iterator it = list.end();
			while (more > 0 && it != null) {
				K ek = it.get();
				V ev = map.get(ek).first;
				if (ev != null && ev.isLocked()) {
					if (lastStatus) {
						return ov;
					}
					it = it.previous();
					continue;
				}
				map.remove(ek);
				more--;
				LinkList<K>.Iterator cit = it;
				it = it.previous();
				cit.remove();
			}
			return ov;
		} finally {
			lock.unlock();
		}
	}

	public V remove(K k) {
		if (k == null) {
			return null;
		}
		try {
			lock.lock();
			Pair<V, LinkList<K>.Iterator> pair = map.remove(k);
			if (pair == null) {
				return null;
			}
			pair.secend.remove();
			return pair.first;
		} finally {
			lock.unlock();
		}
	}

	public void setMaxSize(int size) {
		try {
			lock.lock();
			maxSize = size;
		} finally {
			lock.unlock();
		}
	}

	public void enableLastStatus() {
		try {
			lock.lock();
			lastStatus = true;
		} finally {
			lock.unlock();
		}
	}

	public void removeAllUnlocked() {
		try {
			lock.lock();
			for (LinkList<K>.Iterator it = list.end(); it != null;) {
				K k = it.get();
				V v = map.get(k).first;
				if (v != null && v.isLocked()) {
					it = it.previous();
					continue;
				}
				map.remove(k);
				LinkList<K>.Iterator cit = it;
				it = it.previous();
				cit.remove();
			}
			hit = 0;
			miss = 0;
		} finally {
			lock.unlock();
		}
	}
}
