package org.swift.util.impl;


import java.io.IOException;
import java.sql.CallableStatement;
import java.util.LinkedList;

import org.swift.database.pool.DBConnect;
import org.swift.func.ILock;
import org.swift.func.IProcessor;
import org.swift.util.IResource;
import org.swift.util.ISqlExecuter;
import org.swift.util.IThreadPool;
import org.swift.util.lang.StringUtil;


/**
 * 功能描述：sql异步执行器的实现
 * 
 * 修改记录： 2009.03.02
 * 修改了当stop后，不能再往队列中添加sql语句。（stop后写文件，和线程执行是否结束，暂时没有更好的方法判断和处理，先不考虑） 2009.03.03
 * 修改了线程执行的效率 2009.03.04 修改了stop后，写未执行sql的方法；增加了添加sql语句的同步功能 2009.03.10
 * 增加了任务结束后，执行队列中带锁的sql，再写文件
 */
public class SqlExecuter implements ISqlExecuter, IProcessor {

	private String dbname;
	private LinkedList<SqlTask> sqlQueue;
	private IResource resource;

	public SqlExecuter(String dbname, IResource resource, IThreadPool pool) {
		this.dbname = dbname;
		this.resource = resource;
		sqlQueue = new LinkedList<SqlTask>();
		// 添加任务到线程池
		pool.addWorker(this);
	}

	public boolean update(String sql, ILock lock) {
		return update(sql, (Object) lock);
	}

	public boolean update(String sql, Object lock) {
		if (sql == null || sql.trim().equals("")) {
			return false;
		}
		if (sql.trim().startsWith("{")) {
			return call(sql, lock);
		}
		return addSqlTask(new SqlTask(sql, lock, false));
	}

	public boolean call(String sql, ILock lock) {
		return call(sql, (Object) lock);
	}

	public boolean call(String sql, Object lock) {
		if (sql == null || sql.trim().equals("")) {
			return false;
		}
		if (!sql.trim().startsWith("{")) {
			return false;
		}
		return addSqlTask(new SqlTask(sql, lock, true));
	}

	public synchronized boolean init() {
		// 从文件中读取保存的sql语句，
		if (resource == null || !resource.exists()) {
			return true;
		}
		try {
			resource.open(IResource.OpenMode.ReadOnly);
			String lnsql = resource.read();
			while (lnsql != null) {
				lnsql = lnsql.trim();
				if (lnsql.length() > 0) {
					lnsql = StringUtil.decryptEscapeChar(lnsql);
					addSqlTask(new SqlTask(lnsql, null, lnsql.startsWith("{")));
				}
				lnsql = resource.read();
			}
			resource.open(IResource.OpenMode.WriteOnly);
			resource.write(null);
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				resource.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return true;
	}

	public boolean cycle() {
		while (sqlQueue.size() > 0) {
			SqlTask sqltask = sqlQueue.getFirst();
			sqlQueue.removeFirst();
			if (sqltask == null) {
				return true;
			}
			if (sqltask.isCall) {
				if (!execCall(sqltask.sql)) {
					addSqlTask(sqltask);
					break;
				}
			} else {
				if (!update(sqltask.sql)) {
					addSqlTask(sqltask);
					break;
				}
			}
			if (sqltask.lock instanceof ILock) {
				((ILock) sqltask.lock).unlock();
			}
			return true;
		}
		try {
			Thread.sleep(1000);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		return true;
	}

	public synchronized void destroy() {
		if (resource == null || sqlQueue.size() == 0) {
			return;
		}
		// 保存未执行的sql语句
		try {
			resource.open(IResource.OpenMode.AppendOnly);
			while (!sqlQueue.isEmpty()) {
				String lnsql = StringUtil
						.encryptEscapeChar(sqlQueue.poll().sql);
				resource.write(lnsql);
			}
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				resource.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	private synchronized boolean addSqlTask(SqlTask sqltask) {
		return sqlQueue.add(sqltask);
	}

	private boolean update(String sql) {
		if (dbname == null || "".equals(dbname))
			return false;
		if (sql == null || "".equals(sql.trim()))
			return false;
		DBConnect conn = new DBConnect();
		try {
			conn.init(dbname);
			conn.prepareStatement(sql);
			conn.executeUpdate();
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		} finally {
			conn.close();
		}
		return true;
	}

	private boolean execCall(String sql) {
		if (dbname == null || "".equals(dbname))
			return false;
		if (sql == null || "".equals(sql.trim()))
			return false;
		DBConnect conn = new DBConnect();
		try {
			conn.init(dbname);
			CallableStatement cstmt = conn.prepareCall(sql);
			cstmt.executeQuery();
			cstmt.close();
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		} finally {
			conn.close();
		}
		return true;
	}

	static class SqlTask {
		public String sql;

		public Object lock;

		public boolean isCall;

		public SqlTask(String sql, Object lock, boolean iscall) {
			this.sql = sql;
			this.lock = lock;
			this.isCall = iscall;
		}
	}

	public static void main(String[] args) {
		IResource resource = IResource.Factory.New("D:\\test.sql");
		IThreadPool pool = IThreadPool.Factory.New();
		pool.start();
		ISqlExecuter executer = ISqlExecuter.Factory
				.New("test", resource, pool);
		executer.update("select * from testtest", null);
		pool.stop();
	}
}
